// File          : JDRSegmentLoader.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class dealing with saving and loading {@link JDRPathSegment} segments 
 * to/from JDR and AJR files.
 * @author Nicola L C Talbot
 */
public class JDRSegmentLoader
{
   /**
    * Creates new loader.
    */
   public JDRSegmentLoader()
   {
      listeners_ = new Vector<JDRSegmentLoaderListener>();
   }

   /**
    * Saves the given segment in JDR format. This first writes the
    * ID character specified by 
    * {@link JDRSegmentLoaderListener#getJDRid(float)} and then 
    * writes the segment specifications using 
    * {@link JDRSegmentLoaderListener#writeJDR(JDRObject,DataOutputStream,float)}
    * @param segment the segment that needs to be saved
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if an I/O error occurs
    * @see #loadJDR(double,double,DataInputStream,float)
    */
   public void saveJDR(JDRObject segment,
                    DataOutputStream dout,
                    float version)
      throws IOException,InvalidFormatException
   {
      dout.writeChar(segment.getListener().getJDRid(version));
      segment.getListener().writeJDR(segment,dout,version);
   }

   /**
    * Loads an segment specified in JDR format. This first reads a
    * character and checks through the list of listeners
    * to determine which type is identified by the  
    * character (using {@link JDRSegmentLoaderListener#getJDRid(float)})
    * @param x the current x coordinate
    * @param y the current y coordinate
    * @param din the input stream
    * @param version the JDR version number
    * @return the specified segment
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @see #saveJDR(JDRObject,DataOutputStream,float)
    * @see #addListener(JDRSegmentLoaderListener)
    */
   public JDRObject loadJDR(double x, double y,
      DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      char c = din.readChar();

      for (Enumeration<JDRSegmentLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRSegmentLoaderListener listener = e.nextElement();

         if (listener.getJDRid(version) == c)
         {
            return listener.readJDR(x, y, din, version);
         }
      }

      throw new InvalidSegmentIDException(c);
   }

   public JDRObject loadJDR(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      char c = din.readChar();

      for (Enumeration<JDRSegmentLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRSegmentLoaderListener listener = e.nextElement();

         if (listener.getJDRid(version) == c)
         {
            return listener.readJDR(din, version);
         }
      }

      throw new InvalidSegmentIDException(c);
   }

   /**
    * Saves the given segment in AJR format. This first writes the
    * ID character specified by 
    * {@link JDRSegmentLoaderListener#getAJRid(float)} and then 
    * writes the specifications using 
    * {@link JDRSegmentLoaderListener#writeAJR(JDRObject,PrintWriter,float)}
    * @param segment the segment that needs to be saved
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if an I/O error occurs
    * @see #loadAJR(double,double,BufferedReader,float)
    */
   public void saveAJR(JDRObject segment,
                    PrintWriter out,
                    float version)
      throws IOException,InvalidFormatException
   {
      AJR.writeChar(out, segment.getListener().getAJRid(version));
      segment.getListener().writeAJR(segment,out,version);
   }

   /**
    * Loads a segment specified in AJR format. This first reads a
    * character and checks through the list of listeners
    * to determine which segment type is identified by the  
    * character (using {@link JDRSegmentLoaderListener#getAJRid(float)}).
    * @param x the current x-coordinate
    * @param y the current y-coordinate
    * @param in the input stream
    * @param version the AJR version number
    * @return the specified segment
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the segment format
    * @throws EOFException if the file ends unexpectedly
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * (specified by {@link AJR#buffLength}) is exceeded
    * @see #saveAJR(JDRObject,PrintWriter,float)
    * @see #addListener(JDRSegmentLoaderListener)
    */
   public JDRObject loadAJR(double x, double y,
      BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      char c = AJR.readChar(in);

      for (Enumeration<JDRSegmentLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRSegmentLoaderListener listener = e.nextElement();

         if (listener.getAJRid(version) == c)
         {
            return listener.readAJR(x, y, in, version);
         }
      }

      throw new InvalidSegmentIDException(c, AJR.getLineNum());
   }

   public JDRObject loadAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      char c = AJR.readChar(in);

      for (Enumeration<JDRSegmentLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRSegmentLoaderListener listener = e.nextElement();

         if (listener.getAJRid(version) == c)
         {
            return listener.readAJR(in, version);
         }
      }

      throw new InvalidSegmentIDException(c, AJR.getLineNum());
   }

   /**
    * Adds a new listener.
    * @param listener the new listener
    * @see #getListeners()
    */
   public void addListener(JDRSegmentLoaderListener listener)
   {
      listeners_.add(listener);
   }

   /**
    * Gets all the listeners registered with this loader.
    * @return list of all listeners registered with this loader
    * @see #addListener(JDRSegmentLoaderListener)
    */
   public Vector<JDRSegmentLoaderListener> getListeners()
   {
      return listeners_;
   }

   private Vector<JDRSegmentLoaderListener> listeners_;
}
