// File          : JDRScaledPattern.java
// Date          : 9th April 2011
// Last Modified : 9th April 2011
// Author        : Nicola L. C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;

/**
 *  Class representing a scaled pattern. This has an underlying
 *  shape that gets replicated by scaling relative to a given point.
 *  The overall shape is governed by the following parameters:
 *  <ul>
 *  <li> X scale factor.
 *  <li> Y scale factor.
 *  <li> Anchor.
 *  <li> Number of replicas.
 *  <li> Mode.
 *  </ul>
 *  The mode determines whether the underlying path and replicas are
 *  drawn in one go (single mode) or whether they are drawn independently 
 *  of each other (multi mode).
 *  @author Nicola L C Talbot
 */

public class JDRScaledPattern extends JDRPattern
{
   /**
    * Creates a scaled pattern from the given path.
    * @param path the path
    * @param point the point of rotation
    * @param adjust the adjustment control
    * @param scaleX X scale factor
    * @param scaleY Y scale factor
    * @param singleMode true if single path mode
    * @param replicas the number of times the underlying path is
    * replicated
    */
    public JDRScaledPattern(JDRShape path, JDRPoint point,
       JDRPoint adjust,
       double scaleX, double scaleY, int replicas, boolean singleMode)
       throws InvalidReplicaException
    {
       setNumReplicas(replicas);
       setPatternAnchor(point);
       setPatternAdjust(adjust);

       path_ = path;
       singlemode_ = singleMode;

       scalex_ = scaleX;
       scaley_ = scaleY;

       initIterators();
    }

   /**
    * Creates a scaled pattern from the given path.
    * @param path the path
    * @param point the point of rotation
    * @param adjust the adjustment control
    * @param scaleX X scale factor
    * @param scaleY Y scale factor
    * @param singleMode true if single path mode
    * @param showOriginalShape true if original path is to be drawn
    * @param replicas the number of times the underlying path is
    * replicated
    */
    public JDRScaledPattern(JDRShape path, JDRPoint point,
       JDRPoint adjust,
       double scaleX, double scaleY, int replicas, boolean singleMode,
       boolean showOriginalShape)
       throws InvalidReplicaException
    {
       setNumReplicas(replicas);
       setPatternAnchor(point);
       setPatternAdjust(adjust);

       path_ = path;
       singlemode_ = singleMode;
       showoriginal_ = showOriginalShape;

       scalex_ = scaleX;
       scaley_ = scaleY;

       initIterators();
    }

    public JDRScaledPattern(JDRShape path, JDRPoint point,
       double scaleX, double scaleY, int replicas, boolean singleMode)
       throws InvalidReplicaException
    {
       path_ = path;

       setNumReplicas(replicas);
       setPatternAnchor(point);
       setDefaultPatternAdjust();

       singlemode_ = singleMode;

       scalex_ = scaleX;
       scaley_ = scaleY;

       initIterators();
    }

    public JDRScaledPattern(JDRShape path, JDRPoint point,
       double scaleX, double scaleY, int replicas, boolean singleMode,
       boolean showOriginalShape)
       throws InvalidReplicaException
    {
       path_ = path;

       setNumReplicas(replicas);
       setPatternAnchor(point);
       setDefaultPatternAdjust();

       singlemode_ = singleMode;
       showoriginal_ = showOriginalShape;

       scalex_ = scaleX;
       scaley_ = scaleY;

       initIterators();
    }

    /**
     * Creates an empty path with one replica, scale factors = 2
     * and point at the origin.
     */
    public JDRScaledPattern()
    {
       path_     = new JDRPath();
       point_    = new JDRPatternAnchorPoint();
       adjust_   = new JDRPatternAdjustPoint();
       replicas_ = 1;
       singlemode_ = true;

       scalex_ = 2.0;
       scaley_ = 2.0;

       initIterators();
    }

    public JDRScaledPattern(int capacity, JDRPaint lineColor,
                            JDRPaint fillColor, JDRStroke s)
    {
       path_ = new JDRPath(capacity, lineColor, fillColor, s);

       point_    = new JDRPatternAnchorPoint();
       adjust_   = new JDRPatternAdjustPoint();

       replicas_ = 1;
       singlemode_ = true;

       scalex_ = 2.0;
       scaley_ = 2.0;

       initIterators();
    }

   public boolean hasAdjust()
   {
      return true;
   }

    public JDRPattern createTemplate()
    {
       try
       {
         return new JDRScaledPattern(null, null, null,
          scalex_, scaley_, replicas_, singlemode_);
       }
       catch (InvalidReplicaException e)
       {
          // this shouldn't happen
       }

       return null;
    }

    /**
     * Makes the scale factors for this pattern the same as the
     * scale factors for the given pattern. (Which must also be
     * an instance of JDRScaledPattern.)
     */
    public void makeParametersEqual(JDRPattern object)
    {
       JDRScaledPattern pattern = (JDRScaledPattern)object;

       scalex_ = pattern.scalex_;
       scaley_ = pattern.scaley_;
    }

   public void getReplicaTransform(double[] matrix, int replicaIndex)
   {
      if (replicaIndex == 1)
      {
         matrix[0] = scalex_;
         matrix[1] = 0.0;
         matrix[2] = 0.0;
         matrix[3] = scaley_;
         matrix[4] = point_.x * (1.0 - scalex_);
         matrix[5] = point_.y * (1.0 - scaley_);

         return;
      }

      double repScaleX = replicaIndex * scalex_;
      double repScaleY = replicaIndex * scaley_;

      double shiftX = 0.0;
      double shiftY = 0.0;

      if (adjust_ != null && path_ != null)
      {
         JDRPoint pt = path_.getFirstControl();

         shiftX = adjust_.x - scalex_*pt.x + point_.x*(scalex_ - 1);
         shiftY = adjust_.y - scaley_*pt.y + point_.y*(scaley_ - 1);
      }

      int i = replicaIndex - 1;

      matrix[0] = repScaleX;
      matrix[1] = 0.0;
      matrix[2] = 0.0;
      matrix[3] = repScaleY;
      matrix[4] = point_.x * (1.0 - repScaleX) + shiftX*i;
      matrix[5] = point_.y * (1.0 - repScaleY) + shiftY*i;
   }

   /**
    * Translate offset point but constrain it to the line of
    * scaling.
    */
   public void translateAdjust(double x, double y)
   {
      if (adjust_ != null)
      {
         adjust_.translate(x, y);

         if (path_ != null)
         {
            JDRPoint pt = path_.getFirstControl();

            adjust_.moveToLine(point_.x, point_.y,
               scalex_*pt.x + point_.x*(1-scalex_),
               scaley_*pt.y + point_.y*(1-scaley_));
         }
      }
   }

   public void translateAnchor(double x, double y)
   {
      point_.translate(x, y);

      translateAdjust(-x, -y);
   }


   public JDRObjectLoaderListener getListener()
   {
      return shapeListener;
   }

   public String toString()
   {
      String str = "ScaledPattern: point="+point_
                 +", scalex: " +scalex_
                 +", scalex: " +scaley_
                 + ", replicas: "+replicas_
                 + ", size="+size()+", segments=[";

      for (int i = 0; i < size(); i++)
      {
         str += get(i)+",";
      }

      str += "]";

      return str;
    }


   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "ScaledPattern:"+eol;

      str += "point of rotation: "+point_.info()+eol;

      str += "x scale factor: "+scalex_+eol;

      str += "y scale factor: "+scaley_+eol;

      str += "replicas: "+replicas_;

      str += "Underlying shape:"+path_.info();

      return str;
   }

   public double getScaleX()
   {
      return scalex_;
   }

   public double getScaleY()
   {
      return scaley_;
   }

   public void setScaleX(double scaleX)
   {
      scalex_ = scaleX;
   }

   public void setScaleY(double scaleY)
   {
      scaley_ = scaleY;
   }

   public void setScale(double scaleX, double scaleY)
   {
      scalex_ = scalex_;
      scaley_ = scaleY;
   }

   public String[] getDescriptionInfo()
   {
      return new String[] {""+replicas_, ""+scalex_+", "+scaley_};
   }

   private double scalex_, scaley_;

   private static JDRPathListener shapeListener = new JDRScaledPatternListener();

}

