// File          : JDRRotationalPatternListener.java
// Date          : 9th Sept 2010
// Last Modified : 9th Sept 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for rotational pattern paths. This is for JDR/AJR versions
 * 1.6 and above. If the version number is less than 1.6, saved as
 * a group of paths.
 * @author Nicola L C Talbot
 */

public class JDRRotationalPatternListener extends JDRPathListener
{
   public char getJDRid(float version)
   {
      return (version < 1.6 ? 'G' : 'R');
   }

   public char getAJRid(float version)
   {
      return (version < 1.6 ? 'G' : 'R');
   }

   public void writeJDR(JDRObject object,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException
   {
      JDRRotationalPattern path = (JDRRotationalPattern)object;

      JDRObjectLoader objectLoader = JDR.getObjectLoader();

      if (version < 1.6)
      {
         JDRGroup group = path.separate(null);

         JDRObjectLoaderListener listener = group.getListener();

         listener.writeJDR(group, dout, version);
      }
      else
      {
         objectLoader.saveJDR(path.getUnderlyingShape(), dout, version);

         JDRPoint p = path.getPatternAnchor();

         dout.writeDouble(p.x);
         dout.writeDouble(p.y);

         dout.writeDouble(path.getRotationAngle());
         dout.writeInt(path.getNumReplicas());
         dout.writeBoolean(path.isSinglePath());
         dout.writeBoolean(path.showOriginal());
      }
   }

   public JDRObject readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      JDRObjectLoader objectLoader = JDR.getObjectLoader();

      JDRObject object = objectLoader.loadJDR(din, version);

      if (!(object instanceof JDRShape))
      {
         throw new InvalidFormatException
            ("Shape specs expected for underlying rotational pattern object.");
      }

      double x = din.readDouble();
      double y = din.readDouble();

      double angle = din.readDouble();
      int replicas = din.readInt();
      boolean isSinglePath = din.readBoolean();
      boolean showOriginal = din.readBoolean();

      return new JDRRotationalPattern((JDRShape)object,
         new JDRPatternAnchorPoint(x,y),
         angle, replicas, isSinglePath, showOriginal);
   }

   public void writeAJR(JDRObject object,
                        PrintWriter out,
                        float version)
   throws IOException,InvalidFormatException
   {
      JDRRotationalPattern path = (JDRRotationalPattern)object;

      JDRObjectLoader objectLoader = AJR.getObjectLoader();

      if (version < 1.6)
      {
         JDRGroup group = path.separate(null);

         JDRObjectLoaderListener listener = group.getListener();

         listener.writeAJR(group, out, version);
      }
      else
      {
         objectLoader.saveAJR(path.getUnderlyingShape(), out, version);

         JDRPoint p = path.getPatternAnchor();

         AJR.writeDouble(out, p.x);
         AJR.writeDouble(out, p.y);

         AJR.writeDouble(out, path.getRotationAngle());
         AJR.writeInt(out, path.getNumReplicas());
         AJR.writeBoolean(out, path.isSinglePath());
         AJR.writeBoolean(out, path.showOriginal());

         out.println();
      }
   }

   public JDRObject readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      JDRObjectLoader objectLoader = AJR.getObjectLoader();

      JDRObject object = objectLoader.loadAJR(in, version);

      if (!(object instanceof JDRShape))
      {
         throw new InvalidFormatException
            ("Shape specs expected for underlying rotational pattern object",
             AJR.getLineNum());
      }

      double x = AJR.readDouble(in);
      double y = AJR.readDouble(in);

      double angle = AJR.readDouble(in);
      int replicas = AJR.readInt(in);
      boolean isSinglePath = AJR.readBoolean(in);
      boolean showOriginal = AJR.readBoolean(in);

      return new JDRRotationalPattern((JDRShape)object,
         new JDRPatternAnchorPoint(x,y),
         angle, replicas, isSinglePath, showOriginal);
   }
}
