// File          : JDRRotationalPattern.java
// Date          : 9th Sept 2010
// Last Modified : 6th April 2011
// Author        : Nicola L. C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;

/**
 *  Class representing a rotational pattern. This has an underlying
 *  shape that gets replicated by rotating around a given point. The
 *  overall shape is governed by the following parameters:
 *  <ul>
 *  <li> Point of rotation.
 *  <li> Angle of rotation.
 *  <li> Number of replicas.
 *  <li> Mode.
 *  </ul>
 *  The mode determines whether the underlying path and replicas are
 *  drawn in one go (single mode) or whether they are drawn independently 
 *  of each other (multi mode).
 *  @author Nicola L C Talbot
 */

public class JDRRotationalPattern extends JDRPattern
{
   /**
    * Creates a rotational pattern from the given path.
    * @param path the path
    * @param point the point of rotation
    * @param angle the angle of rotation (Radians)
    * @param singleMode true if single path mode
    * @param replicas the number of times the underlying path is
    * replicated
    */
    public JDRRotationalPattern(JDRShape path, JDRPoint point,
       double angle, int replicas, boolean singleMode)
       throws InvalidReplicaException
    {
       setNumReplicas(replicas);

       path_ = path;

       setPatternAnchor(point);

       angle_    = angle;
       singlemode_ = singleMode;

       initIterators();
    }

   /**
    * Creates a rotational pattern from the given path.
    * @param path the path
    * @param point the point of rotation
    * @param angle the angle of rotation (Radians)
    * @param singleMode true if single path mode
    * @param showOriginalShape true if original should be drawn
    * @param replicas the number of times the underlying path is
    * replicated
    */
    public JDRRotationalPattern(JDRShape path, JDRPoint point,
       double angle, int replicas, boolean singleMode, boolean showOriginalShape)
       throws InvalidReplicaException
    {
       setNumReplicas(replicas);

       path_ = path;

       setPatternAnchor(point);

       angle_    = angle;
       singlemode_ = singleMode;
       showoriginal_ = showOriginalShape;

       initIterators();
    }

    /**
     * Creates an empty path with one replica, angle of PI radians
     * and point at the origin.
     */
    public JDRRotationalPattern()
    {
       path_     = new JDRPath();
       point_    = new JDRPatternAnchorPoint();
       angle_    = Math.PI;
       replicas_ = 1;
       singlemode_ = true;

       initIterators();
    }

    public JDRRotationalPattern(int capacity, JDRPaint lineColor,
                            JDRPaint fillColor, JDRStroke s)
    {
       path_ = new JDRPath(capacity, lineColor, fillColor, s);

       point_    = new JDRPatternAnchorPoint();
       angle_    = Math.PI*0.5;
       replicas_ = 4;
       singlemode_ = true;

       initIterators();
    }

    public JDRPattern createTemplate()
    {
       try
       {
         return new JDRRotationalPattern(null, null,
          angle_, replicas_, singlemode_);
       }
       catch (InvalidReplicaException e)
       {
          // this shouldn't happen
       }

       return null;
    }

    /**
     * Makes the angle of rotation for this pattern the same as the
     * angle of rotation for the given pattern. (Which must also be
     * an instance of JDRRotationalPattern.)
     */
    public void makeParametersEqual(JDRPattern object)
    {
       JDRRotationalPattern pattern = (JDRRotationalPattern)object;

       angle_ = pattern.angle_;
    }

   public void getReplicaTransform(double[] matrix, int replicaIndex)
   {
      double repAngle = replicaIndex*angle_;

      double cosAngle = Math.cos(repAngle);
      double sinAngle = Math.sin(repAngle);

      matrix[0] = cosAngle;
      matrix[1] = sinAngle;
      matrix[2] = -sinAngle;
      matrix[3] = cosAngle;
      matrix[4] = point_.x - point_.x * cosAngle + point_.y*sinAngle;
      matrix[5] = point_.y - point_.x * sinAngle - point_.y*cosAngle;
   }


   public JDRObjectLoaderListener getListener()
   {
      return shapeListener;
   }

   public String toString()
   {
      String str = "RotationalPattern: point="+point_+", angle: " +angle_
                 + ", replicas: "+replicas_
                 + ", size="+size()+", segments=[";

      for (int i = 0; i < size(); i++)
      {
         str += get(i)+",";
      }

      str += "]";

      return str;
    }


   public String info()
   {
      String eol = System.getProperty("line.separator", "\n");

      String str = "RotationalPattern:"+eol;

      str += "point of rotation: "+point_.info()+eol;

      str += "angle of rotation: "+angle_+eol;

      str += "replicas: "+replicas_;

      str += "Underlying shape:"+path_.info();

      return str;
   }

   public double getRotationAngle()
   {
      return angle_;
   }

   public void setRotationAngle(double angle)
   {
      angle_ = angle;
   }

   public String[] getDescriptionInfo()
   {
      return new String[] {""+replicas_, ""+angle_};
   }

   private double angle_;

   private static JDRPathListener shapeListener = new JDRRotationalPatternListener();

}

