// File          : JDRRectangularGridListener.java
// Date          : 17th August 2010
// Last Modified : 17th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for rectangular grids.
 * @author Nicola L C Talbot
 */

public class JDRRectangularGridListener implements JDRGridLoaderListener
{
   public byte getJDRid(float version)
   {
      return 0;
   }

   public void writeJDR(JDRGrid object,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException
   {
      JDRRectangularGrid grid = (JDRRectangularGrid)object;

      dout.writeByte((byte)grid.getUnit().getID());

      if (version < 1.6f)
      {
         dout.writeInt((int)Math.round(grid.getMajorInterval()));
      }
      else
      {
         dout.writeDouble(grid.getMajorInterval());
      }

      dout.writeInt(grid.getSubDivisions());
   }

   public JDRGrid readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      byte unitID = din.readByte();

      JDRUnit unit = JDRUnit.getUnit(unitID);

      if (unit == null)
      {
         throw new InvalidUnitIDException(unitID);
      }

      double majorDivisions;

      if (version < 1.6f)
      {
         majorDivisions = din.readInt();
      }
      else
      {
         majorDivisions = din.readDouble();
      }

      int subDivisions = din.readInt();

      return new JDRRectangularGrid(unit, majorDivisions, subDivisions);
   }

   public byte getAJRid(float version)
   {
      return 0;
   }

   public void writeAJR(JDRGrid object,
                        PrintWriter out,
                        float version)
   throws IOException,InvalidFormatException
   {
      JDRRectangularGrid grid = (JDRRectangularGrid)object;

      AJR.writeInt(out, grid.getUnit().getID());

      if (version < 1.6f)
      {
         AJR.writeInt(out, (int)Math.round(grid.getMajorInterval()));
      }
      else
      {
         AJR.writeDouble(out, grid.getMajorInterval());
      }

      AJR.writeInt(out, grid.getSubDivisions());
   }

   public JDRGrid readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      int unitID = AJR.readInt(in);

      JDRUnit unit = JDRUnit.getUnit(unitID);

      if (unit == null)
      {
         throw new InvalidUnitIDException(unitID, AJR.getLineNum());
      }

      double majorDivisions;

      if (version < 1.6f)
      {
         majorDivisions = AJR.readInt(in);
      }
      else
      {
         majorDivisions = AJR.readDouble(in);
      }

      int subDivisions = AJR.readInt(in);

      return new JDRRectangularGrid(unit, majorDivisions, subDivisions);
   }

}
