//File          : JDRRectangularGrid.java
//Description   : Represents a rectangular grid.
//Author        : Nicola L.C. Talbot
//Date          : 17th August 2010
//Last Modified : 17th August 2010
//              http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr;

import java.awt.*;
import java.awt.geom.*;
import java.text.*;

/**
 * Class representing a rectangular grid. (Left-handed Cartesian.)
 * The origin is the top left corner of the page.
 * @author Nicola L C Talbot
 */
public class JDRRectangularGrid extends JDRGrid
{
   /**
    * Initialises using the default settings. The defaults are:
    * bp units, major interval width of 100 units and 10
    * sub-divisions per major interval.
    */
   public JDRRectangularGrid()
   {
      try
      {
         set(JDRUnit.bp, 100.0, 10);
      }
      catch (InvalidGridArgumentException ignore)
      {
      }
   }

   /**
    * Initialises using the given settings.
    * @param gridUnit the grid units
    * @param majorDiv the width of the major interval in terms of
    * the grid unit
    * @param subDiv the number of sub-divisions per major interval
    * @throws InvalidGridArgumentException if the major interval
    * width is &lt;= 0 or if the sub divisions are negative.
    */
   public JDRRectangularGrid(JDRUnit gridUnit, double majorDiv, int subDiv)
     throws InvalidGridArgumentException
   {
      set(gridUnit, majorDiv, subDiv);
   }

   public Object clone()
   {
      JDRRectangularGrid grid = null;

      try
      {
         grid = new JDRRectangularGrid(unit, majorDivisions, subDivisions);
      }
      catch (InvalidGridArgumentException ignore)
      {
      }

      return grid;
   }

   /**
    * Change the grid settings settings.
    * @param gridUnit the grid units
    * @param majorDiv the width of the major interval in terms of
    * the grid unit
    * @param subDiv the number of sub-divisions per major interval
    * @throws InvalidGridArgumentException if the major interval
    * width is &lt;= 0 or if the sub divisions are negative.
    */
   public void set(JDRUnit gridUnit, double majorDiv, int subDiv)
     throws InvalidGridArgumentException
   {
      setUnit(gridUnit);
      setMajorInterval(majorDiv);
      setSubDivisions(subDiv);
   }

   /**
    * Sets the grid unit.
    * @param gridUnit the grid units
    */
   public void setUnit(JDRUnit gridUnit)
   {
      unit = gridUnit;
   }

   /**
    * Sets the major interval width.
    * @param majorDiv the major interval width in terms of the
    * grid's unit
    * @throws InvalidGridArgumentException if the argument is &lt;= 0
    */
   public void setMajorInterval(double majorDiv)
      throws InvalidGridArgumentException
   {
      if (majorDiv <= 0.0)
      {
         throw new InvalidGridArgumentException(majorDiv);
      }

      majorDivisions = majorDiv;
   }

   /**
    * Sets the number of sub divisions within the major interval.
    * @param subDiv the number of sub divisions
    * @throws InvalidGridArgumentException if the argument is &lt; 0
    */
   public void setSubDivisions(int subDiv)
     throws InvalidGridArgumentException
   {
      if (subDiv < 0)
      {
         throw new InvalidGridArgumentException(subDiv);
      }

      subDivisions = subDiv;
   }

   /**
    * Gets the grid unit.
    * @return the grid unit
    */
   public JDRUnit getUnit()
   {
      return unit;
   }

   /**
    * Gets the width of the major intervals.
    * @return the width of the major intervals in terms of the
    * grid's unit.
    */
   public double getMajorInterval()
   {
      return majorDivisions;
   }

   /**
    * Gets the number of sub divisions within a major interval.
    * @return the number of sub divisions.
    */
   public int getSubDivisions()
   {
      return subDivisions;
   }

   public Point2D getMajorTicDistance()
   {
      double major = unit.toBp(majorDivisions);

      return new Point2D.Double(major, major);
   }

   public Point2D getMinorTicDistance()
   {
      Point2D p = new Point2D.Double(0, 0);

      if (subDivisions > 0)
      {
         double distance = unit.toBp(majorDivisions)/subDivisions;

         p.setLocation(distance, distance);
      }

      return p;
   }

   public Point2D getFromCartesian(JDRPaper paper, double x, double y)
   {
      return new Point2D.Double(unit.fromBp(x), unit.fromBp(y));
   }

   public void getFromCartesian(JDRPaper paper, Point2D cartesianPoint,
      Point2D target)
   {
      target.setLocation(unit.fromBp(cartesianPoint.getX()),
                         unit.fromBp(cartesianPoint.getY()));
   }

   public void getCartesian(JDRPaper paper, Point2D original,
      Point2D target)
   {
      target.setLocation(unit.toBp(original.getX()),
                         unit.toBp(original.getY()));
   }

   public Point2D getClosestTic(JDRPaper paper, double x, double y)
   {
      double major = unit.toBp(majorDivisions);

      double maxX = paper.getWidth();
      double maxY = paper.getHeight();

      int n = (int)Math.floor(x/major);
      int m = (int)Math.floor(y/major);

      if (subDivisions == 0)
      {
         double px1 = n*major;
         double py1 = m*major;

         double dx = px1-x;
         double dy = py1-y;

         double distance1 = dx*dx + dy*dy;

         double px2 = px1 + major;
         double py2 = py1;

         dx = px2-x;
         dy = py2-y;

         double distance2 = dx*dx + dy*dy;

         double px3 = px1;
         double py3 = py1+major;

         dx = px3-x;
         dy = py3-y;

         double distance3 = dx*dx + dy*dy;

         double px4 = px2;
         double py4 = py3;

         dx = px4-x;
         dy = py4-y;

         double distance4 = dx*dx + dy*dy;

         if (distance1 <= distance2
          && distance1 <= distance3 
          && distance1 <= distance4)
         {
            return new Point2D.Double(px1, py1);
         }

         if (distance2 <= distance1
          && distance2 <= distance3
          && distance2 <= distance4)
         {
            return new Point2D.Double(px2, py2);
         }

         if (distance3 <= distance1
          && distance3 <= distance2
          && distance3 <= distance4)
         {
            return new Point2D.Double(px3, py3);
         }

         return new Point2D.Double(px4, py4);
      }

      double minor = major/subDivisions;

      double X = n*major;
      double Y = m*major;

      double x1 = x - X;
      double y1 = y - Y;

      int n1 = (int)Math.floor(x1/minor);
      int m1 = (int)Math.floor(y1/minor);

      double px1 = n1*minor;
      double py1 = m1*minor;

      double dx = px1-x1;
      double dy = py1-y1;

      double distance1 = dx*dx + dy*dy;

      double px2 = px1 + minor;
      double py2 = py1;

      dx = px2-x1;
      dy = py2-y1;

      double distance2 = dx*dx + dy*dy;

      double px3 = px1;
      double py3 = py1+minor;

      dx = px3-x1;
      dy = py3-y1;

      double distance3 = dx*dx + dy*dy;

      double px4 = px2;
      double py4 = py3;

      dx = px4-x1;
      dy = py4-y1;

      double distance4 = dx*dx + dy*dy;

      if (distance1 <= distance2
       && distance1 <= distance3 
       && distance1 <= distance4)
      {
         return new Point2D.Double(X+px1, Y+py1);
      }

      if (distance2 <= distance1
       && distance2 <= distance3
       && distance2 <= distance4)
      {
         return new Point2D.Double(X+px2, Y+py2);
      }

      if (distance3 <= distance1
       && distance3 <= distance2
       && distance3 <= distance4)
      {
         return new Point2D.Double(X+px3, Y+py3);
      }

      return new Point2D.Double(X+px4, Y+py4);
   }

   public void drawGrid(Graphics2D g, JDRPaper paper,
      double scale)
   {
      double major = unit.toBp(getMajorInterval());
      double minor = getSubDivisions();

      if (minor > 0)
      {
         minor = major/minor;
      }

      double maxX  = paper.getWidth()*scale;
      double maxY = paper.getHeight()*scale;

      double minX = 0;
      double minY = 0;

      double majorInc = major*scale;
      double minorInc = minor*scale;

      Rectangle bounds = g.getClipBounds();

      if (bounds != null)
      {
         minX = Math.floor(bounds.getX()/majorInc);
         minY = Math.floor(bounds.getY()/majorInc);

         maxX = Math.min(maxX, bounds.getX()+bounds.getWidth());
         maxY = Math.min(maxY, bounds.getY()+bounds.getHeight());
      }

      if (subDivisions == 0) g.setColor(majorGridColor);

      for (double x = minX; x < maxX; x += majorInc)
      {
         for (double y = minY; y < maxY; y += majorInc)
         {
            if (subDivisions > 0)
            {
               g.setColor(minorGridColor);

               for (double dx = 0; dx <= majorInc; dx += minorInc)
               {
                  int ticX = (int)Math.round(x+dx);

                  if (ticX > maxX) break;

                  for (double dy = 0; dy <= majorInc; dy += minorInc)
                  {
                     int ticY = (int)Math.round(y+dy);

                     if (ticY > maxY) break;

                     drawMinorTic(g, ticX, ticY);
                  }
               }
            }

            if (subDivisions != 0) g.setColor(majorGridColor);

            drawMajorTic(g, (int)Math.round(x), (int)Math.round(y));
         }
      }
   }

   public String getUnitLabel()
   {
      return unit.getLabel();
   }

   public JDRUnit getMainUnit()
   {
      return unit;
   }

   public String formatLocation(Point2D point)
   {
      DecimalFormat f 
        = new DecimalFormat(unit == JDRUnit.bp ? "0" : "0.0");

      return f.format(point.getX())+","
            +f.format(point.getY())+" "
            +unit.getLabel();
   }

   public JDRGridLoaderListener getListener()
   {
      return listener;
   }

   private static JDRRectangularGridListener listener = new JDRRectangularGridListener();

   /**
    * Stores the distance between the major tick marks in terms of
    * the unit given by {@link #unit}.
    */
   private double majorDivisions;

   /**
    * Stores the number of subdivisions within a major grid
    * interval.
    */
   private int subDivisions;

   private JDRUnit unit;
}
