//File          : JDRRadialPoint.java
//Description   : Represents a radial point.
//Author        : Nicola L.C. Talbot
//Date          : 16th August 2010
//Last Modified : 16th August 2010
//              http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr;

import java.awt.geom.*;

/**
 * Class representing a radial co-ordinate.
 * @author Nicola L C Talbot
 */
public class JDRRadialPoint extends Point2D
{
   /**
    * Creates a new point at the origin.
    */
   public JDRRadialPoint()
   {
      radius = 0;
      angle = 0;
   }

   /**
    * Creates a new point with the given radius and angle (in
    * radians). The radius must be non-negative.
    */
   public JDRRadialPoint(double theRadius, double theAngle)
      throws IllegalArgumentException
   {
      setRadius(theRadius);
      setAngle(theAngle);
   }

   /**
    * Sets the radius and angle.
    */
   public void setParameters(double theRadius, double theAngle)
      throws IllegalArgumentException
   {
      setRadius(theRadius);
      setAngle(theAngle);
   }

   /**
    * Sets the angle.
    */
   public void setAngle(double theAngle)
   {
      angle  = theAngle;
   }

   /**
    * Sets the radius.
    */
   public void setRadius(double theRadius)
      throws IllegalArgumentException
   {
      if (radius < 0)
      {
         throw new IllegalArgumentException("Radius must be non-negative. R="+theRadius);
      }

      radius = theRadius;
   }

   /**
    * Gets the radius.
    */
   public double getRadius()
   {
      return radius;
   }

   /**
    * Gets the angle in radians.
    */
   public double getAngle()
   {
      return angle;
   }

   public double getX()
   {
      return radius*Math.cos(angle);
   }

   public double getY()
   {
      return radius*Math.sin(angle);
   }

   /**
    * Gets this point in (X,Y) form.
    * @return this point in (X,Y) form
    */
   public Point2D.Double getPoint2D()
   {
      return new Point2D.Double(getX(), getY());
   }

   /**
    * Converts this point to (X,Y) form. The result is stored in the
    * argument.
    * @param target object in which to store the result
    */
   public void getPoint2D(Point2D.Double target)
   {
      target.setLocation(getX(), getY());
   }

   /**
    * Creates a radial point from the given (X,Y) point.
    */
   public static JDRRadialPoint createRadialPoint(Point2D point)
   {
      JDRRadialPoint radialP = new JDRRadialPoint();
      radialP.setLocation(point);

      return radialP;
   }

   /**
    * Sets this to the given point.
    * @param point the specified point to which this should be set
    */
   public void setLocation(Point2D point)
   {
      if (point instanceof JDRRadialPoint)
      {
         angle  = ((JDRRadialPoint)point).getAngle();
         radius = ((JDRRadialPoint)point).getRadius();
      }
      else
      {
         setLocation(point.getX(), point.getY());
      }
   }

   /**
    * Sets this to the radial equivalent of the given (X,Y) point.
    * @param x
    * @param y
    */
   public void setLocation(double x, double y)
   {
      angle = Math.atan2(y, x);
      radius = Math.sqrt(x*x +y*y);
   }

   public Object clone()
   {
      JDRRadialPoint p = new JDRRadialPoint();

      p.angle = angle;
      p.radius = radius;

      return p;
   }

   public String toString()
   {
      return "JDRRadialPoint[radius="+radius+",angle="+angle+"]";
   }

   private double radius;
   private double angle;
}
