// File          : JDRRadialListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for radial paint.
 * @author Nicola L C Talbot
 */

public class JDRRadialListener implements JDRPaintLoaderListener
{
   /**
    * Gets the character identifying {@link JDRRadial} paint in
    * JDR format. JDR versions
    * before 1.3 did not support radial paint, so if version 
    * &lt; 1.3, returns {@link JDRGradient} ID.
    * @return the {@link JDRRadial} paint ID ('D') if
    * version &gt;= 1.3,
    * otherwise the {@link JDRGradient} paint ID ('G')
    */
   public char getJDRid(float version)
   {
      // convert to gradient paint if version < 1.3
      return (version < 1.3f ? 'G' : 'D');
   }

   /**
    * Writes specified radial paint in JDR format. JDR versions
    * before 1.3 did not support radial paint, so if version 
    * &lt; 1.3, the nearest gradient equivalent is saved instead.
    * @param paint the paint to save (must be of type 
    * {@link JDRRadial})
    */
   public void writeJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
   throws IOException
   {
      JDRPaintLoader loader = JDR.getPaintLoader();

      if (version < 1.3f && (paint instanceof JDRRadial))
      {
         JDRGradient c = ((JDRRadial)paint).getJDRGradient();
         JDRPaint startColor = c.getStartColor();
         JDRPaint endColor = c.getEndColor();

         loader.saveJDR(startColor,dout,version);
         loader.saveJDR(endColor,dout,version);

         dout.writeInt(c.getDirection());
      }
      else
      {
         JDRRadial c = (JDRRadial)paint;
         JDRPaint startColor = c.getStartColor();
         JDRPaint endColor = c.getEndColor();

         loader.saveJDR(startColor,dout,version);
         loader.saveJDR(endColor,dout,version);

         dout.writeInt(c.getStartLocation());
      }
   }

   /**
    * Reads radial paint specified in JDR format. JDR versions
    * prior to 1.3 did not support radial paint, so if 
    * version &lt; 1.3, returns {@link JDRGradient} paint 
    * otherwise returns {@link JDRRadial} paint.
    */
   public JDRPaint readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      JDRPaintLoader loader = JDR.getPaintLoader();
      JDRPaint startPaint = loader.loadJDR(din, version);

      if (startPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      JDRPaint endPaint = loader.loadJDR(din, version);

      if (endPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      int direction = din.readInt();

      if (version < 1.3f)
      {
         return new JDRGradient(direction, startPaint, endPaint);
      }
      else
      {
         return new JDRRadial(direction, startPaint, endPaint);
      }
   }

   /**
    * Gets the character identifying {@link JDRRadial} paint in
    * AJR format. AJR versions
    * before 1.3 did not support radial paint, so if version 
    * &lt; 1.3, returns {@link JDRGradient} ID.
    * @return the {@link JDRRadial} paint ID ('D') if
    * version &gt;= 1.3,
    * otherwise the {@link JDRGradient} paint ID ('G')
    */
   public char getAJRid(float version)
   {
      // convert to gradient paint if version < 1.3
      return (version < 1.3f ? 'G' : 'D');
   }

   /**
    * Writes specified radial paint in AJR format. AJR versions
    * before 1.3 did not support radial paint, so if version 
    * &lt; 1.3, the nearest gradient equivalent is saved instead.
    * @param paint the paint to save (must be of type 
    * {@link JDRRadial})
    */
   public void writeAJR(JDRPaint paint,
                        PrintWriter out,
                        float version)
   throws IOException
   {
      JDRPaintLoader loader = AJR.getPaintLoader();

      if (version < 1.3f && (paint instanceof JDRRadial))
      {
         JDRGradient c = ((JDRRadial)paint).getJDRGradient();
         JDRPaint startColor = c.getStartColor();
         JDRPaint endColor = c.getEndColor();

         loader.saveAJR(startColor,out,version);
         loader.saveAJR(endColor,out,version);

         AJR.writeInt(out, c.getDirection());
      }
      else
      {
         JDRRadial c = (JDRRadial)paint;
         JDRPaint startColor = c.getStartColor();
         JDRPaint endColor = c.getEndColor();

         loader.saveAJR(startColor,out,version);
         loader.saveAJR(endColor,out,version);

         AJR.writeInt(out, c.getStartLocation());
      }
   }

   /**
    * Reads radial paint specified in AJR format. AJR versions
    * prior to 1.3 did not support radial paint, so if 
    * version &lt; 1.3, returns {@link JDRGradient} paint 
    * otherwise returns {@link JDRRadial} paint.
    */
   public JDRPaint readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      JDRPaintLoader loader = AJR.getPaintLoader();
      JDRPaint startPaint = loader.loadAJR(in, version);

      if (startPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      JDRPaint endPaint = loader.loadAJR(in, version);

      if (endPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      int direction = AJR.readInt(in);

      if (version < 1.3f)
      {
         return new JDRGradient(direction, startPaint, endPaint);
      }
      else
      {
         return new JDRRadial(direction, startPaint, endPaint);
      }
   }

   public int getConfigId()
   {
      return 4;
   }

   /**
    * @param paint the paint to save (must be {@link JDRRadial}).
    */
   public String getConfigString(JDRPaint paint)
   {
      JDRRadial c = (JDRRadial)paint;
      JDRPaint startColor = c.getStartColor();
      JDRPaint endColor = c.getEndColor();

      JDRPaintLoader loader = JDR.getPaintLoader();

      String specs = loader.getConfigString(startColor)
                   + ","
                   + loader.getConfigString(endColor)
                   + ","
                   + c.getStartLocation();

      return specs;
   }

   public JDRPaint parseConfig(String specs)
   throws InvalidFormatException
   {
      JDRPaintLoader loader = JDR.getPaintLoader();
      JDRPaint startPaint = loader.parseConfig(specs);

      specs = loader.getConfigRemainder();

      JDRPaint endPaint = loader.parseConfig(specs);

      specs = loader.getConfigRemainder();

      String[] split = specs.split(",", 2);

      int direction = 0;

      try
      {
         direction = Integer.parseInt(split[0]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidFormatException
            ("Invalid radial paint specs: start location missing");
      }

      if (split.length == 1)
      {
         remainder = "";
      }
      else
      {
         remainder = split[1];
      }

      return new JDRRadial(direction, startPaint, endPaint);
   }

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder()
   {
      return remainder;
   }

   private String remainder="";
}
