// File          : JDRRadialGridListener.java
// Date          : 17th August 2010
// Last Modified : 17th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for radial grids.
 * @author Nicola L C Talbot
 */

public class JDRRadialGridListener implements JDRGridLoaderListener
{
   public byte getJDRid(float version)
   {
      return 1;
   }

   public void writeJDR(JDRGrid object,
                    DataOutputStream dout,
                    float version)
   throws IOException
   {
      JDRRadialGrid grid = (JDRRadialGrid)object;

      if (version < 1.6f)
      {
         dout.writeByte((byte)grid.getUnit().getID());
         dout.writeInt((int)Math.round(grid.getMajorInterval()));
         dout.writeInt(grid.getSubDivisions());
      }
      else
      {
         dout.writeByte((byte)grid.getUnit().getID());
         dout.writeDouble(grid.getMajorInterval());
         dout.writeInt(grid.getSubDivisions());
         dout.writeInt(grid.getSpokes());
      }
   }

   public JDRGrid readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      byte unitID = din.readByte();

      JDRUnit unit = JDRUnit.getUnit(unitID);

      if (unit == null)
      {
         throw new InvalidUnitIDException(unitID);
      }

      double majorDivisions;

      if (version < 1.6f)
      {
         majorDivisions = din.readInt();
      }
      else
      {
         majorDivisions = din.readDouble();
      }

      int subDivisions = din.readInt();

      if (version < 1.6f)
      {
         return new JDRRectangularGrid(unit, majorDivisions, subDivisions);
      }

      int spokes = din.readInt();

      return new JDRRadialGrid(unit, majorDivisions, subDivisions, spokes);
   }

   public byte getAJRid(float version)
   {
      return 1;
   }

   public void writeAJR(JDRGrid object,
                        PrintWriter out,
                        float version)
   throws IOException
   {
      JDRRadialGrid grid = (JDRRadialGrid)object;

      AJR.writeInt(out, grid.getUnit().getID());

      if (version < 1.6f)
      {
         AJR.writeInt(out, (int)Math.round(grid.getMajorInterval()));
         AJR.writeInt(out, grid.getSubDivisions());
      }
      else
      {
         AJR.writeDouble(out, grid.getMajorInterval());
         AJR.writeInt(out, grid.getSubDivisions());
         AJR.writeInt(out, grid.getSpokes());
      }
   }

   public JDRGrid readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      int unitID = AJR.readInt(in);

      JDRUnit unit = JDRUnit.getUnit(unitID);

      if (unit == null)
      {
         throw new InvalidUnitIDException(unitID, AJR.getLineNum());
      }

      double majorDivisions;

      if (version < 1.6f)
      {
         majorDivisions = AJR.readInt(in);
      }
      else
      {
         majorDivisions = AJR.readDouble(in);
      }

      int subDivisions = AJR.readInt(in);

      if (version < 1.6f)
      {
         return new JDRRectangularGrid(unit, majorDivisions, subDivisions);
      }

      int spokes = AJR.readInt(in);

      return new JDRRadialGrid(unit, majorDivisions, subDivisions, spokes);
   }

}
