//File          : JDRRadialGrid.java
//Description   : Represents a radial grid.
//Author        : Nicola L.C. Talbot
//Date          : 17th August 2010
//Last Modified : 17th August 2010
//              http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr;

import java.awt.*;
import java.awt.geom.*;
import java.text.*;

/**
 * Class representing a radial grid. The origin is on the centre of
 * the page.
 * @author Nicola L C Talbot
 */
public class JDRRadialGrid extends JDRGrid
{
   /**
    * Initialises using the default settings. The defaults are:
    * bp units, major interval width of 100 units, 10
    * sub-divisions per major interval and 8 spokes.
    */
   public JDRRadialGrid()
   {
      try
      {
         set(JDRUnit.bp, 100.0, 10, 8);
      }
      catch (InvalidGridArgumentException ignore)
      {
      }
   }

   /**
    * Initialises using the given settings.
    * @param gridUnit the grid units
    * @param majorDiv the width of the major interval in terms of
    * the grid unit
    * @param subDiv the number of sub-divisions per major interval
    * @param gridSpokes the number of spokes in the grid
    * @throws InvalidGridArgumentException if the major interval
    * width is &lt;= 0 or if the sub divisions are negative or if
    * the number of spokes is &lt;= 0.
    */
   public JDRRadialGrid(JDRUnit gridUnit, double majorDiv, int subDiv,
     int gridSpokes)
     throws InvalidGridArgumentException
   {
      set(gridUnit, majorDiv, subDiv, gridSpokes);
   }

   public Object clone()
   {
      JDRRadialGrid grid = null;

      try
      {
         grid = new JDRRadialGrid(unit, majorDivisions, subDivisions,
                                  spokes);
      }
      catch (InvalidGridArgumentException ignore)
      {
      }

      return grid;
   }

   /**
    * Change the grid settings settings.
    * @param gridUnit the grid units
    * @param majorDiv the width of the major interval in terms of
    * the grid unit
    * @param subDiv the number of sub-divisions per major interval
    * @param gridSpokes the number of spokes in the grid
    * @throws InvalidGridArgumentException if the major interval
    * width is &lt;= 0 or if the sub divisions are negative or if
    * the number of spokes is &lt;= 0.
    */
   public void set(JDRUnit gridUnit, double majorDiv, int subDiv,
     int gridSpokes)
     throws InvalidGridArgumentException
   {
      setUnit(gridUnit);
      setMajorInterval(majorDiv);
      setSubDivisions(subDiv);
      setSpokes(gridSpokes);
   }

   /**
    * Sets the grid unit.
    * @param gridUnit the grid units
    */
   public void setUnit(JDRUnit gridUnit)
   {
      unit = gridUnit;
   }

   /**
    * Sets the major interval width.
    * @param majorDiv the major interval width in terms of the
    * grid's unit
    * @throws InvalidGridArgumentException if the argument is &lt;= 0
    */
   public void setMajorInterval(double majorDiv)
      throws InvalidGridArgumentException
   {
      if (majorDiv <= 0.0)
      {
         throw new InvalidGridArgumentException(majorDiv);
      }

      majorDivisions = majorDiv;
   }

   /**
    * Sets the number of sub divisions within the major interval.
    * @param subDiv the number of sub divisions
    * @throws InvalidGridArgumentException if the argument is &lt; 0
    */
   public void setSubDivisions(int subDiv)
     throws InvalidGridArgumentException
   {
      if (subDiv < 0)
      {
         throw new InvalidGridArgumentException(subDiv);
      }

      subDivisions = subDiv;
   }

   /**
    * Sets the number of spokes.
    * @param gridSpokes the number of spokes in the grid
    * @throws InvalidGridArgumentException if the number of spokes 
    * is &lt;= 0.
    */
   public void setSpokes(int gridSpokes)
      throws InvalidGridArgumentException
   {
      if (gridSpokes < 0)
      {
         throw new InvalidGridArgumentException(gridSpokes);
      }

      spokes = gridSpokes;
   }

   /**
    * Gets the grid unit.
    * @return the grid unit
    */
   public JDRUnit getUnit()
   {
      return unit;
   }

   /**
    * Gets the width of the major intervals.
    * @return the width of the major intervals in terms of the
    * grid's unit.
    */
   public double getMajorInterval()
   {
      return majorDivisions;
   }

   /**
    * Gets the number of sub divisions within a major interval.
    * @return the number of sub divisions.
    */
   public int getSubDivisions()
   {
      return subDivisions;
   }

   /**
    * Gets the number of spokes.
    * @return the number of spokes
    */
   public int getSpokes()
   {
      return spokes;
   }

   public Point2D getMajorTicDistance()
   {
      double major = unit.toBp(majorDivisions);

      return new Point2D.Double(major, major);
   }

   public Point2D getMinorTicDistance()
   {
      Point2D p = new Point2D.Double(0, 0);

      if (subDivisions > 0)
      {
         double minor = unit.toBp(majorDivisions)/subDivisions;

         p.setLocation(minor, minor);
      }

      return p;
   }

   public Point2D getFromCartesian(JDRPaper paper, double x, double y)
   {
      JDRRadialPoint p = new JDRRadialPoint();

      p.setLocation(
         unit.fromBp(x)-0.5*paper.getWidth(),
         unit.fromBp(y)-0.5*paper.getHeight());

      return p;
   }

   public void getFromCartesian(JDRPaper paper,
                                Point2D cartesianPoint,
                                Point2D target)
   {
      target.setLocation(
         unit.fromBp(cartesianPoint.getX())-0.5*paper.getWidth(),
         unit.fromBp(cartesianPoint.getY())-0.5*paper.getHeight());
   }

   public void getCartesian(JDRPaper paper,
                            Point2D original,
                            Point2D target)
   {
      JDRRadialPoint radialPoint = (JDRRadialPoint)original;

      // Shift the origin to the top left corner

      target.setLocation(
         unit.toBp(radialPoint.getX())+0.5*paper.getWidth(),
         unit.toBp(radialPoint.getY())+0.5*paper.getHeight());
   }

   public void drawGrid(Graphics2D g, JDRPaper paper,
      double scale)
   {
      double major = unit.toBp(getMajorInterval());
      double minor = getSubDivisions();

      if (minor > 0)
      {
         minor = major/minor;
      }

      double spokeAngle = getSpokeAngle();

      JDRRadialPoint rpt   = new JDRRadialPoint();
      Point2D.Double point = new Point2D.Double();

      double maxRadius;

      double halfW = 0.5*paper.getWidth();
      double halfH = 0.5*paper.getHeight();

      double halfPi = 0.5*Math.PI;
      double twoPi  = 2*Math.PI;

      double theta1 = Math.atan(halfH/halfW);
      double theta2 = halfPi - theta1;

      if (subDivisions == 0) g.setColor(majorGridColor);

      for (double angle = 0.0; angle < twoPi; angle += spokeAngle)
      {
         try
         {
            rpt.setParameters(0.0, angle);

            if (angle <= theta1)
            {
               maxRadius = halfW/Math.cos(angle);
            }
            else if (angle <= halfPi + theta2)
            {
               maxRadius = halfH/Math.sin(angle);
            }
            else if (angle <= Math.PI + theta1)
            {
               maxRadius = -halfW/Math.cos(angle);
            }
            else
            {
               maxRadius = -halfH/Math.sin(angle);
            }

            for (double radius = 0.0; radius <= maxRadius;
                 radius += major)
            {
               rpt.setRadius(radius);
               getCartesian(paper, rpt, point);

               if (subDivisions == 0)
               {
                  drawMajorTic(g, (int)Math.round(point.getX()*scale),
                                  (int)Math.round(point.getY()*scale));
               }
               else
               {
                  g.setColor(majorGridColor);
                  drawMajorTic(g, (int)Math.round(point.getX()*scale),
                                  (int)Math.round(point.getY()*scale));

                  g.setColor(minorGridColor);

                  for (int i = 1; i < getSubDivisions(); i++)
                  {
                     rpt.setRadius(radius+i*minor);
                     getCartesian(paper, rpt, point);

                     drawMinorTic(g, (int)Math.round(point.getX()*scale),
                                     (int)Math.round(point.getY()*scale));
                  }
               }
            }
         }
         catch (IllegalArgumentException ignore)
         {
         }
      }
   }

   /**
    * Gets the angle between spokes.
    * @return the angle between spokes in radians.
    */
   public double getSpokeAngle()
   {
      return 2.0*Math.PI/spokes;
   }

   public String getUnitLabel()
   {
      return unit.getLabel();
   }

   public JDRUnit getMainUnit()
   {
      return unit;
   }

   public String formatLocation(Point2D point)
   {
      JDRRadialPoint rpt = (JDRRadialPoint)point;

      DecimalFormat f = new DecimalFormat("0.0");

      return f.format(rpt.getRadius())+unit.getLabel()+","
            +f.format(Math.toDegrees(rpt.getAngle()));
   }

   public Point2D getClosestTic(JDRPaper paper, double x, double y)
   {
      // Convert the original point to radial co-ordinates

      JDRRadialPoint point = (JDRRadialPoint)getFromCartesian(paper,x,y);

      // Determine the closest spoke

      double angle = getSpokeAngle();

      int n = (int)Math.round(point.getAngle()/angle);

      // Closest spoke is the nth spoke

      point.setAngle(n*angle);

      // Now find the closest tic mark along this slope

      if (subDivisions > 0)
      {
         n = (int)Math.floor(point.getRadius()/majorDivisions);

         double R = n*majorDivisions;

         // lies between n and n+1 major tick mark along this slope

         double r0 = point.getRadius() - R;

         // Compute the length of each sub-division

         double length = majorDivisions/subDivisions;

         int n0 = (int)Math.round(r0/length);

         // point is closest to the n0 minor tick along this
         // interval.

         point.setRadius(R + n0*length);
      }
      else
      {
         n = (int)Math.round(point.getRadius()/majorDivisions);

         point.setRadius(n*majorDivisions);
      }

      // Get this point in Cartesian co-ordinates

      Point2D target = new Point2D.Double();

      getCartesian(paper, point, target);

      return target;
   }

   public JDRGridLoaderListener getListener()
   {
      return listener;
   }

   private static JDRRadialGridListener listener = new JDRRadialGridListener();

   /**
    * Stores the distance between the major tick marks in terms of
    * the unit given by {@link #unit}.
    */
   private double majorDivisions;

   /**
    * Stores the number of subdivisions within a major grid
    * interval.
    */
   private int subDivisions;

   /**
    * Stores the number of spokes.
    */
   private int spokes;

   /**
    * The unit used by the grid.
    */
   private JDRUnit unit;
}
