// File          : JDRPoint.java
// Date          : 1st February 2006
// Last Modified : 13th January 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.awt.*;
import java.awt.geom.*;
import java.io.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing a control point. Despite being a point it has
 * an associated area surrounding it (given by {@link #pointSize})
 * that allows a GUI user to select the point approximately.
 * @author Nicola L C Talbot
 */
public class JDRPoint extends JDRObject
{
   /**
    * Creates a new control point at the origin.
    */
   public JDRPoint()
   {
      super();
      x = 0;
      y = 0;
   }

   /**
    * Creates a control point at the given location.
    * @param p the location of the new point
    */
   public JDRPoint(Point p)
   {
      super();
      x = p.x;
      y = p.y;
   }

   /**
    * Creates a control point at the given location.
    * @param p the location of the new point
    */
   public JDRPoint(Point2D p)
   {
      super();
      x = p.getX();
      y = p.getY();
   }

   /**
    * Creates a control point at the given location.
    * @param px the x co-ordinate
    * @param py the y co-ordinate
    */
   public JDRPoint(double px, double py)
   {
      super();
      x = px;
      y = py;
   }

   /**
    * Sets the location of this point.
    * @param newX the new x co-ordinate
    * @param newY the new y co-ordinate
    */
   public void set(double newX, double newY)
   {
      x = newX;
      y = newY;
   }

   public double getX()
   {
      return x;
   }

   public double getY()
   {
      return y;
   }

   /**
    * Gets the paint used to draw this point when it is selected.
    */
   public Paint getSelectedPaint()
   {
      return selectColor;
   }

   /**
    * Sets the paint use to draw points of this class when selected.
    */
   public void setSelectedPaint(Paint paint)
   {
      selectColor = paint;
   }

   /**
    * Gets the paint used to draw this point when it isn't selected.
    */
   public Paint getUnselectedPaint()
   {
      return controlColor;
   }

   /**
    * Sets the paint used to draw points of this class when not
    * selected.
    */
   public void setUnselectedPaint(Paint paint)
   {
      controlColor = paint;
   }

   /**
    * Draws this control point (including bounding area given by
    * {@link #pointSize}).
    * @param g the graphics device on which to draw
    */
   public void draw(Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;

      Paint oldPaint = g2.getPaint();
      if (selected) g2.setPaint(getUnselectedPaint());

      double halfPointSize = pointSize*0.5;

      Rectangle2D.Double rect = 
         new Rectangle2D.Double(x-halfPointSize, y-halfPointSize,
                                pointSize, pointSize);
      g2.draw(rect);

      rect.x++;
      rect.y++;
      rect.width-=2;
      rect.height-=2;
      g2.setPaint(innerColour);
      g2.draw(rect);

      g2.setPaint(oldPaint);
   }

   /**
    * Draws control point in edit mode at given scale factor.
    * Note that this scales the location of the control point
    * and its bounding box, but it does not transform the
    * graphics device (which would affect the line width).
    * @param g graphics device
    * @param scale scaling factor
    */
   public void draw(Graphics g, double scale)
   {
      draw(g, scale,
           selected ? getSelectedPaint() : getUnselectedPaint(),
           pointSize);
   }

   public void draw(Graphics g, double scale, Paint color, double size)
   {
      Graphics2D g2 = (Graphics2D)g;

      g2.setPaint(color);

      double halfPointSize = size*0.5;

      double scalePointSize = size*scale;

      Rectangle2D.Double rect = 
         new Rectangle2D.Double((x-halfPointSize)*scale, 
                                (y-halfPointSize)*scale,
                                scalePointSize, scalePointSize);
      g2.draw(rect);

      double doubleScale = 2.0*scale;

      rect.x += scale;
      rect.y += scale;
      rect.width  -= doubleScale;
      rect.height -= doubleScale;
      g2.setPaint(innerColour);
      g2.draw(rect);
   }

   public BBox getBBox()
   {
      double halfSize = pointSize*0.5;

      return new BBox(x-halfSize,
                      y-halfSize,
                      x+halfSize,
                      y+halfSize);
   }

   public void mergeBBox(BBox box)
   {
      double halfSize = pointSize*0.5;

      double minX = x-halfSize;
      double minY = y-halfSize;
      double maxX = x+halfSize;
      double maxY = y+halfSize;

      box.merge(minX, minY, maxX, maxY);
   }

   public boolean contains(JDRPoint p)
   {
      if (this == p) return true;

      return (x-pointSize*0.5-1 <= p.x && p.x <= x+pointSize*0.5
            && y-pointSize*0.5-1 <= p.y && p.y <= y+pointSize*0.5);
   }

   public boolean contains(Point p)
   {
      return (x-pointSize*0.5-1 <= p.x && p.x <= x+pointSize*0.5
            && y-pointSize*0.5-1 <= p.y && p.y <= y+pointSize*0.5);
   }

   public boolean contains(Point2D p)
   {
      return (x-pointSize*0.5-1 <= p.getX() && p.getX() <= x+pointSize*0.5
            && y-pointSize*0.5-1 <= p.getY() && p.getY() <= y+pointSize*0.5);
   }

   public void transform(double[] matrix)
   {
      double newx = matrix[0]*x+matrix[2]*y+matrix[4];
      double newy = matrix[1]*x+matrix[3]*y+matrix[5];
      x = newx;
      y = newy;
   }

   public void translate(double px, double py)
   {
      x += px;
      y += py;
   }

   /**
    * Moves this point to the nearest point on the given line.
    * @see #moveToLine(double,double,double,double)
    */
   public void moveToLine(JDRLine line)
   {
      moveToLine(line.start.x, line.start.y, line.end.x, line.end.y);
   }

   /**
    * Moves this point to the nearest point on the given line.
    * @param x0 x co-ordinate of first point defining line
    * @param y0 y co-ordinate of first point defining line
    * @param x1 x co-ordinate of second point defining line
    * @param y1 y co-ordinate of second point defining line
    * @see #moveToLine(JDRLine)
    */
   public void moveToLine(double x0, double y0,
                          double x1, double y1)
   {
      // Need to trap near vertical and near horizontal lines of
      // symmetry to prevent arithmetic errors

      double dx = x0 - x1;
      double dy = y0 - y1;

      double m = dy/dx;
      double minv = dx/dy;

      if (Math.abs(dx) < 1)
      {
         // Line is approximately vertical

         x = minv*(y - y0) + x0;
      }
      else if (Math.abs(dy) < 1)
      {
         // Line is approximately horizontal

         y = m*(x - x0) + y0;
      }
      else
      {
         double px = (x * minv + y + m*x0 - y0)
                   / (m + minv);

         double py = m*(px - x0) + y0;

         x = px;
         y = py;
      }
   }

   public void scale(double factorX, double factorY)
   {
      x = factorX*x;
      y = factorY*y;
   }

   public void scaleX(double factor)
   {
      x = factor*x;
   }

   public void scaleY(double factor)
   {
      y = factor*y;
   }

   public void shear(double factorX, double factorY)
   {
      double oldx = x;
      double oldy = y;

      // left handed co-ordinate system
      x = oldx-factorX*oldy;
      y = oldy-factorY*oldx;
   }

   public void shear(double factor)
   {
      shear(factor, factor);
   }

   public void shearX(double factor)
   {
      shear(factor,0.0);
   }

   public void shearY(double factor)
   {
      shear(0.0,factor);
   }

   // angle in radians
   public void rotate(double angle)
   {
      double cosTheta = Math.cos(angle);
      double sinTheta = Math.sin(angle);

      double old_x = x;
      double old_y = y;

      x = old_x*cosTheta - old_y*sinTheta;
      y = old_x*sinTheta + old_y*cosTheta;
   }

   /**
    * Converts this point to a {@link Point2D} object.
    * @return this point as a {@link Point2D} object
    */
   public Point2D getPoint2D()
   {
      return new Point2D.Double(x,y);
   }

   /**
    * Gets a reflection of this point about the given line.
    * @param line the line of symmetry
    * @return reflected point
    */
   public Point2D getReflection(JDRLine line)
   {
      return getReflection(x, y, line);
   }

   /**
    * Reflects this point about the given line.
    * @param line the line of symmetry
    */
   public void reflect2D(JDRLine line)
   {
      Point2D p = getReflection(x, y, line);

      x = p.getX();
      y = p.getY();
   }

   public static Point2D getReflection(double px, double py, JDRLine line)
   {
      double newX, newY;

      // Need to trap near vertical and near horizontal lines of
      // symmetry to prevent arithmetic errors

      double dx = line.start.x - line.end.x;
      double dy = line.start.y - line.end.y;

      if (Math.abs(dx) < 1)
      {
         // The line is approximately vertical

         newX = 2*line.end.x - px;
         newY = py;
      }
      else if (Math.abs(dy) < 1)
      {
         // The line is approximately horizontal

         newX = px;
         newY = 2*line.end.y - py;
      }
      else
      {
         double m = dy/dx;
         double minv = dx/dy;

         // (xi, yi) is the point of intersection of the line of
         // symmetry and the line formed by this point and its
         // reflection.

         double xi = (px*minv + py + m*line.start.x - line.start.y)
                   / (m + minv);
         double yi = m*(xi-line.start.x) + line.start.y;

         newX = 2.0*xi - px;
         newY = 2.0*yi - py;
      }

      return new Point2D.Double(newX, newY);
   }

   /**
    * Converts this point to a {@link Point} object.
    * The co-ordinates are rounded to the nearest integer
    * using {@link Math#round(double)}.
    * @return this point as a {@link Point} object
    */
   public Point getPoint()
   {
      return new Point((int)Math.round(x),(int)Math.round(y));
   }

   public Object clone()
   {
      JDRPoint p = new JDRPoint(x,y);
      p.makeEqual(this);
      return p;
   }

   /**
    * Makes this point equal to another point.
    * @param p the other point
    */
   public void makeEqual(JDRPoint p)
   {
      super.makeEqual((JDRObject)p);
      x = p.x;
      y = p.y;
   }

   /**
    * Determines if this point equals the other object.
    * This calls {@link JDRObject#equals(Object)} in addition
    * to checking the co-ordinates.
    * @param obj the other object
    * @return true if this point is equivalent to the other object
    */
   public boolean equals(Object obj)
   {
      if (!super.equals(obj)) return false;

      if (!(obj instanceof JDRPoint)) return false;

      JDRPoint p = (JDRPoint)obj;

      return (x == p.x && y == p.y);
   }

   /**
    * Gets string representation of this point.
    * @return string representation of this point
    */
   public String toString()
   {
      return new String("JDRPoint("+x+","+y+")");
   }

   public String pgf(AffineTransform af)
   {
      return PGF.point(af, x, y);
   }

   public void saveSVG(PrintWriter out) throws IOException
   {
      SVG.savePoint(out, x, y);
   }

   public void saveEPS(PrintWriter out) throws IOException
   {
      EPS.savePoint(out, x, y);
   }

   public String info()
   {
      return "("+x+","+y+")";
   }

   /**
    * The x co-ordinate of this point.
    */
   public double x;
   /**
    * The y co-ordinate of this point.
    */
   public double y;
   /**
    * The size of the active area around this point.
    */
   public static double pointSize=10;

   public static Color innerColour = new Color(192,192,192,127);

   /**
    * The colour to draw control points.
    */
   public static Paint controlColor = new Color(255,200,0,200);

   /**
    * The colour to draw selected points.
    */
   public static Paint selectColor = new Color(255,0,0,200);

}
