// File          : JDRPathStyleLoader.java
// Date          : 29th August 2010
// Last Modified : 29th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class dealing with saving and loading path styles (line/fill
 * paint and stroke) to/from JDR and AJR files. This doesn't include
 * the segments that make up the path or the open/close attribute.
 * @author Nicola L C Talbot
 */
public class JDRPathStyleLoader
{
   /**
    * Creates new loader.
    */
   public JDRPathStyleLoader()
   {
      listeners_ = new Vector<JDRPathStyleListener>();
   }

   /**
    * Saves the given path styles in JDR format. This first writes the
    * ID character specified by 
    * {@link JDRPathStyleListener#getJDRid(float)} and then 
    * writes the specifications using 
    * {@link JDRPathStyleListener#writeJDR(JDRShape,DataOutputStream,float)}.
    * @param shape the shape whose style needs to be saved
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if an I/O error occurs
    * @see #loadJDR(DataInputStream,float,JDRShape)
    */
   public void saveJDR(JDRShape shape,
                    DataOutputStream dout,
                    float version)
      throws IOException,InvalidFormatException
   {
      JDRPathStyleListener listener = shape.getPathStyleListener();

      if (listener == null)
      {
         throw new InvalidObjectException("Shape "
            + shape.getClass().getName()
            + " doesn't have a valid listener");
      }

      dout.writeByte(listener.getJDRid(version));

      listener.writeJDR(shape,dout,version);
   }

   /**
    * Loads a path style specified in JDR format. This first reads a
    * byte and checks through the list of listeners
    * to determine which type is identified by that  
    * byte (using {@link JDRPathStyleListener#getJDRid(float)}).
    * @param din the input stream
    * @param version the JDR version number
    * @param shape the shape whose style needs to be set
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @see #saveJDR(JDRShape,DataOutputStream,float)
    * @see #addListener(JDRPathStyleListener)
    */
   public void loadJDR(DataInputStream din, float version, JDRShape shape)
      throws IOException,InvalidFormatException
   {
      byte id = din.readByte();

      for (Enumeration<JDRPathStyleListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRPathStyleListener listener = e.nextElement();

         if (listener.getJDRid(version) == id)
         {
            listener.readJDR(din, version, shape);

            return;
         }
      }

      throw new InvalidPathStyleIDException(id);
   }

   /**
    * Saves the given path style in AJR format. This first writes the
    * ID character specified by 
    * {@link JDRPathStyleListener#getAJRid(float)} and then 
    * writes the specifications using 
    * {@link JDRPathStyleListener#writeAJR(JDRShape,PrintWriter,float)}.
    * @param shape the shape whose style needs to be saved
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if an I/O error occurs
    * @see #loadAJR(BufferedReader,float,JDRShape)
    */
   public void saveAJR(JDRShape shape,
                    PrintWriter out,
                    float version)
      throws IOException,InvalidFormatException
   {
      JDRPathStyleListener listener = shape.getPathStyleListener();

      if (listener == null)
      {
         throw new InvalidObjectException("Shape "
            + shape.getClass().getName()
            + " doesn't have a valid listener");
      }

      AJR.writeInt(out, listener.getAJRid(version));
      listener.writeAJR(shape,out,version);
   }

   /**
    * Loads a path style specified in AJR format. This first reads a
    * integer and checks through the list of listeners
    * to determine which object type is identified by the  
    * integer (using {@link JDRPathStyleListener#getAJRid(float)}).
    * @param in the input stream
    * @param version the AJR version number
    * @param shape the shape whose style needs to be set
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @throws EOFException if the file ends unexpectedly
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * (specified by {@link AJR#buffLength}) is exceeded
    * @see #saveAJR(JDRShape,PrintWriter,float)
    * @see #addListener(JDRPathStyleListener)
    */
   public void loadAJR(BufferedReader in, float version,
      JDRShape shape)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      int id = AJR.readInt(in);

      for (Enumeration<JDRPathStyleListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRPathStyleListener listener = e.nextElement();

         if (listener.getAJRid(version) == id)
         {
            listener.readAJR(in, version, shape);

            return;
         }
      }

      throw new InvalidPathStyleIDException(id);
   }

   /**
    * Adds a new listener.
    * @param listener the new listener
    * @see #getListeners()
    */
   public void addListener(JDRPathStyleListener listener)
   {
      listeners_.add(listener);
   }

   /**
    * Gets all the listeners registered with this loader.
    * @return list of all listeners registered with this loader
    * @see #addListener(JDRPathStyleListener)
    */
   public Vector<JDRPathStyleListener> getListeners()
   {
      return listeners_;
   }

   private Vector<JDRPathStyleListener> listeners_;
}
