// File          : JDRPathStyleListener.java
// Date          : 29th August 2010
// Last Modified : 29th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

/**
 * Path style listener interface.
 * @author Nicola L C Talbot
 * @see JDRPathStyleLoader
 */

public interface JDRPathStyleListener
{
   /**
    * Gets the number identifying the associated path style type
    * in JDR format.
    * @param version the JDR version number
    * @return JDR path style id
    */
   public byte getJDRid(float version);

   /**
    * Writes the path style of the specified shape in JDR format.
    * (Doesn't include
    * the path style ID returned by {@link #getJDRid(float)} which
    * is dealt with by the path style loader.)
    * @param shape the shape whose path style needs to be saved
    * @param dout the output stream on which to write
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    */
   public void writeJDR(JDRShape shape,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException;

   /**
    * Reads path style specified in JDR format. (Doesn't include the
    * path style ID returned by {@link #getJDRid(float)} which is dealt
    * with by the path style loader.)
    * @param din the input stream
    * @param version the JDR version number
    * @param shape the shape whose path style needs to be set
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the object format
    */
   public void readJDR(DataInputStream din, float version, JDRShape shape)
   throws IOException,InvalidFormatException;

   /**
    * Gets the number identifying the associated path style type
    * in AJR format.
    * @param version the AJR version number
    * @return AJR path style id
    */
   public int getAJRid(float version);

   /**
    * Writes the specified path style in AJR format. (Doesn't include
    * the path style ID returned by {@link #getAJRid(float)} which
    * is dealt with by the path style loader.)
    * @param shape the shape whose path style needs to be saved
    * @param out the output stream on which to write
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    */
   public void writeAJR(JDRShape shape,
                    PrintWriter out,
                    float version)
   throws IOException,InvalidFormatException;

   /**
    * Reads path style specified in AJR format. (Doesn't include the
    * path style ID returned by {@link #getAJRid(float)} which is
    * dealt with by the path style loader.)
    * @param in the input stream
    * @param version the AJR version number
    * @param shape the shape whose path style needs to be set
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the object format
    * @throws java.nio.BufferOverflowException if the {@link uk.ac.uea.cmp.nlct.jdr.io.AJR}
    * buffer overflows
    * @throws EOFException if the end of file occurs unexpectedly
    */
   public void readAJR(BufferedReader in, float version, JDRShape shape)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException;

}
