// File          : JDRPathListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for paths.
 * @author Nicola L C Talbot
 */

public class JDRPathListener implements JDRObjectLoaderListener
{
   public char getJDRid(float version)
   {
      return 'P';
   }

   public void writeJDR(JDRObject object,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException
   {
      JDRPath path = (JDRPath)object;

      if (version < 1.6)
      {
         JDRPaintLoader paintLoader = JDR.getPaintLoader();

         paintLoader.saveJDR(path.getLinePaint(), dout, version);

         paintLoader.saveJDR(path.getFillPaint(), dout, version);

         path.getStroke().save(dout, version);
      }
      else
      {
         JDRPathStyleLoader pathStyleLoader = JDR.getPathStyleLoader();

         pathStyleLoader.saveJDR(path, dout, version);
      }

      writeJDRPathSpecs(path, dout, version);
   }

   protected void writeJDRPathSpecs(JDRShape path,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException
   {
      dout.writeChar(path.isClosed() ? 'C' : 'O');

      int n = path.size();
      dout.writeInt(n);

      JDRSegmentLoader segmentLoader = JDR.getSegmentLoader();

      for (int i = 0; i < n; i++)
      {
         JDRSegment segment = (JDRSegment)path.get(i);

         if (i == 0 && version >= 1.3f)
         {
            dout.writeDouble(segment.getStartX());
            dout.writeDouble(segment.getStartY());
         }

         segmentLoader.saveJDR(segment, dout, version);
      }
   }

   public JDRObject readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      JDRPath path = new JDRPath();

      if (version < 1.6)
      {
         JDRPaintLoader paintLoader = JDR.getPaintLoader();

         path.setLinePaint(paintLoader.loadJDR(din, version));
         path.setFillPaint(paintLoader.loadJDR(din, version));
         path.setStroke(JDRBasicStroke.read(din, version));
      }
      else
      {
         JDRPathStyleLoader pathStyleLoader = JDR.getPathStyleLoader();

         pathStyleLoader.loadJDR(din, version, path);
      }

      readJDRPathSpecs(path, din, version);

      return path;
   }

   protected void readJDRPathSpecs(JDRShape path, DataInputStream din, 
      float version)
   throws IOException,InvalidFormatException
   {
      char c = din.readChar();
      boolean closedPath;

      if (c == 'O')
      {
         closedPath = false;
      }
      else if (c == 'C')
      {
         closedPath = true;
      }
      else
      {
         throw new InvalidOpenCloseIDException(c);
      }

      int n = din.readInt();

      if (n > path.getCapacity())
      {
         path.setCapacity(n);
      }

      double x = 0;
      double y = 0;

      if (version >= 1.3f)
      {
         x = din.readDouble();
         y = din.readDouble();
      }

      JDRSegmentLoader segmentLoader = JDR.getSegmentLoader();

      for (int i = 0; i < n; i++)
      {
         JDRSegment segment 
            = (JDRSegment)segmentLoader.loadJDR(x, y, din, version);

         if (closedPath && i == n-1)
         {
            path.close(segment);
         }
         else
         {
            path.add(segment);
         }

         x = segment.end.x;
         y = segment.end.y;
      }
   }

   public char getAJRid(float version)
   {
      return 'P';
   }

   protected void writeAJRPathSpecs(JDRShape path,
                    PrintWriter out,
                    float version)
   throws IOException,InvalidFormatException
   {
      AJR.writeChar(out, path.isClosed() ? 'C' : 'O');

      int n = path.size();
      AJR.writeInt(out, n);

      JDRSegmentLoader segmentLoader = AJR.getSegmentLoader();

      for (int i = 0; i < n; i++)
      {
         JDRSegment segment = (JDRSegment)path.get(i);

         if (i == 0 && version >= 1.3f)
         {
            AJR.writeDouble(out, segment.getStartX());
            AJR.writeDouble(out, segment.getStartY());
         }

         segmentLoader.saveAJR(segment, out, version);
      }
   }

   public void writeAJR(JDRObject object,
                        PrintWriter out,
                        float version)
   throws IOException,InvalidFormatException
   {
      JDRPath path = (JDRPath)object;

      if (version < 1.6)
      {
         JDRPaintLoader paintLoader = AJR.getPaintLoader();

         paintLoader.saveAJR(path.getLinePaint(), out, version);

         paintLoader.saveAJR(path.getFillPaint(), out, version);

         path.getStroke().saveAJR(out, version);
      }
      else
      {
         JDRPathStyleLoader pathStyleLoader = AJR.getPathStyleLoader();

         pathStyleLoader.saveAJR(path, out, version);
      }

      writeAJRPathSpecs(path, out, version);
   }

   protected void readAJRPathSpecs(JDRShape path, BufferedReader in,
      float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      char c = AJR.readChar(in);
      boolean closedPath;

      if (c == 'O')
      {
         closedPath = false;
      }
      else if (c == 'C')
      {
         closedPath = true;
      }
      else
      {
         throw new InvalidOpenCloseIDException(c);
      }

      int n = AJR.readInt(in);

      if (n > path.getCapacity())
      {
         path.setCapacity(n);
      }

      double x = 0;
      double y = 0;

      if (version >= 1.3f)
      {
         x = AJR.readDouble(in);
         y = AJR.readDouble(in);
      }

      JDRSegmentLoader segmentLoader = AJR.getSegmentLoader();

      for (int i = 0; i < n; i++)
      {
         JDRSegment segment 
            = (JDRSegment)segmentLoader.loadAJR(x, y, in, version);

         if (closedPath && i == n-1)
         {
            path.close(segment);
         }
         else
         {
            path.add(segment);
         }

         x = segment.end.x;
         y = segment.end.y;
      }
   }

   public JDRObject readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      JDRPath path = new JDRPath();

      if (version < 1.6)
      {
         JDRPaintLoader paintLoader = AJR.getPaintLoader();

         path.setLinePaint(paintLoader.loadAJR(in, version));
         path.setFillPaint(paintLoader.loadAJR(in, version));
         path.setStroke(JDRBasicStroke.readAJR(in, version));
      }
      else
      {
         JDRPathStyleLoader pathStyleLoader = AJR.getPathStyleLoader();

         pathStyleLoader.loadAJR(in, version, path);
      }

      readAJRPathSpecs(path, in, version);

      return path;
   }
}
