// File          : JDRPartialSegment.java
// Date          : 26th July 2010
// Last Modified : 26th July 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;
import uk.ac.uea.cmp.nlct.jdr.marker.*;

/**
 * Class representing a partial segment. This is used as a joining
 * segment for symmetric paths where the end point isn't anchored to
 * the line of symmetry.
 * @author Nicola L C Talbot
 * @see JDRPath
 */
public class JDRPartialSegment extends JDRObject implements JDRPathSegment
{
   /**
    * Creates a new partial segment with null starting point and
    * null line of symmetry.
    */
   public JDRPartialSegment()
   {
      start = null;
   }

   /**
    * Creates a new partial segment with given starting point and
    * line of symmetry.
    */
   public JDRPartialSegment(JDRPoint point, JDRLine line)
   {
      start = point;
      line_ = line;
   }

   public void setStart(JDRPoint point)
   {
      start = point;
   }

   public void setStart(Point2D point)
   {
      if (start == null)
      {
         start = new JDRPoint(point);
      }
      else
      {
         start.x = point.getX();
         start.y = point.getY();
      }
   }

   public void setEnd(JDRPoint point)
   {
      Point2D p = point.getReflection(line_); 

      if (start == null)
      {
         start = new JDRPoint(p);
      }
      else
      {
         start.x = p.getX();
         start.y = p.getY();
      }
   }

   public JDRPathSegment getReflection(JDRLine line)
   {
      return new JDRPartialSegment(getEnd(), line);
   }

   public JDRPathSegment reverse()
   {
     return getReflection(line_);
   }

   public void setSymmetryLine(JDRLine line)
   {
      line_ = line;
   }

   public JDRLine getSymmetryLine()
   {
      return line_;
   }

   public JDRPathSegment split()
   {
      Point2D midP = getP(0.5);

      return new JDRPartialSegment(new JDRPoint(midP), line_);
   }

   public Object clone()
   {
      return new JDRPartialSegment((JDRPoint)start.clone(), line_);
   }

   /**
    * Appends this segment to the given path.
    * @param path the path to which this segment must be appended
    */
   public void appendToGeneralPath(GeneralPath path)
   {
      Point2D p = start.getReflection(line_);

      path.moveTo((float)p.getX(), (float)p.getY());
   }

   public JDRPathSegment convertToSegment()
   {
      return this;
   }

   public JDRPathSegment convertToLine()
   {
      return new JDRPartialLine(start, line_);
   }

   public JDRPathSegment convertToBezier()
   {
      return new JDRPartialBezier(start, line_);
   }

   public double getStartX()
   {
      return start.x;
   }

   public double getStartY()
   {
      return start.y;
   }

   public JDRPoint getStart()
   {
      return start;
   }

   public JDRPoint getEnd()
   {
      Point2D p = start.getReflection(line_);

      return new JDRPoint(p);
   }

   public Point2D getEnd2D()
   {
      return start.getReflection(line_);
   }

   public JDRSegment getFullSegment()
   {
      return new JDRSegment(getStart(), getEnd());
   }

   public String pgf(AffineTransform af)
   {
      Point2D p = start.getReflection(line_);

      return "\\pgfpathmoveto{"+PGF.point(af, p.getX(), p.getY())+"}";
   }

   public double getEndX()
   {
      Point2D p = start.getReflection(line_);

      return p.getX();
   }

   public double getEndY()
   {
      Point2D p = start.getReflection(line_);

      return p.getY();
   }

   public Point2D getP(double t)
   {
      Point2D p = start.getReflection(line_);

      return new Point2D.Double(start.x+(p.getX()-start.x)*t,
                                start.y+(p.getY()-start.y)*t);
   }

   public Point2D getdP(double t)
   {
      return getdP();
   }

   public Point2D getdP()
   {
      Point2D p = start.getReflection(line_);

      p.setLocation(p.getX()-start.getX(),
                    p.getY()-start.getY());

      return p;
   }

   public Point2D getdP0()
   {
      return getdP();
   }

   public Point2D getdP1()
   {
      return getdP();
   }

   public void setSelected(boolean flag)
   {
      selected = flag;

      if (flag == false)
      {
         start.selected = false;
      }
   }

   public void setEditedControls(boolean flag)
   {
      start.selected = flag;
   }

   public boolean isEdited()
   {
      return false;
   }

   public void drawControls(Graphics g, boolean endPoint, double scale)
   {
   }

   public void drawSelectedNoControls(Graphics g, double scale)
   {
      Graphics2D g2 = (Graphics2D) g;

      Point2D p = start.getReflection(line_);

      Stroke oldStroke = g2.getStroke();
      g2.setPaint(start.getSelectedPaint());
      g2.setStroke(JDRSegment.guideStroke);

      g2.drawLine((int)(start.x*scale), (int)(start.y*scale),
                  (int)(p.getX()*scale), (int)(p.getY()*scale));

      g2.setStroke(oldStroke);
   }

   public void drawDraft(Graphics g, double scale, boolean drawEnd)
   {
      Graphics2D g2 = (Graphics2D) g;

      draw(g, scale);

      if (isSelected())
      {
         drawSelectedNoControls(g, scale);
      }
   }

   public void draw(Graphics g, double scale)
   {
   }

   /**
    * Gets this segment's bounding box including control points.
    * The size of the control point is given by 
    * {@link JDRPoint#pointSize}
    */
   public BBox getControlBBox()
   {
      BBox box = start.getBBox();

      box.merge(start.getReflection(line_));

      return box;
   }

   public void mergeControlBBox(BBox box)
   {
      start.mergeBBox(box);
      box.merge(start.getReflection(line_));
   }

   public void draw(Graphics g)
   {
      draw(g, 1.0);
   }

   /**
    * Doesn't transform the starting point. Only transforms any
    * associated control points.
    */
   public void transform(double[] matrix)
   {
   }


   /**
    * Doesn't translate the starting point. Only translates any
    * associated control points.
    */
   public void translate(double x, double y)
   {
   }

   /**
    * Doesn't rotate the starting point. Only rotates any
    * associated control points.
    */
   public void rotate(Point2D p, double angle)
   {
   }

   /**
    * Doesn't rotate the starting point. Only rotates any
    * associated control points.
    */
   public void rotate(double angle)
   {
   }

   /**
    * Doesn't scale the starting point. Only scales any
    * associated control points.
    */
   public void scale(Point2D p, double sx, double sy)
   {
   }

   /**
    * Doesn't scale the starting point. Only scales any
    * associated control points.
    */
   public void scale(double sx, double sy)
   {
   }

   /**
    * Doesn't shear the starting point. Only shears any
    * associated control points.
    */
   public void shear(Point2D p, double sx, double sy)
   {
   }

   /**
    * Doesn't shear the starting point. Only shears any
    * associated control points.
    */
   public void shear(double sx, double sy)
   {
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;
      if (obj == null) return false;
      if (!(obj instanceof JDRPartialSegment)) return false;

      JDRPartialSegment seg = (JDRPartialSegment)obj;

      return seg.start.equals(start)
          && line_.equals(seg.getSymmetryLine());
   }

   public void makeEqual(JDRPartialSegment seg)
   {
      start = seg.start;
      line_ = seg.line_;
   }

   public void saveSVG(PrintWriter out) throws IOException
   {
      out.println("M ");
      SVG.savePoint(out, getEnd2D());
   }

   public void saveEPS(PrintWriter out) throws IOException
   {
      EPS.savePoint(out, getEnd2D());
   }

   public JDRMarker getStartMarker()
   {
      return startMarker;
   }

   public JDRMarker getEndMarker()
   {
      return endMarker;
   }

   public void setStartMarker(JDRMarker marker)
   {
      startMarker = marker;
   }

   public void setEndMarker(JDRMarker marker)
   {
      endMarker = marker;
   }

   public int controlCount()
   {
      return 1;
   }

   public JDRPoint getControl(int index)
      throws IndexOutOfBoundsException
   {
      if (index == 0) return start;

      throw new IndexOutOfBoundsException("No control point at index "+index);
   }

   public int getControlIndex(JDRPoint point)
      throws NoSuchElementException
   {
      if (point == start) return 0;

      throw new NoSuchElementException();
   }

   public JDRObjectLoaderListener getListener()
   {
      return listener;
   }

   public String info()
   {
      return "partial segment: start="+start.info()
        +", symmetry="+line_.info();
   }

   public BBox getBBox()
   {
      Point2D endPt = getEnd2D();

      double minX = (start.x < endPt.getX() ? start.x : endPt.getX());
      double minY = (start.y < endPt.getY() ? start.y : endPt.getY());
      double maxX = (start.x > endPt.getX() ? start.x : endPt.getX());
      double maxY = (start.y > endPt.getY() ? start.y : endPt.getY());

      return new BBox(minX, minY, maxX, maxY);
   }

   public boolean isGap() {return true;}

   private JDRMarker startMarker=null, endMarker=null;

   protected JDRPoint start;
   protected JDRLine line_;

   private static JDRPartialSegmentLoaderListener listener
      = new JDRPartialSegmentLoaderListener();
}
