// File          : JDRPartialBezier.java
// Date          : 26th July 2010
// Last Modified : 18th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing a partial B&eacute;zier curve.
 * @author Nicola L C Talbot
 * @see JDRPath
 */
public class JDRPartialBezier extends JDRPartialSegment 
{
   /**
    * Creates a new partial B&eacute; with null starting point and
    * null line of symmetry.
    */
   public JDRPartialBezier()
   {
      super();
      control = new JDRPoint();
   }

   /**
    * Creates a new partial B&eacute; with given starting point and
    * line of symmetry.
    */
   public JDRPartialBezier(JDRPoint point, JDRLine line)
   {
      super(point, line);
      control = new JDRPoint();
      flatten();
   }

   /**
    * Creates a new partial B&eacute; with given starting point,
    * control point and line of symmetry.
    */
   public JDRPartialBezier(JDRPoint point, JDRPoint c, JDRLine line)
   {
      super(point, line);
      control = c;
   }

   public Object clone()
   {
      return new JDRPartialBezier((JDRPoint)start.clone(), 
         (JDRPoint)control.clone(), line_);
   }

   public void flatten()
   {
      Point2D dP = ((JDRPartialSegment)this).getdP();
      setGradient(dP);
   }

   public void setGradient(Point2D dP)
   {
      control.set(dP.getX()/3 + start.x,
                  dP.getY()/3 + start.y);
   }

   public Point2D getdP0()
   {
      return new Point2D.Double(3*(control.x-start.x),
                                3*(control.y-start.y));
   }

   public Point2D getdP1()
   {
      Point2D c2 = getControl2();
      Point2D p = start.getReflection(line_);

      p.setLocation(3*(p.getX()-c2.getX()),
                    3*(p.getY()-c2.getY()));

      return p;
   }

   public Point2D getP(double t)
   {
      Point2D c2 = getControl2();
      Point2D p = start.getReflection(line_);

      double one_minus_t = (1-t);
      double one_minus_t_sq = one_minus_t*one_minus_t;
      double t_sq = t*t;
      double J0 = one_minus_t_sq*one_minus_t;
      double J1 = 3*t*one_minus_t_sq;
      double J2 = 3*t_sq*one_minus_t;
      double J3 = t_sq*t;

      p.setLocation
      (
         J0*start.x + J1*control.x + J2*c2.getX() + J3*p.getX(),
         J0*start.y + J1*control.y + J2*c2.getY() + J3*p.getY()
      );

      return p;
   }

   public void setEditedControls(boolean flag)
   {
      super.setEditedControls(flag);
      control.selected = flag;
   }

   public void setSelected(boolean flag)
   {
      selected = flag;

      if (flag == false)
      {
         start.selected = false;
         control.selected = false;
      }
   }

   public boolean isSelected()
   {
      return control.selected;
   }

   public JDRPoint getControl1()
   {
      return control;
   }

   public void setControl(double x, double y)
   {
      control.x = x;
      control.y = y;
   }

   public Point2D getControl2()
   {
      return control.getReflection(line_);
   }

   public JDRPathSegment getReflection(JDRLine line)
   {
      return new JDRPartialBezier(getEnd(),
        new JDRPoint(getControl2()), line);
   }

   public JDRSegment getFullSegment()
   {
      return new JDRBezier(getStart(), (JDRPoint)control.clone(), 
         new JDRPoint(getControl2()), getEnd());
   }

   public String pgf(AffineTransform af)
   {
      Point2D end = start.getReflection(line_);
      Point2D c2 = getControl2();

      return "\\pgfpathcurveto{"
        + control.pgf(af) + "}{"
        + PGF.point(af, c2.getX(), c2.getY()) + "}{"
        + PGF.point(af, end.getX(), end.getY())+"}";
   }

   public void saveSVG(PrintWriter out) throws IOException
   {
      out.println("C ");
      control.saveSVG(out);
      SVG.savePoint(out, getControl2());
      SVG.savePoint(out, getEnd2D());
   }

   public void saveEPS(PrintWriter out) throws IOException
   {
      control.saveEPS(out);
      EPS.savePoint(out, getControl2());
      EPS.savePoint(out, getEnd2D());
      out.println("curveto");
   }

   /**
    * Appends this curve to the given path.
    * @param path the path to which this segment must be appended
    */
   public void appendToGeneralPath(GeneralPath path)
   {
      Point2D p = start.getReflection(line_);
      Point2D c = control.getReflection(line_);

      path.curveTo(
         (float)control.getX(), (float)control.getY(),
         (float)c.getX(), (float)c.getY(),
         (float)p.getX(), (float)p.getY());
   }

   public JDRPathSegment convertToSegment()
   {
      return new JDRPartialSegment(start, line_);
   }

   public JDRPathSegment convertToLine()
   {
      return new JDRPartialLine(start, line_);
   }

   public JDRPathSegment convertToBezier()
   {
      return this;
   }

   public void drawControls(Graphics g, boolean endPoint, double scale)
   {
      Point2D p = start.getReflection(line_);
      Point2D c2 = control.getReflection(line_);

      Graphics2D g2 = (Graphics2D)g;

      Stroke oldStroke = g2.getStroke();
      Paint oldPaint = g2.getPaint();

      g2.setStroke(JDRSegment.guideStroke);
      g2.setPaint(JDRObject.draftColor);

      GeneralPath path = new GeneralPath();

      path.moveTo((float)(start.x*scale), (float)(start.y*scale));
      path.lineTo((float)(control.x*scale), (float)(control.y*scale));
      path.lineTo((float)(c2.getX()*scale), (float)(c2.getY()*scale));
      path.lineTo((float)(p.getX()*scale), (float)(p.getY()*scale));
      g2.draw(path);

      g2.setStroke(oldStroke);
      g2.setPaint(oldPaint);

      control.draw(g, scale);
   }

   public void draw(Graphics g, double scale)
   {
      Point2D p = start.getReflection(line_);
      Point2D c = control.getReflection(line_);

      Graphics2D g2 = (Graphics2D)g;

      g2.draw(new CubicCurve2D.Double(scale*start.x, scale*start.y,
         scale*control.getX(), scale*control.getY(),
         scale*c.getX(), scale*c.getY(),
         scale*p.getX(), scale*p.getY()));
   }

   public void drawSelectedNoControls(Graphics g, double scale)
   {
      Graphics2D g2 = (Graphics2D) g;

      Stroke oldStroke = g2.getStroke();
      g2.setPaint(start.getSelectedPaint());
      g2.setStroke(JDRSegment.guideStroke);

      draw(g, scale);

      g2.setStroke(oldStroke);
   }

   public void drawDraft(Graphics g, double scale, boolean drawEnd)
   {
      Graphics2D g2 = (Graphics2D) g;

      draw(g, scale);

      if (isSelected())
      {
         Stroke oldStroke = g2.getStroke();
         g2.setPaint(start.getSelectedPaint());
         g2.setStroke(JDRSegment.guideStroke);

         draw(g, scale);

         g2.setStroke(oldStroke);
      }

      drawControls(g, drawEnd, scale);
   }

   public BBox getControlBBox()
   {
      BBox box = super.getControlBBox();

      control.mergeBBox(box);
      box.merge(control.getReflection(line_));

      return box;
   }

   public void mergeControlBBox(BBox box)
   {
      super.mergeControlBBox(box);
      control.mergeBBox(box);
      box.merge(control.getReflection(line_));
   }


   public void transform(double[] matrix)
   {
      control.transform(matrix);
   }

   public void translate(double x, double y)
   {
      control.translate(x, y);
   }

   public void rotate(Point2D p, double angle)
   {
      control.rotate(p, angle);
   }

   public void rotate(double angle)
   {
      control.rotate(angle);
   }

   public void scale(Point2D p, double sx, double sy)
   {
      control.scale(p, sx, sy);
   }

   public void scale(double sx, double sy)
   {
      control.scale(sx, sy);
   }

   public void shear(Point2D p, double sx, double sy)
   {
      control.shear(p, sx, sy);
   }

   public void shear(double sx, double sy)
   {
      control.shear(sx, sy);
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;
      if (obj == null) return false;
      if (!(obj instanceof JDRPartialBezier)) return false;

      JDRPartialBezier seg = (JDRPartialBezier)obj;

      return seg.start.equals(start)
          && seg.control.equals(control)
          && line_.equals(seg.getSymmetryLine());
   }

   public JDRObjectLoaderListener getListener()
   {
      return listener;
   }

   public int controlCount()
   {
      return 2;
   }

   public JDRPoint getControl(int index)
      throws IndexOutOfBoundsException
   {
      if (index == 0) return start;

      if (index == 1) return control;

      throw new IndexOutOfBoundsException("No control point at index "+index);
   }

   public int getControlIndex(JDRPoint point)
      throws NoSuchElementException
   {
      if (point == start) return 0;

      if (point == control) return 1;

      throw new NoSuchElementException();
   }

   public BBox getBBox()
   {
      Point2D endPt = getEnd2D();
      Point2D c2 = getControl2();

      double minX = start.x;
      double minY = start.y;
      double maxX = start.x;
      double maxY = start.y;

      if (minX > control.x)    minX = control.x;
      if (minX > c2.getX())    minX = c2.getX();
      if (minX > endPt.getX()) minX = endPt.getX();

      if (minY > control.y)    minY = control.y;
      if (minY > c2.getY())    minY = c2.getY();
      if (minY > endPt.getY()) minY = endPt.getY();

      if (maxX < control.x)    maxX = control.x;
      if (maxX < c2.getX())    maxX = c2.getX();
      if (maxX < endPt.getX()) maxX = endPt.getX();

      if (maxY < control.y)    maxY = control.y;
      if (maxY < c2.getY())    maxY = c2.getY();
      if (maxY < endPt.getX()) maxY = endPt.getX();

      return new BBox(minX, minY, maxX, maxY);
   }

   public boolean isGap() {return false;}

   public String info()
   {
      return "partial Bezier: start="+start.info()
        +", control="+control.info()
        +", symmetry="+line_.info();
   }

   protected JDRPoint control;

   private static JDRPartialBezierLoaderListener listener
      = new JDRPartialBezierLoaderListener();
}
