// File          : JDRPaintLoaderListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

/**
 * JDRPaint loader listener interface.
 * @author Nicola L C Talbot
 * @see JDRPaintLoader
 */

public interface JDRPaintLoaderListener
{
   /**
    * Gets the character identifying the associated paint type
    * in JDR format.
    * @param version the JDR version number
    * @return JDR paint id
    */
   public char getJDRid(float version);

   /**
    * Writes the specified paint in JDR format. (Doesn't include
    * the paint ID returned by {@link #getJDRid(float)} which is 
    * dealt with by the paint loader.)
    * @param paint the paint to save
    * @param dout the output stream on which to write
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    */
   public void writeJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
   throws IOException;

   /**
    * Reads paint specified in JDR format. (Doesn't include the
    * paint ID returned by {@link #getJDRid(float)} which is dealt
    * with by the paint loader.)
    * @param din the input stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the paint format
    */
   public JDRPaint readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException;

   /**
    * Gets the character identifying the associated paint type
    * in AJR format.
    * @param version the AJR version number
    * @return AJR paint id
    */
   public char getAJRid(float version);

   /**
    * Writes the specified paint in AJR format. (Doesn't include
    * the paint ID returned by {@link #getAJRid(float)} which is 
    * dealt with by the paint loader.)
    * @param paint the paint to save
    * @param out the output stream on which to write
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    */
   public void writeAJR(JDRPaint paint,
                    PrintWriter out,
                    float version)
   throws IOException;

   /**
    * Reads paint specified in AJR format. (Doesn't include the
    * paint ID returned by {@link #getAJRid(float)} which is dealt
    * with by the paint loader.)
    * @param in the input stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the paint format
    * @throws java.nio.BufferOverflowException if the {@link uk.ac.uea.cmp.nlct.jdr.io.AJR}
    * buffer overflows
    * @throws EOFException if the end of file occurs unexpectedly
    */
   public JDRPaint readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException;

   /**
    * Gets configuration ID associated with the paint type.
    * This is used to identify paint in JpgfDraw's configuration
    * file.
    * @return configuration id
    */
   public int getConfigId();

   /**
    * Gets configuration file representation of the specified
    * paint. This does not include the paint ID which is 
    * specified by {@link #getConfigId()}.
    * @param paint the specified paint
    */
   public String getConfigString(JDRPaint paint);

   /**
    * Parses configuration file paint specification.
    * This does not include the paint ID which is 
    * specified by {@link #getConfigId()}.
    * @param paintspecs the paint specification
    * @throws InvalidFormatException if there is something wrong
    * with the paint format
    * @return the paint described by <code>paintspecs</code>
    */
   public JDRPaint parseConfig(String paintspecs)
   throws InvalidFormatException;

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder();
}
