// File          : JDRPaintLoader.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class dealing with saving and loading {@link JDRPaint} objects 
 * to/from JDR and AJR files.
 * @author Nicola L C Talbot
 */
public class JDRPaintLoader
{
   /**
    * Creates new loader.
    */
   public JDRPaintLoader()
   {
      listeners_ = new Vector<JDRPaintLoaderListener>();
   }

   /**
    * Saves the given paint in JDR format. This first writes the
    * ID character specified by 
    * {@link JDRPaintLoaderListener#getJDRid(float)} and then 
    * writes the paint specifications using 
    * {@link JDRPaintLoaderListener#writeJDR(JDRPaint,DataOutputStream,float)}
    * @param paint the paint that needs to be saved
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if an I/O error occurs
    * @see #loadJDR(DataInputStream,float)
    */
   public void saveJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
      throws IOException
   {
      if (paint == null)
      {
         dout.writeChar('T');
      }
      else
      {
         dout.writeChar(paint.getListener().getJDRid(version));
         paint.getListener().writeJDR(paint,dout,version);
      }
   }

   /**
    * Loads a paint specified in JDR format. This first reads a
    * character and checks through the list of paint listeners
    * to determine which paint type is identified by the  
    * character (using {@link JDRPaintLoaderListener#getJDRid(float)})
    * @param din the input stream
    * @param version the JDR version number
    * @return the specified paint
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the paint format
    * @see #saveJDR(JDRPaint,DataOutputStream,float)
    * @see #addListener(JDRPaintLoaderListener)
    */
   public JDRPaint loadJDR(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      char c = din.readChar();

      for (Enumeration<JDRPaintLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRPaintLoaderListener listener = e.nextElement();

         if (listener.getJDRid(version) == c)
         {
            return listener.readJDR(din, version);
         }
      }

      throw new InvalidPaintIDException(c);
   }

   /**
    * Saves the given paint in AJR format. This first writes the
    * ID character specified by 
    * {@link JDRPaintLoaderListener#getAJRid(float)} and then 
    * writes the paint specifications using 
    * {@link JDRPaintLoaderListener#writeAJR(JDRPaint,PrintWriter,float)}
    * @param paint the paint that needs to be saved
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if an I/O error occurs
    * @see #loadAJR(BufferedReader,float)
    */
   public void saveAJR(JDRPaint paint,
                    PrintWriter out,
                    float version)
      throws IOException
   {
      if (paint == null)
      {
         AJR.writeChar(out, 'T');
      }
      else
      {
         AJR.writeChar(out, paint.getListener().getAJRid(version));
         paint.getListener().writeAJR(paint,out,version);
      }
   }

   /**
    * Loads a paint specified in AJR format. This first reads a
    * character and checks through the list of paint listeners
    * to determine which paint type is identified by the  
    * character (using {@link JDRPaintLoaderListener#getAJRid(float)}).
    * @param in the input stream
    * @param version the AJR version number
    * @return the specified paint
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the paint format
    * @throws EOFException if the file ends unexpectedly
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * (specified by {@link AJR#buffLength}) is exceeded
    * @see #saveAJR(JDRPaint,PrintWriter,float)
    * @see #addListener(JDRPaintLoaderListener)
    */
   public JDRPaint loadAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      char c = AJR.readChar(in);

      for (Enumeration<JDRPaintLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRPaintLoaderListener listener = e.nextElement();

         if (listener.getAJRid(version) == c)
         {
            return listener.readAJR(in, version);
         }
      }

      throw new InvalidPaintIDException(c);
   }

   /**
    * Gets string describing paint for JpgfDraw's configuration file.
    * This returns the paint ID as specified by
    * {@link JDRPaintLoaderListener#getConfigId()}
    * optionally followed by a comma and
    * {@link JDRPaintLoaderListener#getConfigString(JDRPaint)}.
    * If {@link JDRPaintLoaderListener#getConfigString(JDRPaint)}
    * returns an empty string, only 
    * {@link JDRPaintLoaderListener#getConfigId()}
    * is returned.
    * @param paint the paint to describe
    */
   public String getConfigString(JDRPaint paint)
   {
      int id = paint.getListener().getConfigId();
      String config = paint.getListener().getConfigString(paint);

      if (config.equals(""))
      {
         return ""+id;
      }

      return ""+id+","+config;
   }

   /**
    * Parses configuration paint specs. The specs string must 
    * start with an integer identifying the paint (as specified
    * by {@link JDRPaintLoaderListener#getConfigId()}).
    */
   public JDRPaint parseConfig(String specs)
   throws InvalidFormatException
   {
      String[] split = specs.split(",", 2);

      int id = 0;

      try
      {
         id = Integer.parseInt(split[0]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidFormatException(
           "Invalid paint identifier '" + split[0]+"'");
      }

      String paintspecs = (split.length < 2 ? "" : split[1]);

      for (Enumeration<JDRPaintLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRPaintLoaderListener listener = e.nextElement();

         if (listener.getConfigId() == id)
         {
            JDRPaint paint = listener.parseConfig(paintspecs);

            remainder = listener.getConfigRemainder();

            return paint;
         }
      }

      remainder = paintspecs;

      throw new InvalidPaintIDException(id);
   }

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder()
   {
      return remainder;
   }

   /**
    * Adds a new paint listener to this loader.
    * @param listener the new listener to add
    */
   public void addListener(JDRPaintLoaderListener listener)
   {
      listeners_.add(listener);
   }

   /**
    * Gets a list of the paint listeners that have been add
    * to this loader.
    * @return list of paint listeners associated with this loader
    */
   public Vector<JDRPaintLoaderListener> getListeners()
   {
      return listeners_;
   }

   private Vector<JDRPaintLoaderListener> listeners_;

   private String remainder="";
}
