// File          : JDRPaint.java
// Date          : 1st February 2006
// Last Modified : 13th January 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.lang.Math;
import java.util.Hashtable;
import java.util.Enumeration;
import java.awt.*;

/**
 * Paint interface for JDR objects.
 * @author Nicola L C Talbot
 */
public interface JDRPaint
{
   /**
    * Gets the closest matching <code>java.awt.Color</code>.
    * @return closest matching Color.
    */
   public Color getColor();

   /**
    * Gets the <code>java.awt.Paint</code> equivalent.
    * @param box the bounding box of the object to which this
    * paint should be applied (required for gradient style paint)
    * @return the closest matching Paint
    */
   public Paint getPaint(BBox box);

   /**
    * Gets the closest matching {@link JDRColor}. Note that the
    * transformations between colour spaces is not exact.
    * @return the closest matching JDRColor.
    */
   public JDRColor getJDRColor();

   /**
    * Gets the closest matching {@link JDRColorCMYK}. Note that the
    * transformations between colour spaces is not exact.
    * @return the closest matching JDRColorCMYK.
    */
   public JDRColorCMYK getJDRColorCMYK();

   /**
    * Gets the closest matching {@link JDRColorHSB}. Note that the
    * transformations between colour spaces is not exact.
    * @return the closest matching JDRColorHSB.
    */
   public JDRColorHSB getJDRColorHSB();

   /**
    * Gets the closest matching {@link JDRGray}. Note that the
    * transformations between colour spaces is not exact.
    * @return the closest matching JDRGray.
    */
   public JDRGray getJDRGray();

   /**
    * Fades alpha channel.
    * @param value alpha value is scaled by
    */
   public void fade(double value);

   /**
    * Gets ID associated with this colour.
    * @return label uniquely identifying this colour
    */
   public String getID();

   /**
    * Gets PGF commands identifying this colour.
    * @param box the bounding box of the object to which this
    * colour applies (maybe null)
    * @return string containg PGF colour command
    */
   public String pgf(BBox box);

   /**
    * Gets PGF commands identifying this fill colour.
    * @param box the bounding box of the object to which this
    * colour applies (may be null)
    * @return string containg PGF colour command
    */
   public String pgffillcolor(BBox box);

   /**
    * Gets PGF commands identifying this stroke colour.
    * @param box the bounding box of the object to which this
    * colour applies (may be null)
    * @return string containg PGF colour command
    */
   public String pgfstrokecolor(BBox box);

   /**
    * Writes the required EPS commands to set this colour.
    * @param out the output stream
    * @param box the bounding box of the object to which this colour 
    * applies (may be null)
    * @throws IOException if I/O error occurs
    */
   public void saveEPS(PrintWriter out, BBox box)
      throws IOException;

   /**
    * Returns the EPS level supported by this colour.
    * @return the PostScript level that supports this colour
    */
   public int psLevel();

   /**
    * Gets this paint's transparency.
    * @return the transparency
    */
   public double getAlpha();

   /**
    * Gets the SVG commands to identify this paint. This
    * doesn't include the opacity.
    * @return string containing SVG commands identifying this 
    * paint
    */
   public String svg();

   /**
    * Gets the SVG commands to identify this fill paint.
    * This includes <code>fill</code> and <code>fill-opacity</code>
    * tags.
    * @return string containing SVG commands identifying this 
    * fill paint
    */
   public String svgFill();

   /**
    * Gets the SVG commands to identify this line paint.
    * This includes <code>stroke</code> and <code>stroke-opacity</code>
    * tags.
    * @return string containing SVG commands identifying this 
    * line paint
    */
   public String svgLine();

   /**
    * Gets the paint loader listener associated with this paint.
    */
   public JDRPaintLoaderListener getListener();

   /**
    * Gets a copy of this paint.
    * @return a copy of this paint
    */
   public Object clone();

   /**
    * Gets string representation of this paint.
    * @return string representation of this paint
    */
   public String toString();
}
