// File          : JDRObjectLoaderListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

/**
 * JDRObject loader listener interface.
 * @author Nicola L C Talbot
 * @see JDRObjectLoader
 */

public interface JDRObjectLoaderListener
{
   /**
    * Gets the character identifying the associated object type
    * in JDR format.
    * @param version the JDR version number
    * @return JDR object id
    */
   public char getJDRid(float version);

   /**
    * Writes the specified object in JDR format. (Doesn't include
    * the object ID returned by {@link #getJDRid(float)} or
    * flowframe or description data which are 
    * dealt with by the object loader.)
    * @param object the object to save
    * @param dout the output stream on which to write
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    */
   public void writeJDR(JDRObject object,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException;

   /**
    * Reads object specified in JDR format. (Doesn't include the
    * object ID returned by {@link #getJDRid(float)} or flowframe
    * or description data which are dealt
    * with by the object loader.)
    * @param din the input stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the object format
    */
   public JDRObject readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException;

   /**
    * Gets the character identifying the associated object type
    * in AJR format.
    * @param version the AJR version number
    * @return AJR object id
    */
   public char getAJRid(float version);

   /**
    * Writes the specified object in AJR format. (Doesn't include
    * the object ID returned by {@link #getAJRid(float)} or
    * flowframe or description data which are 
    * dealt with by the object loader.)
    * @param object the object to save
    * @param out the output stream on which to write
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    */
   public void writeAJR(JDRObject object,
                    PrintWriter out,
                    float version)
   throws IOException,InvalidFormatException;

   /**
    * Reads object specified in AJR format. (Doesn't include the
    * object ID returned by {@link #getAJRid(float)} or flowframe
    * or description data which are dealt
    * with by the object loader.)
    * @param in the input stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the object format
    * @throws java.nio.BufferOverflowException if the {@link uk.ac.uea.cmp.nlct.jdr.io.AJR}
    * buffer overflows
    * @throws EOFException if the end of file occurs unexpectedly
    */
   public JDRObject readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException;

}
