// File          : JDRObjectLoader.java
// Date          : 29th February 2008
// Last Modified : 18th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class dealing with saving and loading {@link JDRObject} objects 
 * to/from JDR and AJR files.
 * @author Nicola L C Talbot
 */
public class JDRObjectLoader
{
   /**
    * Creates new loader.
    */
   public JDRObjectLoader()
   {
      listeners_ = new Vector<JDRObjectLoaderListener>();
   }

   /**
    * Saves the given object in JDR format. This first writes the
    * ID character specified by 
    * {@link JDRObjectLoaderListener#getJDRid(float)} and then 
    * writes the object specifications using 
    * {@link JDRObjectLoaderListener#writeJDR(JDRObject,DataOutputStream,float)}.
    * Additionally writes flowframe and description information if
    * the object is an instance of {@link JDRCompleteObject}.
    * @param object the object that needs to be saved
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if an I/O error occurs
    * @see #loadJDR(DataInputStream,float)
    */
   public void saveJDR(JDRObject object,
                    DataOutputStream dout,
                    float version)
      throws IOException,InvalidFormatException
   {
      JDRObjectLoaderListener listener = object.getListener();

      if (listener == null)
      {
         throw new InvalidObjectException("Object "+object
            + " doesn't have a valid listener");
      }

      dout.writeChar(listener.getJDRid(version));
      listener.writeJDR(object,dout,version);

      if (object instanceof JDRCompleteObject)
      {
         JDRCompleteObject obj = (JDRCompleteObject)object;

         if (obj.flowframe == null)
         {
            dout.writeBoolean(false);
         }
         else
         {
            dout.writeBoolean(true);
            obj.flowframe.save(dout, version);
         }

         if (version >= 1.2f)
         {
            int n = obj.description.length();
            dout.writeInt(n);

            if (n > 0)
            {
               dout.writeChars(obj.description);
            }
         }
      }
   }

   /**
    * Loads an object specified in JDR format. This first reads a
    * character and checks through the list of listeners
    * to determine which type is identified by the  
    * character (using {@link JDRObjectLoaderListener#getJDRid(float)}).
    * Additionally reads flowframe and description information if
    * the object is an instance of {@link JDRCompleteObject}.
    * @param din the input stream
    * @param version the JDR version number
    * @return the specified object (maybe null if bitmap link is
    * invalid, in which case object needs to be discarded)
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @see #saveJDR(JDRObject,DataOutputStream,float)
    * @see #addListener(JDRObjectLoaderListener)
    */
   public JDRObject loadJDR(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      char c = din.readChar();

      for (Enumeration<JDRObjectLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRObjectLoaderListener listener = e.nextElement();

         if (listener.getJDRid(version) == c)
         {
            JDRObject object = listener.readJDR(din, version);

            if (object instanceof JDRCompleteObject)
            {
               JDRCompleteObject obj = (JDRCompleteObject)object;

               FlowFrame flowframe = null;

               if (din.readBoolean())
               {
                  flowframe = (FlowFrame)FlowFrame.read(din, version);
               }

               String description = "";

               if (version >= 1.2f)
               {
                  int n = din.readInt();

                  if (n > 0)
                  {
                     char[] desc = new char[n];

                     for (int i = 0; i < n; i++)
                     {
                        desc[i] = din.readChar();
                     }

                     description = new String(desc);
                  }
                  else if (n < 0)
                  {
                     throw new InvalidDescriptionLengthException(n);
                  }
               }

               if (obj != null)
               {
                  obj.flowframe = flowframe;
                  obj.description = description;
               }
            }

            return object;
         }
      }

      throw new InvalidObjectIDException(c);
   }

   /**
    * Saves the given object in AJR format. This first writes the
    * ID character specified by 
    * {@link JDRObjectLoaderListener#getAJRid(float)} and then 
    * writes the specifications using 
    * {@link JDRObjectLoaderListener#writeAJR(JDRObject,PrintWriter,float)}.
    * Additionally writes flowframe and description data if the
    * object is an instance of {@link JDRCompleteObject}.
    * @param object the object that needs to be saved
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if an I/O error occurs
    * @see #loadAJR(BufferedReader,float)
    */
   public void saveAJR(JDRObject object,
                    PrintWriter out,
                    float version)
      throws IOException,InvalidFormatException
   {
      JDRObjectLoaderListener listener = object.getListener();

      if (listener == null)
      {
         throw new InvalidObjectException("Object "+object
            + " doesn't have a valid listener");
      }

      AJR.writeChar(out, listener.getAJRid(version));
      listener.writeAJR(object,out,version);

      if (object instanceof JDRCompleteObject)
      {
         JDRCompleteObject obj = (JDRCompleteObject)object;

         if (obj.flowframe == null)
         {
            AJR.writeBoolean(out, false);
         }
         else
         {
            AJR.writeBoolean(out, true);
            obj.flowframe.saveAJR(out, version);
         }

         if (version >= 1.2f)
         {
            int n = obj.description.length();
            AJR.writeInt(out, n);

            if (n > 0)
            {
               out.println(obj.description);
            }
         }
      }
   }

   /**
    * Loads an object specified in AJR format. This first reads a
    * character and checks through the list of listeners
    * to determine which object type is identified by the  
    * character (using {@link JDRObjectLoaderListener#getAJRid(float)}).
    * Additionally reads flowframe and description information if
    * the object is an instance of {@link JDRCompleteObject}.
    * @param in the input stream
    * @param version the AJR version number
    * @return the specified object (maybe null if bitmap link is
    * invalid, in which case object needs to be discarded)
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the paint format
    * @throws EOFException if the file ends unexpectedly
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * (specified by {@link AJR#buffLength}) is exceeded
    * @see #saveAJR(JDRObject,PrintWriter,float)
    * @see #addListener(JDRObjectLoaderListener)
    */
   public JDRObject loadAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      char c = AJR.readChar(in);

      for (Enumeration<JDRObjectLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRObjectLoaderListener listener = e.nextElement();

         if (listener.getAJRid(version) == c)
         {
            JDRObject object = listener.readAJR(in, version);

            if (object instanceof JDRCompleteObject)
            {
               JDRCompleteObject obj = (JDRCompleteObject)object;

               FlowFrame flowframe = null;

               if (AJR.readBoolean(in))
               {
                  flowframe = (FlowFrame)FlowFrame.readAJR(in, version);
               }

               String description = "";

               if (version >= 1.2f)
               {
                  int n = AJR.readInt(in);

                  if (n > 0)
                  {
                     description = AJR.readString(in, n);
                  }
                  else if (n < 0)
                  {
                     throw new InvalidDescriptionLengthException(n);
                  }
               }

               if (object != null)
               {
                  obj.flowframe = flowframe;
                  obj.description = description;
               }
            }

            return object;
         }
      }

      throw new InvalidObjectIDException(c);
   }

   /**
    * Adds a new listener.
    * @param listener the new listener
    * @see #getListeners()
    */
   public void addListener(JDRObjectLoaderListener listener)
   {
      listeners_.add(listener);
   }

   /**
    * Gets all the listeners registered with this loader.
    * @return list of all listeners registered with this loader
    * @see #addListener(JDRObjectLoaderListener)
    */
   public Vector<JDRObjectLoaderListener> getListeners()
   {
      return listeners_;
   }

   private Vector<JDRObjectLoaderListener> listeners_;
}
