// File          : JDRObject.java
// Date          : 1st February 2006
// Last Modified : 18th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.Vector;
import java.util.NoSuchElementException;
import java.util.Arrays;
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.lang.Math;
import java.text.*;

import java.awt.*;
import java.awt.geom.*;
import java.awt.font.TextLayout;
import java.awt.font.FontRenderContext;

import javax.swing.*;

/**
 * Class representing a JDR object.
 * @author Nicola L C Talbot
 */
public abstract class JDRObject implements Serializable,Cloneable
{
   /**
    * Creates a new object.
    */
   public JDRObject()
   {
   }

   /**
    * Sets whether this object is selected.
    * @param flag indicates whether this object is selected
    * @see #isSelected()
    */
   public void setSelected(boolean flag)
   {
      selected = flag;
   }

   /**
    * Determines whether this object is selected.
    * @return true if this object is selected otherwise false
    * @see #setSelected(boolean)
    */
   public boolean isSelected()
   {
      return selected;
   }

   /**
    * Gets this object's bounding box. Returns <code>null</code>
    * if this object has no size.
    * @return this object's bounding box or <code>null</code> if
    * this object has no size
    * @see #getControlBBox()
    */
   public BBox getBBox()
   {
      return null;
   }

   /**
    * Merges the given bounding box with the bounding box for this
    * object.
    * @param box the given bounding box
    */
   public void mergeBBox(BBox box)
   {
      BBox thisBox = getBBox();

      if (thisBox != null) box.merge(thisBox);
   }

   /**
    * Gets this object's bounding box including control points.
    * This is the same as {@link #getBBox()}.
    * @return this object's bounding box including control points
    * or <code>null</code> if this object has no size
    * @see #getBBox()
    */
   public BBox getControlBBox()
   {
      return getBBox();
   }

   /**
    * Transforms this object.
    * @param matrix the affine transformation matrix (stored in 
    * flat format)
    */
   public abstract void transform(double[] matrix);

   /**
    * Shifts this object.
    * @param p the x and y shift
    * @see #translate(double,double)
    */
   public void translate(Point p)
   {
      translate(p.x, p.y);
   }

   /**
    * Shifts this object.
    * @param p the x and y shift
    * @see #translate(double,double)
    */
   public void translate(JDRPoint p)
   {
      translate(p.x, p.y);
   }

   /**
    * Shifts this object.
    * @param p the x and y shift
    * @see #translate(double,double)
    */
   public void translate(Point2D p)
   {
      translate(p.getX(), p.getY());
   }

   /**
    * Shifts this object. Subclasses need to override this method.
    * @param x the x shift
    * @param y the y shift
    */
   public abstract void translate(double x, double y);

   /**
    * Scales this object. Subclasses need to override this method.
    * @param factorX the x scale factor
    * @param factorY the y scale factor
    * @see #scale(double)
    */
   public abstract void scale(double factorX, double factorY);

   /**
    * Scales this object.
    * @param factor the x and y scale factor
    * @see #scale(double,double)
    */
   public void scale(double factor)
   {
      scale(factor, factor);
   }

   /**
    * Scales this object relative to the given point.
    * @param p the scaling origin
    * @param factorX the x scale factor
    * @param factorY the y scale factor
    * @see #scale(double,double)
    */
   public void scale(Point2D p, double factorX, double factorY)
   {
      translate(new Point2D.Double(-p.getX(), -p.getY()));
      scale(factorX, factorY);
      translate(p);
   }

   /**
    * Scales this object relative to the given point.
    * @param p the scaling origin
    * @param factor the x and y scale factor
    * @see #scale(Point2D,double,double)
    * @see #scale(double,double)
    */
   public void scale(Point2D p, double factor)
   {
      scale(p, factor, factor);
   }

   /**
    * Scales this object relative to the given point.
    * @param p the scaling origin
    * @param factor the x and y scale factor
    * @see #scale(JDRPoint,double,double)
    * @see #scale(double,double)
    */
   public void scale(JDRPoint p, double factor)
   {
      scale(p.getPoint2D(), factor);
   }

   /**
    * Scales this object relative to the given point.
    * @param p the scaling origin
    * @param factorX the x scale factor
    * @param factorY the y scale factor
    * @see #scale(double,double)
    */
   public void scale(JDRPoint p, double factorX, double factorY)
   {
      scale(p.getPoint2D(), factorX, factorY);
   }

   /**
    * Scales this object relative to the given point.
    * @param p the scaling origin
    * @param factor the x and y scale factor
    * @see #scale(double,double)
    */
   public void scale(Point p, double factor)
   {
      scale(new Point2D.Double(p.x, p.y),factor);
   }

   /**
    * Scales this object horizontally.
    * @param factor the x scale factor
    * @see #scale(double,double)
    */
   public void scaleX(double factor)
   {
      scale(factor,1.0);
   }

   /**
    * Scales this object horizontally relative to the given point.
    * @param p the scaling origin
    * @param factor the x scale factor
    * @see #scaleX(double)
    */
   public void scaleX(Point2D p, double factor)
   {
      scale(p, factor, 1.0);
   }

   /**
    * Scales this object horizontally relative to the given point.
    * @param p the scaling origin
    * @param factor the x scale factor
    * @see #scaleX(double)
    */
   public void scaleX(JDRPoint p, double factor)
   {
      scaleX(p.getPoint2D(),factor);
   }

   /**
    * Scales this object horizontally relative to the given point.
    * @param p the scaling origin
    * @param factor the x scale factor
    * @see #scaleX(double)
    */
   public void scaleX(Point p, double factor)
   {
      scaleX(new Point2D.Double(p.x,p.y),factor);
   }

   /**
    * Scales this object vertically.
    * @param factor the y scale factor
    * @see #scale(double,double)
    */
   public void scaleY(double factor)
   {
      scale(1.0,factor);
   }

   /**
    * Scales this object vertically relative to the given point.
    * @param p the scaling origin
    * @param factor the y scale factor
    * @see #scaleY(double)
    */
   public void scaleY(JDRPoint p, double factor)
   {
      scaleY(p.getPoint2D(),factor);
   }

   /**
    * Scales this object vertically relative to the given point.
    * @param p the scaling origin
    * @param factor the y scale factor
    * @see #scaleY(double)
    */
   public void scaleY(Point p, double factor)
   {
      scaleY(new Point2D.Double(p.x,p.y),factor);
   }

   /**
    * Scales this object vertically relative to the given point.
    * @param p the scaling origin
    * @param factor the y scale factor
    * @see #scaleY(double)
    */
   public void scaleY(Point2D p, double factor)
   {
      scale(p, 1.0, factor);
   }

   /**
    * Shears this object. This method needs to be overridden by
    * subclasses.
    * @param factorX the x shear factor
    * @param factorY the y shear factor
    */
   public abstract void shear(double factorX, double factorY);

   /**
    * Shears this object.
    * @param factor the x and y shear factor
    * @see #shear(double,double)
    */
   public void shear(double factor)
   {
      shear(factor, factor);
   }

   /**
    * Shears this object relative to the given point.
    * @param p the shearing origin
    * @param factor the x and y shear factor
    * @see #shear(Point2D,double,double)
    */
   public void shear(Point2D p, double factor)
   {
      translate(new Point2D.Double(-p.getX(), -p.getY()));
      shear(factor);
      translate(p);
   }

   /**
    * Shears this object relative to the given point.
    * @param p the shearing origin
    * @param factorX the x shear factor
    * @param factorY the y shear factor
    * @see #shear(double,double)
    */
   public void shear(Point2D p, double factorX, double factorY)
   {
      translate(new Point2D.Double(-p.getX(), -p.getY()));
      shear(factorX, factorY);
      translate(p);
   }

   /**
    * Shears this object relative to the given point.
    * @param p the shearing origin
    * @param factorX the x shear factor
    * @param factorY the y shear factor
    * @see #shear(double,double)
    */
   public void shear(JDRPoint p, double factorX, double factorY)
   {
      translate(-p.x, -p.y);
      shear(factorX, factorY);
      translate(p.x, p.y);
   }

   /**
    * Shears this object relative to the given point.
    * @param p the shearing origin
    * @param factor the x and y shear factor
    * @see #shear(JDRPoint,double,double)
    */
   public void shear(JDRPoint p, double factor)
   {
      shear(p.getPoint2D(),factor);
   }

   /**
    * Shears this object relative to the given point.
    * @param p the shearing origin
    * @param factor the x and y shear factor
    * @see #shear(JDRPoint,double,double)
    */
   public void shear(Point p, double factor)
   {
      shear(new Point2D.Double(p.x, p.y),factor);
   }

   /**
    * Shears this object horizontally.
    * @param factor the x shear factor
    * @see #shear(double,double)
    * @see #shearY(double)
    */
   public void shearX(double factor)
   {
      shear(factor, 0.0);
   }

   /**
    * Shears this object horizontally relative to the given point.
    * @param p the shearing origin
    * @param factor the x shear factor
    * @see #shearX(double)
    * @see #shear(Point2D,double,double)
    */
   public void shearX(Point2D p, double factor)
   {
      translate(new Point2D.Double(-p.getX(), -p.getY()));
      shearX(factor);
      translate(p);
   }

   /**
    * Shears this object horizontally relative to the given point.
    * @param p the shearing origin
    * @param factor the x shear factor
    * @see #shearX(double)
    * @see #shear(Point2D,double,double)
    */
   public void shearX(JDRPoint p, double factor)
   {
      shearX(p.getPoint2D(),factor);
   }

   /**
    * Shears this object horizontally relative to the given point.
    * @param p the shearing origin
    * @param factor the x shear factor
    * @see #shearX(double)
    * @see #shear(Point2D,double,double)
    */
   public void shearX(Point p, double factor)
   {
      shearX(new Point2D.Double(p.x,p.y),factor);
   }

   /**
    * Shears this object vertically.
    * @param factor the y shear factor
    * @see #shear(double,double)
    * @see #shearX(double)
    */
   public void shearY(double factor)
   {
      shear(0.0, factor);
   }

   /**
    * Shears this object vertically relative to the given point.
    * @param p the shearing origin
    * @param factor the y shear factor
    * @see #shearY(double)
    * @see #shear(double,double)
    * @see #shear(JDRPoint,double,double)
    */
   public void shearY(JDRPoint p, double factor)
   {
      shearY(p.getPoint2D(),factor);
   }

   /**
    * Shears this object vertically relative to the given point.
    * @param p the shearing origin
    * @param factor the y shear factor
    * @see #shearY(double)
    * @see #shear(double,double)
    * @see #shear(Point2D,double,double)
    */
   public void shearY(Point p, double factor)
   {
      shearY(new Point2D.Double(p.x,p.y),factor);
   }

   /**
    * Shears this object vertically relative to the given point.
    * @param p the shearing origin
    * @param factor the y shear factor
    * @see #shearY(double)
    * @see #shear(double,double)
    * @see #shear(Point2D,double,double)
    */
   public void shearY(Point2D p, double factor)
   {
      translate(new Point2D.Double(-p.getX(), -p.getY()));
      shearY(factor);
      translate(p);
   }

   /**
    * Rotates this object. Subclasses need to
    * override this method.
    * @param angle the angle of rotation
    */
   public abstract void rotate(double angle);

   /**
    * Rotates this object about the given point.
    * @param p the point of rotation
    * @param angle the angle of rotation
    * @see #rotate(double)
    */
   public void rotate(Point2D p, double angle)
   {
      translate(new Point2D.Double(-p.getX(), -p.getY()));
      rotate(angle);
      translate(p);
   }

   /**
    * Rotates this object about the given point.
    * @param p the point of rotation
    * @param angle the angle of rotation
    * @see #rotate(double)
    */
   public void rotate(JDRPoint p, double angle)
   {
      rotate(p.getPoint2D(),angle);
   }

   /**
    * Rotates this object about the given point.
    * @param p the point of rotation
    * @param angle the angle of rotation
    * @see #rotate(double)
    */
   public void rotate(Point p, double angle)
   {
      rotate(new Point2D.Double(p.x,p.y),angle);
   }

   /**
    * Draws the control points associated with this object. 
    * This method needs to be overridden by
    * subclasses that have control points (i.e. paths).
    * @param g the graphics device
    */
   public void drawControls(Graphics g, boolean endPoint)
   {
   }

   /**
    * Draws this object. This just calls 
    * {@link #draw(Graphics,boolean)} and ignores the component.
    * @param g the graphics device
    * @param draft determines whether to use draft mode
    * drawn
    * @param comp the associated graphics component
    */
   public void draw(Graphics g, boolean draft, JComponent comp)
   {
      draw(g, draft);
   }

   /**
    * Draws this object in non-draft mode.
    * @param g the graphics device
    */
   public abstract void draw(Graphics g);

   /**
    * Draws this object in draft mode. It assumes that the graphics state
    * has already been set to use the relevant draft colour. This
    * just calls {@link #draw(Graphics)}
    * @param g the graphics device
    */
   public void drawDraft(Graphics g)
   {
      draw(g);
   }

   /**
    * Draws this object. Sets the draft paint if draft mode. Calls
    * {@link #drawDraft(Graphics)} if draft mode on, otherwise calls
    * {@link #draw(Graphics)}
    * @param g the graphics device
    * @param draft determines whether to use draft mode
    */
   public void draw(Graphics g, boolean draft)
   {
      Graphics2D g2 = (Graphics2D)g;

      if (draft)
      {
         g2.setPaint(draftColor);
         drawDraft(g);
      }
      else
      {
         draw(g);
      }
   }

   /**
    * Saves this object in EPS format.
    * @param out the output stream
    * @throws IOException if I/O error occurs
    */
   public abstract void saveEPS(PrintWriter out)
      throws IOException;

   /**
    * Returns the EPS level supported by this object. This
    * defaults to Level 1.
    * @return the PostScript level that supports this object
    */
   public int psLevel()
   {
      return 1;
   }

   /**
    * Saves this object in SVG format.
    * @param out the output stream
    * @throws IOException if I/O error occurs
    */
   public abstract void saveSVG(PrintWriter out)
      throws IOException;

   /**
    * Gets PGF commands for this object.
    * @param af transformation to apply
    * @return string containing relevant PGF commands
    */
   public abstract String pgf(AffineTransform af);

   /**
    * Gets a copy of this object.
    * @return a copy of this object
    */
   public abstract Object clone();

   /**
    * Makes this object identical to the other object.
    * @param object the other object
    */
   public void makeEqual(JDRObject object)
   {
      selected = object.isSelected();
   }

   /**
    * Determines if this object is the same as another object.
    * @param o the other object
    * @return true if this object is equal to the other object
    */
   public boolean equals(Object o)
   {
      if (o == null)
      {
         return false;
      }

      if (this == o) return true;

      if (!(o instanceof JDRObject))
      {
         return false;
      }

      JDRObject jdrobj = (JDRObject)o;

      if (selected != jdrobj.selected) return false;

      return true;
   }

  /**
    * Gets the control point that contains the given point.
    * This method returns <code>null</code>. Subclasses that
    * have control points need to override this method.
    * @param p the given point
    * @param endPoint if true include the end point in the search
    * otherwise don't check the end point.
    * @return the control point that contains the given point or
    * null if none of the control points contain that point
    */
   public JDRPoint getControl(Point p, boolean endPoint)
   {
      return null;
   }

   /**
    * Gets the object loader listener associated with this object.
    * May be null if there is no listener associated with this 
    * object.
    * @return the listener used to load and save this object
    * or <code>null</code> if no available listener
    */
   public JDRObjectLoaderListener getListener()
   {
      return null;
   }

   /**
    * The colour to draw objects in draft mode.
    */
   public static Paint draftColor = Color.lightGray;

   /**
    * Indicates whether this object is selected.
    */
   protected boolean selected;
}
