// File          : JDRLineLoaderListener.java
// Date          : 29th February 2008
// Last Modified : 18th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for line segments.
 * @author Nicola L C Talbot
 */

public class JDRLineLoaderListener extends JDRSegmentLoaderListener
{
   /**
    * Gets the character identifying the line segment type
    * in JDR format.
    * @param version the JDR version number
    * @return JDR segment id
    */
   public char getJDRid(float version)
   {
      return 'L';
   }

   /**
    * Reads line segment specified in JDR format. (Doesn't include the
    * segment ID returned by {@link #getJDRid(float)} which is dealt
    * with by the segment loader.)
    * @param x x co-ordinate of current point
    * @param y y co-ordinate of current point
    * @param din the input stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the segment format
    * @return the line segment read from the input stream
    */
   public JDRObject readJDR(double x, double y,
       DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      double x0, x1, y0, y1;

      if (version < 1.3f)
      {
         x0 = din.readDouble();
         y0 = din.readDouble();
      }
      else
      {
         x0 = x;
         y0 = y;
      }

      x1 = din.readDouble();
      y1 = din.readDouble();

      return new JDRLine(x0, y0, x1, y1);
   }

   /**
    * Gets the character identifying the line segment type
    * in AJR format.
    * @param version the AJR version number
    * @return AJR segment id
    */
   public char getAJRid(float version)
   {
      return 'L';
   }

   /**
    * Reads line segment specified in AJR format. (Doesn't include the
    * segment ID returned by {@link #getAJRid(float)} which is dealt
    * with by the segment loader.)
    * @param x x co-ordinate of current point
    * @param y y co-ordinate of current point
    * @param in the input stream
    * @param version the AJR version number
    * @return the line segment read from the input stream
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the segment format
    * @throws java.nio.BufferOverflowException if the {@link uk.ac.uea.cmp.nlct.jdr.io.AJR}
    * buffer overflows
    * @throws EOFException if the end of file occurs unexpectedly
    */
   public JDRObject readAJR(double x, double y,
      BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      double x0, x1, y0, y1;

      if (version < 1.3f)
      {
         x0 = AJR.readDouble(in);
         y0 = AJR.readDouble(in);
      }
      else
      {
         x0 = x;
         y0 = y;
      }

      x1 = AJR.readDouble(in);
      y1 = AJR.readDouble(in);

      return new JDRLine(x0, y0, x1, y1);
   }
}
