// File          : JDRLine.java
// Date          : 1st February 2006
// Last Modified : 25th July 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;
/**
 * Class representing a line segment.
 * @author Nicola L C Talbot
 * @see JDRPath
 */

public class JDRLine extends JDRSegment
{
   /**
    * Creates a line segment between the given start and end points.
    * @param p0x the x co-ordinate of the start
    * @param p0y the y co-ordinate of the start
    * @param p1x the x co-ordinate of the end
    * @param p1y the y co-ordinate of the end
    */
   public JDRLine(double p0x, double p0y, double p1x, double p1y)
   {
      super(p0x, p0y, p1x, p1y);
   }

   /**
    * Creates a line segment between the given start and end points.
    * @param p0 the start location
    * @param p1 the end location
    */
   public JDRLine(Point p0, Point p1)
   {
      super(p0, p1);
   }

   /**
    * Creates a line segment between the given start and end points.
    * @param p0 the start location
    * @param p1 the end location
    */
   public JDRLine(Point2D p0, Point2D p1)
   {
      super(p0, p1);
   }

   /**
    * Creates a line segment between the given start and end points.
    * @param p0 the start location
    * @param p1 the end location
    */
   public JDRLine(JDRPoint p0, JDRPoint p1)
   {
      super(p0, p1);
   }

   /**
    * Creates a line segment whose start and end points are at the
    * origin.
    */
   public JDRLine()
   {
      super();
   }

   public JDRLine reverse()
   {
      JDRLine line = (JDRLine)clone();

      line.start.x = end.x;
      line.start.y = end.y;
      line.end.x   = start.x;
      line.end.y   = start.y;

      return line;
   }

   public JDRPathSegment split()
   {
      JDRPoint midPt = new JDRPoint(getP(0.5));

      JDRLine newSegment = new JDRLine(midPt, end);

      end = midPt;

      return newSegment;
   }

   public Object clone()
   {
      JDRLine seg = new JDRLine((JDRPoint)start.clone(),
                          (JDRPoint)end.clone());
      seg.makeEqual(this);
      return seg;
   }

   public boolean equals(Object obj)
   {
      if (!super.equals(obj)) return false;

      // super.equals(obj) has already checked start and end

      return (obj instanceof JDRLine);
   }

   public void draw(Graphics g)
   {
      g.drawLine((int)start.x, (int)start.y, (int)end.x, (int)end.y);
   }

   public void draw(Graphics g, double scale)
   {
      g.drawLine((int)(scale*start.x), (int)(scale*start.y),
                 (int)(scale*end.x), (int)(scale*end.y));
   }

   public void drawDraft(Graphics g, double scale, boolean drawEnd)
   {
      Graphics2D g2 = (Graphics2D)g;

      g2.setPaint(JDRObject.draftColor);

      g2.drawLine((int)(start.x*scale), (int)(start.y*scale),
                  (int)(end.x*scale), (int)(end.y*scale));

      if (isSelected())
      {
         Stroke oldStroke = g2.getStroke();
         g2.setPaint(start.getSelectedPaint());
         g2.setStroke(guideStroke);

         g2.drawLine((int)(start.x*scale), (int)(start.y*scale),
                     (int)(end.x*scale), (int)(end.y*scale));

         g2.setStroke(oldStroke);
      }

      start.draw(g, scale);

      if (drawEnd)
      {
         end.draw(g, scale);
      }
   }

   public void saveSVG(PrintWriter out) throws IOException
   {
      out.print("L ");
      end.saveSVG(out);
   }

   public void saveEPS(PrintWriter out) throws IOException
   {
      end.saveEPS(out);
      out.println("lineto");
   }

   public String pgf(AffineTransform af)
   {
      return "\\pgfpathlineto{"+end.pgf(af)+"}";
   }

   public void appendToGeneralPath(GeneralPath path)
   {
      path.lineTo((float)end.x, (float)end.y);
   }

   public void appendReflectionToGeneralPath(GeneralPath path, JDRLine line)
   {
      Point2D p = start.getReflection(line);

      path.lineTo((float)p.getX(), (float)p.getY());
   }

   public JDRPathSegment getReflection(JDRLine line)
   {
      Point2D p1 = start.getReflection(line);
      Point2D p2 = end.getReflection(line);

      return new JDRLine(p1, p2);
   }

   /**
    * Gets the transformation representing a reflection in this
    * line.
    * @param af if non-null, the result is stored in here
    * @return the transformation representing a reflection in this
    * line.
    */
   public AffineTransform getReflectionTransform(AffineTransform af)
   {
      if (af == null)
      {
         af = new AffineTransform();
      }

      // Trap near vertical and near horizontal lines to prevent
      // arithmetic errors

      double dx = start.x - end.x;
      double dy = start.y - end.y;

      if (Math.abs(dx) < 1)
      {
         // Line is approximately vertical

         af.setTransform(-1.0, 0.0, 0.0, 1.0, 2*end.x, 0.0);
      }
      else if (Math.abs(dy) < 1)
      {
         // Line is approximately horizontal

         af.setTransform(1.0, 0.0, 0.0, -1.0, 0.0, 2*end.y);
      }
      else
      {
         double m = dy/dx;
         double minv = dx/dy;

         double a = 2.0/(m+minv);

         af.setTransform(a*minv-1.0, a, a, a*m-1.0,
           a*(m*start.x-start.y), a*(minv*start.y-start.x));
      }

      return af;
   }

   public JDRObjectLoaderListener getListener()
   {
      return listener;
   }

   public String toString()
   {
      return "JDRLine:("+start.x+","+start.y+")("+end.x+","+end.y+"),startMarker="+startMarker+",endMarker="+endMarker;
   }

   public String info()
   {
      return "line["+start.info()+","+end.info()+"]";
   }

   public boolean isGap() {return false;}

   private static JDRLineLoaderListener listener
      = new JDRLineLoaderListener();
}
