// File          : JDRGridLoader.java
// Date          : 17th August 2010
// Last Modified : 17th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class dealing with saving and loading {@link JDRGrid} objects 
 * to/from JDR and AJR files.
 * @author Nicola L C Talbot
 */
public class JDRGridLoader
{
   /**
    * Creates new loader.
    */
   public JDRGridLoader()
   {
      listeners_ = new Vector<JDRGridLoaderListener>();
   }

   /**
    * Saves the given grid in JDR format. This first writes the
    * ID specified by 
    * {@link JDRGridLoaderListener#getJDRid(float)} and then 
    * writes the grid specifications using 
    * {@link JDRGridLoaderListener#writeJDR(JDRGrid,DataOutputStream,float)}.
    * @param grid the grid that needs to be saved
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if an I/O error occurs
    * @see #loadJDR(DataInputStream,float)
    */
   public void saveJDR(JDRGrid grid,
                    DataOutputStream dout,
                    float version)
      throws IOException,InvalidFormatException
   {
      JDRGridLoaderListener listener = grid.getListener();

      if (listener == null)
      {
         throw new InvalidGridException("Grid "+grid
            + " doesn't have a valid listener");
      }

      dout.writeByte(listener.getJDRid(version));
      listener.writeJDR(grid,dout,version);
   }

   /**
    * Loads a grid specified in JDR format. This first reads a
    * byte and checks through the list of listeners
    * to determine which type is identified by the  
    * number (using {@link JDRGridLoaderListener#getJDRid(float)}).
    * @param din the input stream
    * @param version the JDR version number
    * @return the specified grid
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @see #saveJDR(JDRGrid,DataOutputStream,float)
    * @see #addListener(JDRGridLoaderListener)
    */
   public JDRGrid loadJDR(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      byte id = din.readByte();

      for (Enumeration<JDRGridLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRGridLoaderListener listener = e.nextElement();

         if (listener.getJDRid(version) == id)
         {
            return listener.readJDR(din, version);
         }
      }

      throw new InvalidGridException(id);
   }

   /**
    * Saves the given grid in AJR format. This first writes the
    * ID byte specified by 
    * {@link JDRGridLoaderListener#getAJRid(float)} and then 
    * writes the specifications using 
    * {@link JDRGridLoaderListener#writeAJR(JDRGrid,PrintWriter,float)}.
    * @param grid the grid that needs to be saved
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if an I/O error occurs
    * @see #loadAJR(BufferedReader,float)
    */
   public void saveAJR(JDRGrid grid,
                    PrintWriter out,
                    float version)
      throws IOException,InvalidFormatException
   {
      JDRGridLoaderListener listener = grid.getListener();

      if (listener == null)
      {
         throw new InvalidGridException("Grid "+grid
            + " doesn't have a valid listener");
      }

      AJR.writeInt(out, listener.getAJRid(version));
      listener.writeAJR(grid,out,version);
   }

   /**
    * Loads a grid specified in AJR format. This first reads a
    * number and checks through the list of listeners
    * to determine which grid type is identified by the  
    * number (using {@link JDRGridLoaderListener#getAJRid(float)}).
    * @param in the input stream
    * @param version the AJR version number
    * @return the specified grid
    * @throws IOException if an I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @throws EOFException if the file ends unexpectedly
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * (specified by {@link AJR#buffLength}) is exceeded
    * @see #saveAJR(JDRGrid,PrintWriter,float)
    * @see #addListener(JDRGridLoaderListener)
    */
   public JDRGrid loadAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      byte id = (byte)AJR.readInt(in);

      for (Enumeration<JDRGridLoaderListener> 
             e = listeners_.elements();e.hasMoreElements();)
      {
         JDRGridLoaderListener listener = e.nextElement();

         if (listener.getAJRid(version) == id)
         {
            return listener.readAJR(in, version);
         }
      }

      throw new InvalidGridException(id);
   }

   /**
    * Adds a new listener.
    * @param listener the new listener
    * @see #getListeners()
    */
   public void addListener(JDRGridLoaderListener listener)
   {
      listeners_.add(listener);
   }

   /**
    * Gets all the listeners registered with this loader.
    * @return list of all listeners registered with this loader
    * @see #addListener(JDRGridLoaderListener)
    */
   public Vector<JDRGridLoaderListener> getListeners()
   {
      return listeners_;
   }

   private Vector<JDRGridLoaderListener> listeners_;
}
