//File          : JDRGrid.java
//Description   : Represents a grid.
//Author        : Nicola L.C. Talbot
//Date          : 17th August 2010
//Last Modified : 17th August 2010
//              http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr;

import java.awt.*;
import java.awt.geom.*;

/**
 * Class representing a grid.
 * @author Nicola L C Talbot
 */
public abstract class JDRGrid implements Cloneable
{
   /**
    * Draws a minor tic mark. Note that this doesn't set the colour.
    * @param g the graphics device
    * @param x the x co-ordinate of the centre of the tic mark
    * @param y the y co-ordinate of the centre of the tic mark
    * @see #drawMajorTic(Graphics,int,int)
    */
   public void drawMinorTic(Graphics g, int x, int y)
   {
      g.drawLine(x-2, y, x+2, y);
      g.drawLine(x, y-2, x, y+2);
   }

   /**
    * Draws a major tic mark. Note that this doesn't set the colour.
    * @param g the graphics device
    * @param x the x co-ordinate of the centre of the tic mark
    * @param y the y co-ordinate of the centre of the tic mark
    * @see #drawMinorTic(Graphics,int,int)
    */
   public void drawMajorTic(Graphics g, int x, int y)
   {
      g.drawLine(x-3, y, x+3, y);
      g.drawLine(x, y-3, x, y+3);
   }

   /**
    * Draws the grid on the given graphics device for the given
    * paper size scaled by the given scale factor. Note that the
    * scale factor should not scale the tic marks, just the
    * co-ordinates.
    * @param g the graphics device
    * @param paper the paper size
    * @param scale the factor to scale the co-ordinates and paper
    * size
    */
   public abstract void drawGrid(Graphics2D g, JDRPaper paper,
      double scale);

   /**
    * Converts a co-ordinate from this space to left-handed
    * Cartesian system.
    * @param paper the paper
    * @param original the original point in this space
    * @param target the target point (bp) in left-handed Cartesian space
    */
   public abstract void getCartesian(JDRPaper paper, Point2D original,
      Point2D target);

   /**
    * Converts a co-ordinate from left-handed Cartesian space to
    * this space.
    * @param paper the paper
    * @param cartesianPoint the original point (bp)
    * @param target the target point in this space.
    */
   public abstract void getFromCartesian(JDRPaper paper, Point2D cartesianPoint,
      Point2D target);

   /**
    * Converts a co-ordinate from left-handed Cartesian space to
    * this space.
    * @param paper the paper
    * @param x the x co-ordinate of the original point (bp)
    * @param y the y co-ordinate of the original point (bp)
    */
   public abstract Point2D getFromCartesian(JDRPaper paper, double x,
      double y);

   /**
    * Gets the distance between minor intervals or 0 if no sub
    * divisions.
    * @return the x and y distance (in bp) between minor intervals or 0 if no sub
    * divisions
    */
   public abstract Point2D getMinorTicDistance();

   /**
    * Gets the distance between major intervals.
    * @return the x and y distance (in bp) between major intervals
    */
   public abstract Point2D getMajorTicDistance();

   /**
    * Gets the closest tick mark to the given point.
    * @param x the x co-ordinate of the point
    * @param y the y co-ordinate of the point
    * @return the closest tick mark to the given point in
    * left-handed Cartesian space.
    */
   public abstract Point2D getClosestTic(JDRPaper paper, double x, 
     double y);

   /**
    * Gets the unit label associated with this grid.
    */
   public abstract String getUnitLabel();

   /**
    * Gets the principle unit associated with this grid.
    */
   public abstract JDRUnit getMainUnit();

   /**
    * Returns a formatted string representing the given point in
    * this space.
    */
   public abstract String formatLocation(Point2D point);

   public abstract Object clone();

   /**
    * Gets the listener for this grid.
    */
   public abstract JDRGridLoaderListener getListener();

   public static Color majorGridColor = Color.gray;
   public static Color minorGridColor = new Color(222,222,255);
}
