// File          : JDRGrayListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for grey paint.
 * @author Nicola L C Talbot
 */

public class JDRGrayListener implements JDRPaintLoaderListener
{
   public char getJDRid(float version)
   {
      return 'Y';
   }

   public void writeJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
   throws IOException
   {
      JDRGray c = (JDRGray)paint;

      dout.writeFloat((float)c.getGray());
      dout.writeFloat((float)c.getAlpha());
   }

   public JDRPaint readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      double g = din.readFloat();

      if (g < 0.0 || g > 1.0)
      {
         throw new InvalidGrayValueException(g);
      }

      double a = din.readFloat();

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      return new JDRGray(g,a);
   }

   public char getAJRid(float version)
   {
      return 'Y';
   }

   public void writeAJR(JDRPaint paint,
                        PrintWriter out,
                        float version)
   throws IOException
   {
      JDRGray c = (JDRGray)paint;

      AJR.writeFloat(out, (float)c.getGray());
      AJR.writeFloat(out, (float)c.getAlpha());
   }

   public JDRPaint readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      double grey = AJR.readFloat(in);

      if (grey < 0.0 || grey > 1.0)
      {
         throw new InvalidGrayValueException(grey, AJR.getLineNum());
      }

      double a = AJR.readFloat(in);

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a, AJR.getLineNum());
      }

      return new JDRGray(grey,a);
   }

   public int getConfigId()
   {
      return 6;
   }

   /**
    * @param paint the paint to save (must be {@link JDRGray}).
    */
   public String getConfigString(JDRPaint paint)
   {
      JDRGray c = (JDRGray)paint;

      return "" + c.getGray() 
           +"," + c.getAlpha();
   }

   public JDRPaint parseConfig(String specs)
   throws InvalidFormatException
   {
      String split[] = specs.split(",", 3);

      if (split.length < 2)
      {
         throw new InvalidPaintException("Invalid grey config format");
      }     

      double grey  = 0;
      double alpha = 0;

      try
      {
         grey = Double.parseDouble(split[0]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid grey value "+split[0]);
      }

      try
      {
         alpha = Double.parseDouble(split[1]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid alpha value "+split[1]);
      }

      if (split.length == 3)
      {
         remainder = split[2];
      }
      else
      {
         remainder = "";
      }

      return new JDRGray(grey, alpha);
   }

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder()
   {
      return remainder;
   }

   private String remainder="";
}
