// File          : JDRGray.java
// Date          : 20th March 2007
// Last Modified : 20th March 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing grey.
 * @author Nicola L C Talbot
 */

public class JDRGray implements JDRPaint,Serializable
{
   /**
    * Creates a new grey.
    * The transparency is set to 1 (opaque).
    * The grey value must be in the range 0-1.
    * @param g grey scale
    * @throws InvalidGrayValueException if grey value is less than 0
    * or greater than 1
    */
   public JDRGray(double g) throws InvalidGrayValueException
   {
      if (g < 0.0f && g > 1.0f)
      {
         throw new InvalidGrayValueException(g);
      }

      gray = g;
      alpha = 1.0;
   }

   /**
    * Creates a new grey.
    * The grey and alpha values must be in the range 0-1.
    * @param g grey scale
    * @param a alpha (transparency)
    * @throws InvalidGrayValueException if grey value is less than 0
    * or greater than 1
    * @throws InvalidAlphaValueException if alpha value is less than 0
    * or greater than 1
    */
   public JDRGray(double g, double a)
   throws InvalidGrayValueException,
          InvalidAlphaValueException
   {
      if (g < 0.0f && g > 1.0f)
      {
         throw new InvalidGrayValueException(g);
      }

      if (a < 0.0f && a > 1.0f)
      {
         throw new InvalidAlphaValueException(a);
      }

      gray = g;
      alpha = a;
   }

   /**
    * Creates a new grey (black).
    */
   public JDRGray()
   {
      gray  = 0;
      alpha = 1;
   }

   public JDRGray getJDRGray()
   {
      return this;
   }

   public JDRColorCMYK getJDRColorCMYK()
   {
      double black   = (1-gray);
      double cyan    = 0;
      double magenta = 0;
      double yellow  = 0;

      try
      {
         return new JDRColorCMYK(cyan,magenta,yellow,black,alpha);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRColorCMYK();
   }

   public JDRColor getJDRColor()
   {
      JDRColor c=null;

      try
      {
         c = new JDRColor(gray, gray, gray, alpha);
      }
      catch (InvalidFormatException ignore)
      {
      }

      return c;
   }

   public JDRColorHSB getJDRColorHSB()
   {
      try
      {
         return new JDRColorHSB(0, 0, gray);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRColorHSB();
   }

   public Paint getPaint(BBox box)
   {
      return getColor();
   }

   public Color getColor()
   {
      float val = (float)gray;
      return new Color(val,val,val,(float)alpha);
   }

   public String toString()
   {
      return new String("JDRGray@"+"Gray:"+gray+"Alpha:"+alpha);
   }

   public Object clone()
   {
      try
      {
         return new JDRGray(gray,alpha);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRGray();
   }

   public String pgf(BBox box)
   {
      return "\\color[gray]{"+PGF.format(gray)+"}";
   }

   public String pgffillcolor(BBox box)
   {
      if (alpha == 1.0)
      {
         return  "\\color[gray]{"+PGF.format(gray)+"}";
      }
      else
      {
         return  "\\pgfsetfillopacity{"+PGF.format(alpha)+"}"+
                 "\\color[gray]{"+PGF.format(gray)+"}";
      }
   }

   public String pgfstrokecolor(BBox box)
   {
      if (alpha == 1.0)
      {
         return  "\\color[gray]{"+gray+"}";
      }
      else
      {
         return  "\\pgfsetstrokeopacity{"+alpha+"}"+
                 "\\color[gray]{"+gray+"}";
      }
   }

   public void saveEPS(PrintWriter out, BBox box)
      throws IOException
   {
      out.println(""+gray+" setgray");
   }

   public int psLevel()
   {
      return 1;
   }

   public double getAlpha()
   {
      return alpha;
   }

   /**
    * Gets the gray scale parameter for this paint. This will be a value in the range
    * 0 to 1.
    * @return the gray scale
    */
   public double getGray()
   {
      return gray;
   }

   /**
    * Sets the alpha component for this paint. This value must be
    * in the range 0 to 1.
    * @param a the alpha component
    * @throws InvalidAlphaValueException if a &lt; 0 or a &gt; 1
    */
   public void setAlpha(double a)
      throws InvalidAlphaValueException
   {
      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      alpha = a;
   }

   /**
    * Sets the gray scale for this paint. This value must be
    * in the range 0 to 1.
    * @param g the gray scale
    * @throws InvalidGrayValueException if g &lt; 0 or g &gt; 1
    */
   public void setGray(double g)
      throws InvalidGrayValueException
   {
      if (g < 0.0 || g > 1.0)
      {
         throw new InvalidGrayValueException(g);
      }

      gray = g;
   }


   public String svg()
   {
      return "gray("+(100*gray)+"%)";
   }

   public String svgFill()
   {
      return "fill=\""+svg()+"\" fill-opacity=\""
      +getAlpha()+"\"";
   }

   public String svgLine()
   {
      return "stroke=\""+svg()+"\" stroke-opacity=\""
      +getAlpha()+"\"";
   }

   public String getID()
   {
      return Integer.toHexString((int)(gray*255))
           + "." + Integer.toHexString((int)(gray*255));
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;

      if (obj == null)
      {
         return false;
      }

      if (!(obj instanceof JDRGray))
      {
         return false;
      }

      JDRGray c = (JDRGray)obj;

      return (getGray() == c.getGray()
           && getAlpha() == c.getAlpha());
   }

   public JDRPaintLoaderListener getListener()
   {
      return listener;
   }

   public void fade(double value)
   {
      if (value < 0)
      {
         throw new IllegalArgumentException
            ("illegal fade factor "+value+". (Negative factors not permitted)");
      }

      alpha *= value;

      if (alpha > 1.0)
      {
         alpha = 1.0;
      }
   }

   private double gray, alpha;
   // gray ranges from 0 (black) to 1 (white)

   private static JDRGrayListener listener = new JDRGrayListener();
}
