// File          : JDRGradientListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for gradient paint.
 * @author Nicola L C Talbot
 */

public class JDRGradientListener implements JDRPaintLoaderListener
{
   public char getJDRid(float version)
   {
      return 'G';
   }

   public void writeJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
   throws IOException
   {
      JDRGradient c = (JDRGradient)paint;
      JDRPaint startColor = c.getStartColor();
      JDRPaint endColor = c.getEndColor();

      JDRPaintLoader loader = JDR.getPaintLoader();
      loader.saveJDR(startColor,dout,version);
      loader.saveJDR(endColor,dout,version);

      dout.writeInt(c.getDirection());
   }

   public JDRPaint readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      JDRPaintLoader loader = JDR.getPaintLoader();
      JDRPaint startPaint = loader.loadJDR(din, version);

      if (startPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      JDRPaint endPaint = loader.loadJDR(din, version);

      if (endPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      int direction = din.readInt();

      return new JDRGradient(direction, startPaint, endPaint);
   }

   public char getAJRid(float version)
   {
      return 'G';
   }

   public void writeAJR(JDRPaint paint,
                        PrintWriter out,
                        float version)
   throws IOException
   {
      JDRGradient c = (JDRGradient)paint;
      JDRPaint startColor = c.getStartColor();
      JDRPaint endColor = c.getEndColor();

      JDRPaintLoader loader = AJR.getPaintLoader();
      loader.saveAJR(startColor,out,version);
      loader.saveAJR(endColor,out,version);

      AJR.writeInt(out,c.getDirection());
   }

   public JDRPaint readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      JDRPaintLoader loader = AJR.getPaintLoader();
      JDRPaint startPaint = loader.loadAJR(in, version);

      if (startPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      JDRPaint endPaint = loader.loadAJR(in, version);

      if (endPaint instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      int direction = AJR.readInt(in);

      return new JDRGradient(direction, startPaint, endPaint);
   }

   public int getConfigId()
   {
      return 3;
   }

   /**
    * @param paint the paint to save (must be {@link JDRGradient}).
    */
   public String getConfigString(JDRPaint paint)
   {
      JDRGradient c = (JDRGradient)paint;
      JDRPaint startColor = c.getStartColor();
      JDRPaint endColor = c.getEndColor();

      JDRPaintLoader loader = JDR.getPaintLoader();

      String specs = loader.getConfigString(startColor)
                   + ","
                   + loader.getConfigString(endColor)
                   + ","
                   + c.getDirection();

      return specs;
   }

   public JDRPaint parseConfig(String specs)
   throws InvalidFormatException
   {
      JDRPaintLoader loader = JDR.getPaintLoader();
      JDRPaint startPaint = loader.parseConfig(specs);

      specs = loader.getConfigRemainder();

      JDRPaint endPaint = loader.parseConfig(specs);

      specs = loader.getConfigRemainder();

      String[] split = specs.split(",", 2);

      int direction = 0;

      try
      {
         direction = Integer.parseInt(split[0]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidFormatException
            ("Invalid gradient paint specs: direction missing");
      }

      if (split.length == 1)
      {
         remainder = "";
      }
      else
      {
         remainder = split[1];
      }

      return new JDRGradient(direction, startPaint, endPaint);
   }

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder()
   {
      return remainder;
   }

   private String remainder="";
}
