// File          : JDRGradient.java
// Date          : 1st February 2006
// Last Modified : 18th August 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing linear gradient shading.
 * This shading consists of a start colour, an end colour and
 * a direction which may be one of: North, North-East, East,
 * South-East, South, South-West, West or North-West. The starting
 * and end colours may not be shadings.
 * @author Nicola L C Talbot
 */
public class JDRGradient implements JDRPaint,Serializable,JDRShading
{
   /**
    * Creates a new linear gradient shading with the given start
    * and end colours. The direction is set to {@link #NORTH}.
    * @param sColor the start colour
    * @param eColor the end colour
    * @throws InvalidStartColourException if the start colour 
    * implements {@link JDRShading}
    * @throws InvalidEndColourException if the end colour 
    * implements {@link JDRShading}
    */
   public JDRGradient(JDRPaint sColor, JDRPaint eColor)
      throws InvalidStartColourException,
             InvalidEndColourException
   {
      if (sColor instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      if (eColor instanceof JDRShading)
      {
         throw new InvalidEndColourException();
      }

      startColor = sColor;
      endColor   = eColor;
      direction  = NORTH;
   }

   /**
    * Creates a new linear gradient shading with the given
    * direction, start colour and end colour.
    * @param d the shading direction
    * @param sColor the start colour
    * @param eColor the end colour
    * @throws InvalidStartColourException if the start colour 
    * implements {@link JDRShading}
    * @throws InvalidEndColourException if the end colour 
    * implements {@link JDRShading}
    * @throws InvalidGradientDirectionException if the gradient
    * is not one of: {@link #NORTH}, {@link #NORTH_EAST},
    * {@link #EAST}, {@link #SOUTH_EAST}, {@link #SOUTH},
    * {@link #SOUTH_WEST}, {@link #WEST} or {@link #NORTH_WEST}
    */
   public JDRGradient(int d, JDRPaint sColor, JDRPaint eColor)
      throws InvalidGradientDirectionException,
             InvalidStartColourException,
             InvalidEndColourException
   {
      if (sColor instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      if (eColor instanceof JDRShading)
      {
         throw new InvalidEndColourException();
      }

      if (d < 0 || d > 7)
      {
         throw new InvalidGradientDirectionException(d);
      }

      startColor = sColor;
      endColor   = eColor;
      direction  = d;
   }

   /**
    * Creates a new shading (black). This sets the start and end
    * colour to black and the direction to {@link #NORTH}.
    */
   public JDRGradient()
   {
      startColor = new JDRColor();
      endColor   = new JDRColor();
      direction  = NORTH;
   }

   public Color getColor()
   {
      return startColor.getColor();
   }

   public JDRColor getJDRColor()
   {
      return startColor.getJDRColor();
   }

   public JDRColorCMYK getJDRColorCMYK()
   {
      return startColor.getJDRColorCMYK();
   }

   public JDRColorHSB getJDRColorHSB()
   {
      return startColor.getJDRColorHSB();
   }

   public JDRGray getJDRGray()
   {
      return startColor.getJDRGray();
   }

   public Paint getPaint(BBox box)
   {
      Point startPt, endPt;
      int midX = (int)(box.getMinX() + 0.5*box.getWidth());
      int midY = (int)(box.getMinY() + 0.5*box.getHeight());

      switch (direction)
      {
         case NORTH :
            startPt  = new Point(midX, (int)box.getMaxY());
            endPt    = new Point(midX, (int)box.getMinY());
            break;
         case NORTH_EAST :
            startPt = new Point((int)box.getMinX(),(int)box.getMaxY());
            endPt   = new Point((int)box.getMaxX(),(int)box.getMinY());
            break;
         case EAST :
            startPt = new Point((int)box.getMinX(),midY); 
            endPt   = new Point((int)box.getMaxX(),midY);
            break;
         case SOUTH_EAST :
            startPt = new Point((int)box.getMinX(),(int)box.getMinY());
            endPt   = new Point((int)box.getMaxX(),(int)box.getMaxY());
            break;
         case SOUTH :
            startPt = new Point(midX, (int)box.getMinY());
            endPt   = new Point(midX, (int)box.getMaxY());
            break;
         case SOUTH_WEST :
            startPt = new Point((int)box.getMaxX(),(int)box.getMinY());
            endPt   = new Point((int)box.getMinX(),(int)box.getMaxY());
            break;
         case WEST :
            startPt = new Point((int)box.getMaxX(),midY);
            endPt   = new Point((int)box.getMinX(),midY); 
            break;
         case NORTH_WEST :
            startPt = new Point((int)box.getMaxX(),(int)box.getMaxY());
            endPt   = new Point((int)box.getMinX(),(int)box.getMinY());
            break;
         default :
            startPt = new Point(0,0);
            endPt = new Point(0,0);
      }

      return new GradientPaint(startPt.x, startPt.y, 
                               startColor.getColor(),
                               endPt.x, endPt.y,
                               endColor.getColor());
   }

   public Object clone()
   {
      try
      {
         return new JDRGradient(direction,startColor,endColor);
      }
      catch (InvalidFormatException ignore)
      {
         // this shouldn't happen
      }

      return new JDRGradient();
   }

   public String toString()
   {
      return new String("JDRGradient@D:"+direction+startColor+endColor);
   }

   private String pgfdeclareverticalshading(
      JDRPaint start, JDRPaint end)
   {
      String str = "\\pgfdeclareverticalshading{jdrlinear"+pgfshadeid+"}{"
                 +"100bp}{";

      // pgf shading only uses rgb
      Color startPaint = start.getColor();

      double r = startPaint.getRed()/255.0;
      double g = startPaint.getGreen()/255.0;
      double b = startPaint.getBlue()/255.0;

      String startPaintID = 
         "(" +  PGF.format(r)+","+PGF.format(g)+","+PGF.format(b)+")";

      str += "rgb(0bp)=" + startPaintID+"; ";
      str += "rgb(32.5bp)=" + startPaintID+"; ";

      Color endPaint = end.getColor();

      r = endPaint.getRed()/255.0;
      g = endPaint.getGreen()/255.0;
      b = endPaint.getBlue()/255.0;

      String endPaintID =
         "("+PGF.format(r)+","+PGF.format(g)+","+PGF.format(b)+")";

      str += "rgb(67.5bp)="+endPaintID+"; ";
      str += "rgb(100bp)="+endPaintID;

      str += "}";

      return str;
   }

   private String pgfdeclarehorizontalshading(
      JDRPaint start, JDRPaint end)
   {
      String str = "\\pgfdeclarehorizontalshading{jdrlinear"+pgfshadeid+"}{"
                 + "100bp}{";

      // pgf shading doesn't use cmyk
      Color startPaint = start.getColor();

      double r = startPaint.getRed()/255.0;
      double g = startPaint.getGreen()/255.0;
      double b = startPaint.getBlue()/255.0;

      String startPaintID = 
         "(" + PGF.format(r)+","+PGF.format(g)+","+PGF.format(b)+")";

      str += "rgb(0bp)="+startPaintID+"; ";
      str += "rgb(32.5bp)="+startPaintID+"; ";

      Color endPaint = end.getColor();

      r = endPaint.getRed()/255.0;
      g = endPaint.getGreen()/255.0;
      b = endPaint.getBlue()/255.0;

      String endPaintID = 
         "(" + PGF.format(r)+","+PGF.format(g)+","+PGF.format(b)+")";

      str += "rgb(67.5bp)="+endPaintID+"; ";
      str += "rgb(100bp)="+endPaintID;

      str += "}";

      return str;
   }


   public String pgffillcolor(BBox box)
   {
      if (box == null)
      {
         return startColor.pgffillcolor(box);
      }

      String eol = System.getProperty("line.separator", "\n");

      String str = "";
      int angle=0;
      double height = box.getHeight();
      double width  = box.getWidth();

      switch (direction)
      {
         case NORTH:
           str = pgfdeclareverticalshading(startColor,endColor);
           angle=0;
         break;
         case NORTH_EAST :
           str = pgfdeclareverticalshading(startColor,endColor);
           angle=45;
         break;
         case EAST :
           str = pgfdeclarehorizontalshading(startColor,endColor);
           angle = 0;
         break;
         case SOUTH_EAST :
           str = pgfdeclarehorizontalshading(startColor,endColor);
           angle = 45;
         break;
         case SOUTH :
           str = pgfdeclareverticalshading(endColor,startColor);
           angle = 0;
         break;
         case SOUTH_WEST :
           str = pgfdeclareverticalshading(endColor,startColor);
           angle = 45;
         break;
         case WEST :
           str = pgfdeclarehorizontalshading(endColor,startColor);
           angle = 0;
         break;
         case NORTH_WEST :
           str = pgfdeclarehorizontalshading(endColor,startColor);
           angle = 45;
         break;
      }

      double opacity = getAlpha();

      if (opacity != 1.0)
      {
         str += "\\pgfsetfillopacity{"+PGF.format(opacity)+"}";
      }
      str += eol+ "\\pgfshadepath{jdrlinear"+pgfshadeid+"}{"+PGF.format(-angle)+"}";
      pgfshadeid++;

      return str;
   }

   public String getID()
   {
      return "gradient-"+startColor.getID()+"-"+endColor.getID()+"-"+direction;
   }

   private void svgDef(PrintWriter out) throws IOException
   {
      out.println("      <linearGradient id=\""+getID()+"\"");
      out.println("         gradientUnits=\"objectBoundingBox\"");

      int x1=0, y1=0, x2=0, y2=100;

      switch (direction)
      {
         case NORTH :
            x1 = 50;
            y1 = 100;
            x2 = 50;
            y1 = 0;
         break;
         case NORTH_EAST :
            x1 = 0;
            y1 = 100;
            x2 = 100;
            y2 = 0;
         break;
         case EAST :
            x1 = 0;
            y1 = 50;
            x2 = 100;
            y2 = 50;
         break;
         case SOUTH_EAST :
            x1 = 0;
            y1 = 0;
            x2 = 100;
            y2 = 100;
         break;
         case SOUTH :
            x1 = 50;
            y1 = 0;
            x2 = 50;
            y2 = 100;
         break;
         case SOUTH_WEST :
            x1 = 100;
            y1 = 0;
            x2 = 0;
            y2 = 100;
         break;
         case WEST :
            x1 = 100;
            y1 = 50;
            x2 = 0;
            y2 = 50;
         break;
         case NORTH_WEST :
            x1 = 100;
            y1 = 100;
            x2 = 0;
            y2 = 0;
         break;
      }

      out.println("         x1=\""+x1+"%\" "
        +"y1=\""+y1+"%\" " + "x2=\""+x2+"%\" "
        +"y2=\""+y2+"%\">");

      out.println("         <stop offset=\"0%\" stop-color=\""+
         startColor.svg()+"\" stroke-opacity=\""
        +startColor.getAlpha()+"\"/>");
      out.println("         <stop offset=\"100%\" stop-color=\""+
         endColor.svg()+"\" stroke-opacity=\""
        +endColor.getAlpha()+"\"/>");
      out.println("      </linearGradient>");
   }

   /**
    * Iterates through all elements of the specified group 
    * and writes the SVG definitions for any gradient paint
    * used. {@link #svgFill()} and {@link #svgLine()} use
    * these gradient definitions.
    * @param out the output stream
    * @param group the JDR image
    * @throws IOException if I/O error occurs
    */
   public static void svgDefs(PrintWriter out, JDRGroup group)
      throws IOException
   {
      Hashtable<String,JDRGradient> gradients
         = new Hashtable<String,JDRGradient>();

      for (int i = 0; i < group.size(); i++)
      {
         JDRCompleteObject object = group.get(i);

         JDRPaint p;

         if (object instanceof JDRShape)
         {
            p = ((JDRShape)object).getLinePaint();

            if (p instanceof JDRGradient)
            {
               gradients.put(((JDRGradient)p).getID(), (JDRGradient)p);
            }

            p = ((JDRShape)object).getFillPaint();

            if (p instanceof JDRGradient)
            {
               gradients.put(((JDRGradient)p).getID(), (JDRGradient)p);
            }
         }

         if (object instanceof JDRTextual)
         {
            p = ((JDRTextual)object).getTextPaint();

            if (p instanceof JDRGradient)
            {
               gradients.put(((JDRGradient)p).getID(), (JDRGradient)p);
            }
         }
      }

      for (Enumeration e = gradients.keys(); e.hasMoreElements(); )
      {
         String id = (String)e.nextElement();

         JDRGradient p = (JDRGradient)gradients.get(id);
         p.svgDef(out);
      }
   }

   public String svgFill()
   {
      return "fill=\"url(#"+getID()+")\"";
   }

   public String svgLine()
   {
      return "stroke=\"url(#"+getID()+")\"";
   }

   public String svg()
   {
      return "url(#"+getID()+")";
   }

   public double getAlpha()
   {
      return 0.5*(startColor.getAlpha()+endColor.getAlpha());
   }

   public String pgf(BBox box)
   {
      return startColor.pgf(box);
   }

   public String pgfstrokecolor(BBox box)
   {
      return startColor.pgfstrokecolor(box);
   }

   public int psLevel()
   {
      return 2;
   }

   public void saveEPS(PrintWriter out, BBox box)
      throws IOException
   {
      if (box == null)
      {
         startColor.saveEPS(out, box);
         return;
      }

      double x0=0, y0=0, x1=0, y1=0;
      double minX = box.getMinX();
      double maxX = box.getMaxX();
      double minY = box.getMinY();
      double maxY = box.getMaxY();

      switch (direction)
      {
         case NORTH:
            x0 = minX+0.5*box.getWidth();
            y1 = minY;
            x1 = x0;
            y0 = maxY;
         break;
         case NORTH_EAST :
            x0 = minX;
            y1 = minY;
            x1 = maxX;
            y0 = maxY;
         break;
         case EAST :
            x0 = minX;
            y1 = minY+0.5*box.getHeight();
            x1 = maxX;
            y0 = y1;
         break;
         case SOUTH_EAST :
            x0 = minX;
            y1 = maxY;
            x1 = maxX;
            y0 = minY;
         break;
         case SOUTH :
            x1 = minX+0.5*box.getWidth();
            y0 = minY;
            x0 = x0;
            y1 = maxY;
         break;
         case SOUTH_WEST :
            x0 = maxX;
            y1 = maxY;
            x1 = minX;
            y0 = minY;
         break;
         case WEST :
            x1 = minX;
            y0 = minY+0.5*box.getHeight();
            x0 = maxX;
            y1 = y0;
         break;
         case NORTH_WEST :
            x0 = maxX;
            y1 = minY;
            x1 = minX;
            y0 = maxY;
         break;
      }

      out.println("<<");

      out.println("   /ShadingType 2");

      String c0, c1;

      if (startColor instanceof JDRColor)
      {
         JDRColor c = (JDRColor)startColor;
         out.println("   /ColorSpace /DeviceRGB");
         c0 = "/C0 ["+c.getRed()+" "+c.getGreen()+" "+c.getBlue()+"]";

         c = (endColor instanceof JDRColor) ? 
             (JDRColor)endColor :
             ((JDRColorCMYK)endColor).getJDRColor();

         c1 = "/C1 ["+c.getRed()+" "+c.getGreen()+" "+c.getBlue()+"]";
      }
      else
      {
         JDRColorCMYK c = (JDRColorCMYK)startColor;
         out.println("   /ColorSpace /DeviceCMYK");
         c0 = "/C0 ["+c.getCyan()+" "+c.getMagenta()+" "+c.getYellow()+" "+c.getKey()+"]";

         c = (endColor instanceof JDRColorCMYK) ? 
             (JDRColorCMYK)endColor :
             ((JDRColor)endColor).getJDRColorCMYK();

         c1 = "/C1 ["+c.getCyan()+" "+c.getMagenta()+" "+c.getYellow()+" "+c.getKey()+"]";
      }

      out.println("   /BBox ["+minX+" "+minY+" "+maxX+" "+maxY+"]");
      out.println("   /Coords ["+x0+" "+y0+" "+x1+" "+y1+"]");
      out.println("      /Function <<");
      out.println("         /FunctionType 2");
      out.println("         /Domain [0 1]");
      out.println("         "+c0);
      out.println("         "+c1);
      out.println("         /N 1");
      out.println("   >>");
      out.print(">> ");
   }

   /**
    * Converts this shading to a radial shading. This creates a
    * new radial shading with the same start and end colour.
    * @return a radial shading with the same start and end colour
    */
   public JDRRadial getJDRRadial()
   {
      try
      {
         return new JDRRadial(direction, startColor, endColor);
      }
      catch (InvalidFormatException ignore)
      {
         // this shouldn't happen
      }

      return new JDRRadial();
   }

   /**
    * Converts this shading to the given shading type.
    * If <code>label</code> is <code>"JDRGradient"</code>, returns
    * this shading. If <code>label</code> is 
    * <code>"JDRRadial"</code>, returns {@link #getJDRRadial()}.
    * Otherwise throws {@link InvalidFormatException}.
    */
   public JDRShading convertShading(String label)
   throws InvalidFormatException
   {
      if (label.equals("JDRRadial"))
      {
         return getJDRRadial();
      }
      else if (label.equals("JDRGradient"))
      {
         return this;
      }

      throw new InvalidFormatException("Can't convert from JDRGradient to "+label);
   }

   public JDRPaintLoaderListener getListener()
   {
      return listener;
   }

   /**
    * Gets this shading's start colour.
    * @return this shading's start colour
    */
   public JDRPaint getStartColor()
   {
      return startColor;
   }

   /**
    * Gets this shading's end colour.
    * @return this shading's end colour
    */
   public JDRPaint getEndColor()
   {
      return endColor;
   }

   /**
    * Gets the direction of this shading.
    * @return the direction of this shading
    */
   public int getDirection()
   {
      return direction;
   }

   /**
    * Sets this shading's start colour.
    * @throws InvalidStartColourException if the start colour 
    * implements {@link JDRShading}
    */
   public void setStartColor(JDRPaint sColor)
   throws InvalidStartColourException
   {
      if (sColor instanceof JDRShading)
      {
         throw new InvalidStartColourException();
      }

      startColor = sColor;
   }

   /**
    * Sets this shading's end colour.
    * @throws InvalidEndColourException if the end colour 
    * implements {@link JDRShading}
    */
   public void setEndColor(JDRPaint eColor)
   throws InvalidEndColourException
   {
      if (eColor instanceof JDRShading)
      {
         throw new InvalidEndColourException();
      }

      endColor = eColor;
   }

   /**
    * Sets this shading's direction.
    * @throws InvalidGradientDirectionException if the gradient
    * is not one of: {@link #NORTH}, {@link #NORTH_EAST},
    * {@link #EAST}, {@link #SOUTH_EAST}, {@link #SOUTH},
    * {@link #SOUTH_WEST}, {@link #WEST} or {@link #NORTH_WEST}
    */
   public void setDirection(int d)
   throws InvalidGradientDirectionException
   {
      if (d < 0 || d > 7)
      {
         throw new InvalidGradientDirectionException(d);
      }

      direction = d;
   }

   public void reduceToGreyScale()
   {
      startColor = startColor.getJDRGray();
      endColor = endColor.getJDRGray();
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;

      if (obj == null)
      {
         return false;
      }

      if (!(obj instanceof JDRGradient))
      {
         return false;
      }

      JDRGradient c = (JDRGradient)obj;

      return (getDirection() == c.getDirection()
           && startColor.equals(c.startColor)
           && endColor.equals(c.endColor));
   }

   public void fade(double value)
   {
      startColor.fade(value);
      endColor.fade(value);
   }

   private JDRPaint startColor, endColor;
   private int direction;

   /**
    * Indicates that the shading goes from South to North.
    */
   public final static int NORTH=0;
   /**
    * Indicates that the shading goes from South-West to North-East.
    */
   public final static int NORTH_EAST=1;
   /**
    * Indicates that the shading goes from West to East.
    */
   public final static int EAST=2;
   /**
    * Indicates that the shading goes from North-West to South-East.
    */
   public final static int SOUTH_EAST=3;
   /**
    * Indicates that the shading goes from North to South.
    */
   public final static int SOUTH=4;
   /**
    * Indicates that the shading goes from North-East to South-West.
    */
   public final static int SOUTH_WEST=5;
   /**
    * Indicates that the shading goes from East to West.
    */
   public final static int WEST=6;
   /**
    * Indicates that the shading goes from South-East to North-West.
    */
   public final static int NORTH_WEST=7;

   private static int pgfshadeid=0;

   private static JDRGradientListener listener = new JDRGradientListener();
}
