// File          : JDRFont.java
// Date          : 1st February 2006
// Last Modified : 1 March 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing font styles. This is based on the way LaTeX
 * deals with fonts. That is, the font style consists of the
 * following attributes:
 *<ul>
 *<li>Family : the name identifying the font family. 
 * This may be a generic name like "Serif" or a name such as
 * "Lucida Bright".
 * <p>
 * <li> Series : the font series or weight. This may be either bold
 * or medium.
 * <p>
 * <li> Shape : the font shape. This may be one of: upright,
 * emphasized, italic, slanted or small caps.
 * <p>
 * <li> Size: the font size in PostScript points.
 *</ul>
 * The LaTeX equivalents (<code>&#092;rmfamily</code> etc) are stored in 
 * {@link LaTeXFont} and the font size conversion (to declarations
 * such as <code>&#092;large</code>) is implemented by 
 * {@link LaTeXFontBase}.
 * @see LaTeXFont
 * @see LaTeXFontBase
 */
public class JDRFont implements Cloneable,Serializable
{
   /**
    * Creates a new font style.
    * @param fontFamily the font family (e.g. "Serif")
    * @param fontWeight the font weight (may be either
    * {@link #SERIES_MEDIUM} or {@link #SERIES_BOLD})
    * @param fontShape the font shape (may be one of:
    * {@link #SHAPE_UPRIGHT}, {@link #SHAPE_EM}, {@link #SHAPE_ITALIC},
    * {@link #SHAPE_SLANTED} or {@link #SHAPE_SC})
    * @param fontSize the font size (in PostScript points)
    * @throws InvalidFontShapeException if font shape is not one if:
    * {@link #SHAPE_UPRIGHT}, {@link #SHAPE_EM}, {@link #SHAPE_ITALIC},
    * {@link #SHAPE_SLANTED} or {@link #SHAPE_SC}
    * @throws InvalidFontWeightException if font weight is not one of:
    * {@link #SERIES_MEDIUM} or {@link #SERIES_BOLD}
    * @throws InvalidFontSizeException if the font size is negative
    */
   public JDRFont(String fontFamily, int fontWeight, int fontShape, int fontSize)
      throws InvalidFontShapeException,InvalidFontWeightException,
         InvalidFontSizeException
   {
      setFamily(fontFamily);
      setWeight(fontWeight);
      setShape(fontShape);
      setSize(fontSize);
   }

   /**
    * Creates default font style. (SanSerif family, medium weight,
    * upright shape and 10bp size.)
    */
   public JDRFont()
   {
      family = "SansSerif";
      weight = 0;
      shape  = 0;
      size   = 10;
   }

   /**
    * Gets this style's font family.
    * @return the name of the font family for this style
    */
   public String getFamily()
   {
      return family;
   }

   /**
    * Gets this style's font weight.
    * @return the font weight for this style
    */
   public int getWeight()
   {
      return weight;
   }

   /**
    * Gets this style's font shape.
    * @return the font shape for this style
    */
   public int getShape()
   {
      return shape;
   }

   /**
    * Gets this style's font size.
    * @return the font size for this style
    */
   public int getSize()
   {
      return size;
   }

   /**
    * Sets the font family for this style.
    * @param fontFamily the font family name
    */
   public void setFamily(String fontFamily)
   {
      family = fontFamily;
   }

   /**
    * Sets the font weight for this style.
    * @param fontWeight the font weight
    * @throws InvalidFontWeightException if font weight is not one of:
    * {@link #SERIES_MEDIUM} or {@link #SERIES_BOLD}
    */
   public void setWeight(int fontWeight)
      throws InvalidFontWeightException
   {
      if (fontWeight < SERIES_MEDIUM || fontWeight > SERIES_BOLD)
      {
         throw new InvalidFontWeightException(fontWeight);
      }

      weight = fontWeight;
   }

   /**
    * Sets the font shape for this style.
    * @param fontShape the font shape
    * @throws InvalidFontShapeException if font shape is not one if:
    * {@link #SHAPE_UPRIGHT}, {@link #SHAPE_EM}, {@link #SHAPE_ITALIC},
    * {@link #SHAPE_SLANTED} or {@link #SHAPE_SC}
    */
   public void setShape(int fontShape)
      throws InvalidFontShapeException
   {
      if (fontShape < SHAPE_UPRIGHT || fontShape > SHAPE_SC)
      {
         throw new InvalidFontShapeException(fontShape);
      }

      shape = fontShape;
   }

   /**
    * Sets the font size for this style.
    * @param fontSize the font size
    * @throws InvalidFontSizeException if the font size is negative
    */
   public void setSize(int fontSize)
      throws InvalidFontSizeException
   {
      if (fontSize < 0)
      {
         throw new InvalidFontSizeException(fontSize);
      }

      size = fontSize;
   }

   /**
    * Saves this font style in JDR format.
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    * @see #read(DataInputStream,float)
    * @see #saveAJR(PrintWriter,float)
    */
   public void save(DataOutputStream dout, float version)
      throws IOException
   {
      dout.writeInt(family.length());
      dout.writeChars(family);
      dout.writeByte((byte)shape);
      dout.writeByte((byte)weight);
      dout.writeInt(size);
   }

   /**
    * Reads a font style in JDR format.
    * @param din the input stream
    * @param version the JDR version
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if the input stream is not
    * in the required format
    * @return the font style read from the input stream
    * @see #save(DataOutputStream,float)
    * @see #readAJR(BufferedReader,float)
    */
   public static JDRFont read(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      int n = din.readInt();

      JDRFont font = new JDRFont();

      if (n <= 0)
      {
         throw new InvalidFontFamilyLengthException(n);
      }

      char[] family = new char[n];

      for (int i = 0; i < n; i++)
      {
         family[i] = din.readChar();
      }

      font.setFamily(new String(family));

      int shape = (int)din.readByte();

      font.setShape(shape);

      int series = (int)din.readByte();

      font.setWeight(series);

      int fsize = din.readInt();

      font.setSize(fsize);

      return font;
   }

   /**
    * Saves this font style in AJR format.
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    * @see #readAJR(BufferedReader,float)
    * @see #save(DataOutputStream,float)
    */
   public void saveAJR(PrintWriter out, float version)
      throws IOException
   {
      AJR.writeInt(out, family.length());

      out.print(family+" ");

      AJR.writeInt(out, shape);
      AJR.writeInt(out, weight);
      AJR.writeInt(out, size);
      out.println();
   }

   /**
    * Reads a font style in AJR format.
    * @param in the input stream
    * @param version the AJR version
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if the input stream is not
    * in the required format
    * @throws EOFException if the file terminates unexpectedly
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * overflows
    * @return the font style read from the input stream
    * @see #saveAJR(PrintWriter,float)
    * @see #read(DataInputStream,float)
    */
   public static JDRFont readAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      int n = AJR.readInt(in);

      JDRFont font = new JDRFont();

      if (n <= 0)
      {
         throw new InvalidFontFamilyLengthException(n, AJR.getLineNum());
      }

      String family = AJR.readString(in, n);

      font.setFamily(family);

      int shape = AJR.readInt(in);

      font.setShape(shape);

      int series = AJR.readInt(in);

      font.setWeight(series);

      int fsize = AJR.readInt(in);

      font.setSize(fsize);

      return font;
   }

   /**
    * Gets this font style in SVG format.
    * @return SVG tags specifying this font style
    */
   public String svg()
   {
      return "font-family=\""+family+"\""
       + " font-weight=\""
       + (weight == SERIES_MEDIUM ? "normal" : "bold")
       + "\""
       + " font-style=\""
       + (shape == SHAPE_UPRIGHT ? "normal" : "italic")
       + "\""
       + " font-size=\"" + size + "\"";
   }

   /**
    * Not yet implemented.
    * @return empty string
    */
   public String saveEPS()
   {
      // not yet implemented

      return "";
   }

   /**
    * Gets a copy of this style.
    * @return a copy of this font style
    */
   public Object clone()
   {
      try
      {
         return new JDRFont(family, weight, shape, size);
      }
      catch (InvalidFormatException e)
      {
         // this shouln't happen
      }

      return new JDRFont();
   }

   /**
    * Sets this font style to be the same as another font style.
    * @param font the other font style
    */
   public void makeEqual(JDRFont font)
   {
      family = font.family;
      weight = font.weight;
      shape  = font.shape;
      size   = font.size;
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;
      if (obj == null) return false;
      if (!(obj instanceof JDRFont)) return false;

      JDRFont f = (JDRFont)obj;
      if (!family.equals(f.family)) return false;
      if (weight != f.weight) return false;
      if (shape != f.shape) return false;
      if (size != f.size) return false;

      return true;
   }

   public String info()
   {
      return "family="+family+",weight="+weight+",shape="+shape
      +",size="+size;
   }

   /**
    * Indicates medium font weight.
    */
   public static final int SERIES_MEDIUM=0;
   /**
    * Indicates bold font weight.
    */
   public static final int SERIES_BOLD=1;

   /**
    * Indicates upright font shape.
    */
   public static final int SHAPE_UPRIGHT = 0;
   /**
    * Indicates emphasized font shape.
    */
   public static final int SHAPE_EM      = 1;
   /**
    * Indicates italic font shape.
    */
   public static final int SHAPE_ITALIC  = 2;
   /**
    * Indicates slanted font shape.
    */
   public static final int SHAPE_SLANTED = 3;
   /**
    * Indicates small caps font shape.
    */
   public static final int SHAPE_SC      = 4;

   private String family="SansSerif";
   private int weight=0, shape=0, size=10;
}

