// File          : JDRColorListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for RGB paint.
 * @author Nicola L C Talbot
 */

public class JDRColorListener implements JDRPaintLoaderListener
{
   public char getJDRid(float version)
   {
      return 'R';
   }

   public void writeJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
   throws IOException
   {
      JDRColor c = (JDRColor)paint;

      dout.writeFloat((float)c.getRed());
      dout.writeFloat((float)c.getGreen());
      dout.writeFloat((float)c.getBlue());
      dout.writeFloat((float)c.getAlpha());
   }

   public JDRPaint readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      double r = din.readFloat();

      double g = din.readFloat();

      double b = din.readFloat();

      double a = din.readFloat();

      return new JDRColor(r,g,b,a);
   }

   public char getAJRid(float version)
   {
      return 'R';
   }

   public void writeAJR(JDRPaint paint,
                        PrintWriter out,
                        float version)
   throws IOException
   {
      JDRColor c = (JDRColor)paint;

      AJR.writeFloat(out, (float)c.getRed());
      AJR.writeFloat(out, (float)c.getGreen());
      AJR.writeFloat(out, (float)c.getBlue());
      AJR.writeFloat(out, (float)c.getAlpha());
   }

   public JDRPaint readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      double red = AJR.readFloat(in);

      if (red < 0.0 || red > 1.0)
      {
         throw new InvalidRedValueException(red, AJR.getLineNum());
      }

      double green = AJR.readFloat(in);

      if (green < 0.0 || green > 1.0)
      {
         throw new InvalidGreenValueException(green, AJR.getLineNum());
      }

      double blue = AJR.readFloat(in);

      if (blue < 0.0 || blue > 1.0)
      {
         throw new InvalidBlueValueException(blue, AJR.getLineNum());
      }

      double a = AJR.readFloat(in);

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a, AJR.getLineNum());
      }

      return new JDRColor(red,green,blue,a);
   }

   public int getConfigId()
   {
      return 1;
   }

   /**
    * @param paint the paint to save (must be {@link JDRColor}).
    */
   public String getConfigString(JDRPaint paint)
   {
      JDRColor c = (JDRColor)paint;

      return "" + c.getRed() 
           +"," + c.getGreen()
           +"," + c.getBlue()
           +"," + c.getAlpha();
   }

   public JDRPaint parseConfig(String specs)
   throws InvalidFormatException
   {
      String split[] = specs.split(",",5);

      if (split.length < 4)
      {
         throw new InvalidPaintException("Invalid RGB config format");
      }     

      double red   = 0;
      double green = 0;
      double blue  = 0;
      double alpha = 0;

      try
      {
         red = Double.parseDouble(split[0]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid red value "+split[0]);
      }

      try
      {
         green = Double.parseDouble(split[1]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid green value "+split[1]);
      }

      try
      {
         blue = Double.parseDouble(split[2]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid blue value "+split[2]);
      }

      try
      {
         alpha = Double.parseDouble(split[3]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid alpha value "+split[3]);
      }

      if (split.length == 5)
      {
         remainder = split[4];
      }
      else
      {
         remainder = "";
      }

      return new JDRColor(red, green, blue, alpha);
   }

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder()
   {
      return remainder;
   }

   private String remainder="";
}
