// File          : JDRColorHSBListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for HSB paint.
 * @author Nicola L C Talbot
 */

public class JDRColorHSBListener implements JDRPaintLoaderListener
{
   public char getJDRid(float version)
   {
      return 'S';
   }

   public void writeJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
   throws IOException
   {
      JDRColorHSB c = (JDRColorHSB)paint;

      dout.writeFloat((float)c.getHue());
      dout.writeFloat((float)c.getSaturation());
      dout.writeFloat((float)c.getBrightness());
      dout.writeFloat((float)c.getAlpha());
   }

   public JDRPaint readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      double h = din.readFloat();

      if (h < 0.0 || h >= 360)
      {
         throw new InvalidHueValueException(h);
      }

      double s = din.readFloat();

      if (s < 0.0 || s > 1.0)
      {
         throw new InvalidSaturationValueException(s);
      }

      double b = din.readFloat();

      if (b < 0.0 || b > 1.0)
      {
         throw new InvalidBrightnessValueException(b);
      }

      double a = din.readFloat();

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      return new JDRColorHSB(h,s,b,a);
   }

   public char getAJRid(float version)
   {
      return 'S';
   }

   public void writeAJR(JDRPaint paint,
                        PrintWriter out,
                        float version)
   throws IOException
   {
      JDRColorHSB c = (JDRColorHSB)paint;

      AJR.writeFloat(out, (float)c.getHue());
      AJR.writeFloat(out, (float)c.getSaturation());
      AJR.writeFloat(out, (float)c.getBrightness());
      AJR.writeFloat(out, (float)c.getAlpha());
   }

   public JDRPaint readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      double hue = AJR.readFloat(in);

      if (hue < 0.0 || hue > 1.0)
      {
         throw new InvalidHueValueException(hue, AJR.getLineNum());
      }

      double saturation = AJR.readFloat(in);

      if (saturation < 0.0 || saturation > 1.0)
      {
         throw new InvalidSaturationValueException(saturation, AJR.getLineNum());
      }

      double brightness = AJR.readFloat(in);

      if (brightness < 0.0 || brightness > 1.0)
      {
         throw new InvalidBrightnessValueException(brightness, AJR.getLineNum());
      }

      double a = AJR.readFloat(in);

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a, AJR.getLineNum());
      }

      return new JDRColorHSB(hue,saturation,brightness,a);
   }

   public int getConfigId()
   {
      return 5;
   }

   /**
    * @param paint the paint to save (must be {@link JDRColorHSB}).
    */
   public String getConfigString(JDRPaint paint)
   {
      JDRColorHSB c = (JDRColorHSB)paint;

      return "" + c.getHue() 
           +"," + c.getSaturation()
           +"," + c.getBrightness()
           +"," + c.getAlpha();
   }

   public JDRPaint parseConfig(String specs)
   throws InvalidFormatException
   {
      String split[] = specs.split(",",5);

      if (split.length < 4)
      {
         throw new InvalidPaintException("Invalid HSB config format");
      }     

      double hue = 0;
      double saturation = 0;
      double brightness  = 0;
      double alpha   = 0;

      try
      {
         hue = Double.parseDouble(split[0]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid hue value "+split[0]);
      }

      try
      {
         saturation = Double.parseDouble(split[1]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid saturation value "+split[1]);
      }

      try
      {
         brightness = Double.parseDouble(split[2]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid brightness value "+split[2]);
      }

      try
      {
         alpha = Double.parseDouble(split[3]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid alpha value "+split[3]);
      }

      if (split.length == 5)
      {
         remainder = split[4];
      }
      else
      {
         remainder = "";
      }

      return new JDRColorHSB(hue, saturation, brightness, alpha);
   }

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder()
   {
      return remainder;
   }

   private String remainder="";
}
