// File          : JDRColorHSB.java
// Date          : 20th March 2007
// Last Modified : 20th May 200*
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing HSB colour.
 * @author Nicola L C Talbot
 */

public class JDRColorHSB implements JDRPaint,Serializable
{
   /**
    * Creates a new HSB colour. The transparency is set to 1 (opaque).
    * The hue must be in the range [0, 360).
    * The saturation and brightness must be in the range 0-1.
    * @param h hue
    * @param s saturation
    * @param b brightness
    * @throws InvalidHueValueException if hue is less than 0 or greater
    * than or equal to 360
    * @throws InvalidSaturationValueException if saturation is less
    * than 0 or greater than 1
    * @throws InvalidBrightnessValueException if brightness is
    * less than 0 or greater than 1
    */
   public JDRColorHSB(double h, double s, double b)
   throws InvalidHueValueException,
          InvalidSaturationValueException,
          InvalidBrightnessValueException
   {
      if (h < 0.0 || h >= 360.0)
      {
         throw new InvalidHueValueException(h);
      }

      if (s < 0.0 || s > 1.0)
      {
         throw new InvalidSaturationValueException(s);
      }

      if (b < 0.0 || b > 1.0)
      {
         throw new InvalidBrightnessValueException(b);
      }

      hue        = h;
      saturation = s;
      brightness = b;
      alpha      = 1.0;
   }

   /**
    * Creates a new HSB colour.
    * The hue must be in the range [0, 360).
    * The saturation and brightness must be in the range 0-1.
    * @param h hue
    * @param s saturation
    * @param b brightness
    * @param a alpha (transparency)
    * @throws InvalidHueValueException if hue is less than 0 or greater
    * than or equal to 360
    * @throws InvalidSaturationValueException if saturation is less
    * than 0 or greater than 1
    * @throws InvalidBrightnessValueException if brightness is
    * less than 0 or greater than 1
    * @throws InvalidAlphaValueException if brightness is
    * less than 0 or greater than 1
    */
   public JDRColorHSB(double h, double s, double b, double a)
   throws InvalidHueValueException,
          InvalidSaturationValueException,
          InvalidBrightnessValueException,
          InvalidAlphaValueException
   {
      if (h < 0.0 || h >= 360.0)
      {
         throw new InvalidHueValueException(h);
      }

      if (s < 0.0 || s > 1.0)
      {
         throw new InvalidSaturationValueException(s);
      }

      if (b < 0.0 || b > 1.0)
      {
         throw new InvalidBrightnessValueException(b);
      }

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      hue        = h;
      saturation = s;
      brightness = b;
      alpha      = a;
   }

   /**
    * Creates a new HSB colour (black).
    */
   public JDRColorHSB()
   {
      hue        = 0;
      saturation = 0;
      brightness = 0;
      alpha      = 1;
   }

   public JDRGray getJDRGray()
   {
      return getJDRColor().getJDRGray();
   }

   public JDRColorCMYK getJDRColorCMYK()
   {
      return getJDRColor().getJDRColorCMYK();
   }

   /**
    * Gets this.
    * @return this colour
    */
   public JDRColorHSB getJDRColorHSB()
   {
      return this;
   }

   public JDRColor getJDRColor()
   {
      // convert from hsb to rgb
      int h = ((int)Math.floor(hue/60)) % 6;
      double f = hue/60 - Math.floor(hue/60);
      double p = brightness*(1-saturation);
      double q = brightness*(1-f*saturation);
      double t = brightness*(1-(1-f)*saturation);
      double red, green, blue;

      switch (h)
      {
         case 0:
            red   = brightness;
            green = t;
            blue  = p;
         break;
         case 1:
            red   = q;
            green = brightness;
            blue  = p;
         break;
         case 2:
            red   = p;
            green = brightness;
            blue  = t;
         break;
         case 3:
            red   = p;
            green = q;
            blue  = brightness;
         break;
         case 4:
            red   = t;
            green = p;
            blue  = brightness;
         break;
         default:
            red   = brightness;
            green = p;
            blue  = q;
      }

      JDRColor c=null;

      try
      {
         c = new JDRColor(red, green,blue,alpha);
      }
      catch (InvalidFormatException e)
      {
      }

      return c;
   }

   public Paint getPaint(BBox box)
   {
      return getColor();
   }

   public Color getColor()
   {
      return getJDRColor().getColor();
   }

   public String toString()
   {
      return new String("JDRColorHSB@"+"H:" +hue+"S:" +saturation+"B:"
                        +brightness+"A:"+alpha);
   }

   public Object clone()
   {
      try
      {
      return new JDRColorHSB(hue,saturation,brightness,alpha);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRColorHSB();
   }

   public String pgf(BBox box)
   {
      return getJDRColor().pgf(box);
   }

   public String pgfstrokecolor(BBox box)
   {
      if (alpha == 1.0)
      {
         return pgf(box);
      }
      else
      {
         return "\\pgfsetstrokeopacity{"+PGF.format(alpha)+"}"
            + pgf(box);
      }
   }

   public String pgffillcolor(BBox box)
   {
      if (alpha==1.0)
      {
         return pgf(box);
      }
      else
      {
         return "\\pgfsetfillopacity{"+PGF.format(alpha)+"}"
            + pgf(box);
      }
   }

   public void saveEPS(PrintWriter out, BBox box)
      throws IOException
   {
      out.println(""+(hue/360.0)+" "+saturation+" "+brightness +" sethsbcolor");
   }

   public int psLevel()
   {
      return 1;
   }

   public double getAlpha()
   {
      return alpha;
   }

   /**
    * Sets the alpha component for this colour. This value must be
    * in the range 0 to 1.
    * @param a the alpha component
    * @throws InvalidAlphaValueException if a &lt; 0 or a &gt; 1
    */
   public void setAlpha(double a)
      throws InvalidAlphaValueException
   {
      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      alpha = a;
   }

   /**
    * Sets the hue component for this colour. This value must be
    * in the range [0, 1).
    * @param h the hue component
    * @throws InvalidHueValueException if h &lt; 0 or h &gt;= 360
    */
   public void setHue(double h)
      throws InvalidHueValueException
   {
      if (h < 0.0 || h >= 360)
      {
         throw new InvalidHueValueException(h);
      }

      hue = h;
   }

   /**
    * Sets the saturation component for this colour. This value must be
    * in the range 0 to 1.
    * @param s the saturation component
    * @throws InvalidSaturationValueException if s &lt; 0 or s &gt; 1
    */
   public void setSaturation(double s)
      throws InvalidSaturationValueException
   {
      if (s < 0.0 || s > 1.0)
      {
         throw new InvalidSaturationValueException(s);
      }

      saturation = s;
   }

   /**
    * Sets the brightness component for this colour. This value must be
    * in the range 0 to 1.
    * @param b the brightness component
    * @throws InvalidBrightnessValueException if b &lt; 0 or b &gt; 1
    */
   public void setBrightness(double b)
      throws InvalidBrightnessValueException
   {
      if (b < 0.0 || b > 1.0)
      {
         throw new InvalidBrightnessValueException(b);
      }

      brightness = b;
   }

   /**
    * Gets the hue component. This will be a value in the range
    * 0 to 1.
    * @return the hue component of this colour
    */
   public double getHue()
   {
      return hue;
   }

   /**
    * Gets the saturation component. This will be a value in the range
    * 0 to 1.
    * @return the saturation component of this colour
    */
   public double getSaturation()
   {
      return saturation;
   }

   /**
    * Gets the brightness component. This will be a value in the range
    * 0 to 1.
    * @return the brightness component of this colour
    */
   public double getBrightness()
   {
      return brightness;
   }

   public String getID()
   {
      return Integer.toHexString((int)hue)
           + "." + Integer.toHexString((int)(255*saturation))
           + "." + Integer.toHexString((int)(255*brightness))
           + "." + Integer.toHexString((int)(255*alpha));
   }

   public String svg()
   {
      return getJDRColor().svg();
   }

   public String svgFill()
   {
      return "fill=\""+getJDRColor().svg()+"\" fill-opacity=\""
      +getAlpha()+"\"";
   }

   public String svgLine()
   {
      return "stroke=\""+getJDRColor().svg()+"\" stroke-opacity=\""
      +getAlpha()+"\"";
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;

      if (obj == null)
      {
         return false;
      }

      if (!(obj instanceof JDRColorHSB))
      {
         return false;
      }

      JDRColorHSB c = (JDRColorHSB)obj;

      return (getHue() == c.getHue()
           && getSaturation() == c.getSaturation()
           && getBrightness() == c.getBrightness()
           && getAlpha() == c.getAlpha());
   }

   public JDRPaintLoaderListener getListener()
   {
      return listener;
   }

   public void fade(double value)
   {
      if (value < 0)
      {
         throw new IllegalArgumentException
            ("illegal fade factor "+value+". (Negative factors not permitted)");
      }

      alpha *= value;

      if (alpha > 1.0)
      {
         alpha = 1.0;
      }
   }

   private double hue, saturation, brightness, alpha;

   private static JDRColorHSBListener listener = new JDRColorHSBListener();
}
