// File          : JDRColorCMYKListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for CMYK paint.
 * @author Nicola L C Talbot
 */

public class JDRColorCMYKListener implements JDRPaintLoaderListener
{
   public char getJDRid(float version)
   {
      return 'C';
   }

   /**
    * Writes CMYK paint in JDR format.
    * @param paint the paint to save (must be {@link JDRColorCMYK}).
    */
   public void writeJDR(JDRPaint paint,
                    DataOutputStream dout,
                    float version)
   throws IOException
   {
      JDRColorCMYK c = (JDRColorCMYK)paint;

      dout.writeFloat((float)c.getCyan());
      dout.writeFloat((float)c.getMagenta());
      dout.writeFloat((float)c.getYellow());
      dout.writeFloat((float)c.getKey());
      dout.writeFloat((float)c.getAlpha());
   }

   public JDRPaint readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      double c = din.readFloat();

      if (c < 0.0 || c > 1.0)
      {
         throw new InvalidCyanValueException(c);
      }

      double m = din.readFloat();

      if (m < 0.0 || m > 1.0)
      {
         throw new InvalidMagentaValueException(m);
      }

      double y = din.readFloat();

      if (y < 0.0 || y > 1.0)
      {
         throw new InvalidYellowValueException(y);
      }

      double k = din.readFloat();

      if (k < 0.0 || k > 1.0)
      {
         throw new InvalidBlackValueException(k);
      }

      double a = din.readFloat();

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }


      return new JDRColorCMYK(c,m,y,k,a);
   }

   public char getAJRid(float version)
   {
      return 'C';
   }

   /**
    * Writes CMYK paint in AJR format.
    * @param paint the paint to save (must be {@link JDRColorCMYK}).
    */
   public void writeAJR(JDRPaint paint,
                        PrintWriter out,
                        float version)
   throws IOException
   {
      JDRColorCMYK c = (JDRColorCMYK)paint;

      AJR.writeFloat(out, (float)c.getCyan());
      AJR.writeFloat(out, (float)c.getMagenta());
      AJR.writeFloat(out, (float)c.getYellow());
      AJR.writeFloat(out, (float)c.getKey());
      AJR.writeFloat(out, (float)c.getAlpha());
   }

   public JDRPaint readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      double c = AJR.readFloat(in);

      if (c < 0.0 || c > 1.0)
      {
         throw new InvalidCyanValueException(c, AJR.getLineNum());
      }

      double m = AJR.readFloat(in);

      if (m < 0.0 || m > 1.0)
      {
         throw new InvalidMagentaValueException(m, AJR.getLineNum());
      }

      double y = AJR.readFloat(in);

      if (y < 0.0 || y > 1.0)
      {
         throw new InvalidYellowValueException(y, AJR.getLineNum());
      }

      double k = AJR.readFloat(in);

      if (k < 0.0 || k > 1.0)
      {
         throw new InvalidBlackValueException(k, AJR.getLineNum());
      }

      double a = AJR.readFloat(in);

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a, AJR.getLineNum());
      }

      return new JDRColorCMYK(c,m,y,k,a);
   }

   public int getConfigId()
   {
      return 2;
   }

   /**
    * @param paint the paint to save (must be {@link JDRColorCMYK}).
    */
   public String getConfigString(JDRPaint paint)
   {
      JDRColorCMYK c = (JDRColorCMYK)paint;

      return "" + c.getCyan() 
           +"," + c.getMagenta()
           +"," + c.getYellow()
           +"," + c.getKey()
           +"," + c.getAlpha();
   }

   public JDRPaint parseConfig(String specs)
   throws InvalidFormatException
   {
      String split[] = specs.split(",",6);

      if (split.length < 5)
      {
         throw new InvalidPaintException("Invalid CMYK config format");
      }     

      double cyan = 0;
      double magenta = 0;
      double yellow  = 0;
      double black   = 0;
      double alpha   = 0;

      try
      {
         cyan = Double.parseDouble(split[0]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid cyan value "+split[0]);
      }

      try
      {
         magenta = Double.parseDouble(split[1]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid magenta value "+split[1]);
      }

      try
      {
         yellow = Double.parseDouble(split[2]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid yellow value "+split[2]);
      }

      try
      {
         black = Double.parseDouble(split[3]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid black value "+split[3]);
      }

      try
      {
         alpha = Double.parseDouble(split[4]);
      }
      catch (NumberFormatException e)
      {
         throw new InvalidPaintException("Invalid alpha value "+split[4]);
      }

      if (split.length == 6)
      {
         remainder = split[5];
      }
      else
      {
         remainder = "";
      }

      return new JDRColorCMYK(cyan, yellow, magenta, black, alpha);
   }

   /**
    * Gets the remainder of the specs String after it has been
    * parsed by {@link #parseConfig(String)}.
    */
   public String getConfigRemainder()
   {
      return remainder;
   }

   private String remainder="";
}
