// File          : JDRColorCMYK.java
// Date          : 1st February 2006
// Last Modified : 16th February 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing CMYK colour.
 * @author Nicola L C Talbot
 */
public class JDRColorCMYK implements JDRPaint,Serializable
{
   /**
    * Creates a new CYMK colour. The transparency is set to 1.0
    * (opaque).
    * The parameters must all be in the range 0-1.
    * @param c cyan
    * @param m magenta
    * @param y yellow
    * @param k black
    * @throws InvalidCyanValueException if c &lt; 0 or c &gt; 1
    * @throws InvalidMagentaValueException if m &lt; 0 or m &gt; 1
    * @throws InvalidYellowValueException if y &lt; 0 or y &gt; 1
    * @throws InvalidBlackValueException if k &lt; 0 or k &gt; 1
    */
   public JDRColorCMYK(double c, double m, double y, double k)
      throws InvalidCyanValueException,
             InvalidMagentaValueException,
             InvalidYellowValueException,
             InvalidBlackValueException
   {
      if (c < 0.0 || c > 1.0)
      {
         throw new InvalidCyanValueException(c);
      }

      if (m < 0.0 || m > 1.0)
      {
         throw new InvalidMagentaValueException(m);
      }

      if (y < 0.0 || y > 1.0)
      {
         throw new InvalidYellowValueException(y);
      }

      if (k < 0.0 || k > 1.0)
      {
         throw new InvalidBlackValueException(k);
      }

      cyan    = c;
      magenta = m;
      yellow  = y;
      key     = k;
      alpha   = 1.0;
   }

   /**
    * Creates a new CYMK colour.
    * The parameters must all be in the range 0-1.
    * @param c cyan
    * @param m magenta
    * @param y yellow
    * @param k black
    * @param a alpha (transparency)
    * @throws InvalidCyanValueException if c &lt; 0 or c &gt; 1
    * @throws InvalidMagentaValueException if m &lt; 0 or m &gt; 1
    * @throws InvalidYellowValueException if y &lt; 0 or y &gt; 1
    * @throws InvalidBlackValueException if k &lt; 0 or k &gt; 1
    * @throws InvalidAlphaValueException if a &lt; 0 or a &gt; 1
    */
   public JDRColorCMYK(double c, double m, double y, double k, double a)
      throws InvalidCyanValueException,
             InvalidMagentaValueException,
             InvalidYellowValueException,
             InvalidBlackValueException,
             InvalidAlphaValueException
   {
      if (c < 0.0 || c > 1.0)
      {
         throw new InvalidCyanValueException(c);
      }

      if (m < 0.0 || m > 1.0)
      {
         throw new InvalidMagentaValueException(m);
      }

      if (y < 0.0 || y > 1.0)
      {
         throw new InvalidYellowValueException(y);
      }

      if (k < 0.0 || k > 1.0)
      {
         throw new InvalidBlackValueException(k);
      }

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      cyan    = c;
      magenta = m;
      yellow  = y;
      key     = k;
      alpha   = a;
   }

   /**
    * Creates a new CMYK colour (black).
    */
   public JDRColorCMYK()
   {
      cyan    = 0;
      magenta = 0;
      yellow  = 0;
      key     = 1;
      alpha   = 1;
   }

   /**
    * Gets this.
    * @return this
    */
   public JDRColorCMYK getJDRColorCMYK()
   {
      return this;
   }

   /**
    * Gets the closest matching {@link JDRGray}. This is obtained
    * as follows: if key=0, the gray value is
    * computed as 1-(cyan+magenta+yellow)/3, otherwise it is
    * computed as 1-key.
    */
   public JDRGray getJDRGray()
   {
      double gray;

      if (key == 0)
      {
         gray = 1-(cyan+magenta+yellow)/3;
      }
      else
      {
         gray = 1-key;
      }

      try
      {
         return new JDRGray(gray, alpha);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRGray();
   }

   public JDRColor getJDRColor()
   {
      // convert from cmyk to rgb
      double red   = 1.0-Math.min(1.0,cyan*(1-key)+key);
      double green = 1.0-Math.min(1.0,magenta*(1-key)+key);
      double blue  = 1.0-Math.min(1.0,yellow*(1-key)+key);

      JDRColor c=null;

      try
      {
         c = new JDRColor(red, green,blue,alpha);
      }
      catch (InvalidFormatException ignore)
      {
      }

      return c;
   }

   public JDRColorHSB getJDRColorHSB()
   {
      return getJDRColor().getJDRColorHSB();
   }

   public Paint getPaint(BBox box)
   {
      return getColor();
   }

   public Color getColor()
   {
      // convert from cmyk to rgb
      double red   = 1.0-Math.min(1.0,cyan*(1-key)+key);
      double green = 1.0-Math.min(1.0,magenta*(1-key)+key);
      double blue  = 1.0-Math.min(1.0,yellow*(1-key)+key);

      return new Color((float)red, (float)green, (float)blue, (float)alpha);
   }

   public String toString()
   {
      return new String("JDRColorCMYK@"+"C:" +cyan+"M:" +magenta+"Y:"
                        +yellow+"K:"+key+"A:"+alpha);
   }

   public Object clone()
   {
      try
      {
         return new JDRColorCMYK(cyan,magenta,yellow,key,alpha);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRColorCMYK();
   }

   public String pgf(BBox box)
   {
      return "\\color[cmyk]{"+PGF.format(cyan)+","
         +PGF.format(magenta)+","+PGF.format(yellow)+","
         +PGF.format(key)+"}";
   }

   public String pgfstrokecolor(BBox box)
   {
      if (alpha == 1.0)
      {
         return "\\color[cmyk]{"+PGF.format(cyan)+","
                 +PGF.format(magenta)+","+PGF.format(yellow)+","
                 +PGF.format(key)+"}";
      }
      else
      {
         return "\\pgfsetstrokeopacity{"+PGF.format(alpha)+"}"+
                "\\color[cmyk]{"+PGF.format(cyan)+","
                 +PGF.format(magenta)+","+PGF.format(yellow)+","
                 +PGF.format(key)+"}";
      }
   }

   public String pgffillcolor(BBox box)
   {
      if (alpha==1.0)
      {
         return "\\color[cmyk]{"+PGF.format(cyan)+","
                +PGF.format(magenta)+","+PGF.format(yellow)+","
                +PGF.format(key)+"}";
      }
      else
      {
         return "\\pgfsetfillopacity{"+PGF.format(alpha)+"}"+
                "\\color[cmyk]{"+PGF.format(cyan)+","
                +PGF.format(magenta)+","+PGF.format(yellow)+","
                +PGF.format(key)+"}";
      }
   }

   public void saveEPS(PrintWriter out, BBox box)
      throws IOException
   {
      out.println(""+cyan+" "+magenta+" "+yellow+" "+key
                 +" setcmykcolor");
   }

   public int psLevel()
   {
      return 2;
   }

   public double getAlpha()
   {
      return alpha;
   }

   /**
    * Sets the alpha component for this colour. This value must be
    * in the range 0 to 1.
    * @param a the alpha component
    * @throws InvalidAlphaValueException if a &lt; 0 or a &gt; 1
    */
   public void setAlpha(double a)
      throws InvalidAlphaValueException
   {
      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      alpha = a;
   }

   /**
    * Gets the cyan component. This will be a value in the range
    * 0 to 1.
    * @return the cyan component of this colour
    */
   public double getCyan()
   {
      return cyan;
   }

   /**
    * Gets the magenta component. This will be a value in the range
    * 0 to 1.
    * @return the magenta component of this colour
    */
   public double getMagenta()
   {
      return magenta;
   }

   /**
    * Gets the yellow component. This will be a value in the range
    * 0 to 1.
    * @return the yellow component of this colour
    */
   public double getYellow()
   {
      return yellow;
   }

   /**
    * Gets the key (black) component. This will be a value in the range
    * 0 to 1.
    * @return the key (black) component of this colour
    */
   public double getKey()
   {
      return key;
   }

   /**
    * Sets the cyan component for this colour. This value must be
    * in the range 0 to 1.
    * @param c the cyan component
    * @throws InvalidCyanValueException if c &lt; 0 or c &gt; 1
    */
   public void setCyan(double c)
      throws InvalidCyanValueException
   {
      if (c < 0.0 || c > 1.0)
      {
         throw new InvalidCyanValueException(c);
      }

      cyan = c;
   }

   /**
    * Sets the magenta component for this colour. This value must be
    * in the range 0 to 1.
    * @param m the magenta component
    * @throws InvalidMagentaValueException if m &lt; 0 or m &gt; 1
    */
   public void setMagenta(double m)
      throws InvalidMagentaValueException
   {
      if (m < 0.0 || m > 1.0)
      {
         throw new InvalidMagentaValueException(m);
      }

      magenta = m;
   }

   /**
    * Sets the yellow component for this colour. This value must be
    * in the range 0 to 1.
    * @param y the yellow component
    * @throws InvalidYellowValueException if y &lt; 0 or y &gt; 1
    */
   public void setYellow(double y)
      throws InvalidYellowValueException
   {
      if (y < 0.0 || y > 1.0)
      {
         throw new InvalidYellowValueException(y);
      }

      yellow = y;
   }

   /**
    * Sets the key (black) component for this colour. This value must be
    * in the range 0 to 1.
    * @param k the key component
    * @throws InvalidBlackValueException if k &lt; 0 or k &gt; 1
    */
   public void setKey(double k)
      throws InvalidBlackValueException
   {
      if (k < 0.0 || k > 1.0)
      {
         throw new InvalidBlackValueException(k);
      }

      key = k;
   }

   public String getID()
   {
      return Integer.toHexString((int)(255*cyan))
           + "." + Integer.toHexString((int)(255*magenta))
           + "." + Integer.toHexString((int)(255*yellow))
           + "." + Integer.toHexString((int)(255*key))
           + "." + Integer.toHexString((int)(255*alpha));
   }

   public String svg()
   {
      // svg doesn't really support CMYK
      return getJDRColor().svg();
   }

   public String svgFill()
   {
      return "fill=\""+svg()+"\" fill-opacity=\""
      +getAlpha()+"\"";
   }

   public String svgLine()
   {
      return "stroke=\""+svg()+"\" stroke-opacity=\""
      +getAlpha()+"\"";
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;

      if (obj == null)
      {
         return false;
      }

      if (!(obj instanceof JDRColorCMYK))
      {
         return false;
      }

      JDRColorCMYK c = (JDRColorCMYK)obj;

      return (getCyan() == c.getCyan()
           && getYellow() == c.getYellow()
           && getMagenta() == c.getMagenta()
           && getKey() == c.getKey()
           && getAlpha() == c.getAlpha());
   }

   public JDRPaintLoaderListener getListener()
   {
      return listener;
   }

   public void fade(double value)
   {
      if (value < 0)
      {
         throw new IllegalArgumentException
            ("illegal fade factor "+value+". (Negative factors not permitted)");
      }

      alpha *= value;

      if (alpha > 1.0)
      {
         alpha = 1.0;
      }
   }

   private double cyan, magenta, yellow, key, alpha;

   private static JDRColorCMYKListener listener = new JDRColorCMYKListener();
}
