// File          : JDRColor.java
// Date          : 1st February 2006
// Last Modified : 16th February 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing RGB colour.
 * @author Nicola L C Talbot
 */

public class JDRColor implements JDRPaint,Serializable
{
   /**
    * Creates a new RGB colour. The transparency is set to 1 (opaque).
    * The RGB parameters must be in the range 0-1.
    * @param r red
    * @param g green
    * @param b blue
    * @throws InvalidRedValueException if red value is less than 0 or
    * greater than 1
    * @throws InvalidGreenValueException if green value is less than 
    * 0 or greater than 1
    * @throws InvalidBlueValueException if blue value is less than 
    * 0 or greater than 1
    */
   public JDRColor(double r, double g, double b)
      throws InvalidRedValueException,
             InvalidGreenValueException,
             InvalidBlueValueException
   {
      if (r < 0.0 || r > 1.0)
      {
         throw new InvalidRedValueException(r);
      }

      if (g < 0.0 || g > 1.0)
      {
         throw new InvalidGreenValueException(g);
      }

      if (b < 0.0 || b > 1.0)
      {
         throw new InvalidBlueValueException(b);
      }

      red = r;
      green = g;
      blue = b;
      alpha = 1.0;
   }

   /**
    * Creates a new RGB colour. 
    * All parameters must be in the range 0-1. A value of a = 0 
    * indicates completely transparent and value of a = 1 
    * indicates completely opaque.
    * @param r red
    * @param g green
    * @param b blue
    * @param a alpha (transparency)
    * @throws InvalidRedValueException if red value is less than 0 or
    * greater than 1
    * @throws InvalidGreenValueException if green value is less than 
    * 0 or greater than 1
    * @throws InvalidBlueValueException if blue value is less than 
    * 0 or greater than 1
    * @throws InvalidAlphaValueException if alpha value is less than 
    * 0 or greater than 1
    */
   public JDRColor(double r, double g, double b, double a)
      throws InvalidRedValueException,
             InvalidGreenValueException,
             InvalidBlueValueException,
             InvalidAlphaValueException
   {
      if (r < 0.0 || r > 1.0)
      {
         throw new InvalidRedValueException(r);
      }

      if (g < 0.0 || g > 1.0)
      {
         throw new InvalidGreenValueException(g);
      }

      if (b < 0.0 || b > 1.0)
      {
         throw new InvalidBlueValueException(b);
      }

      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      red = r;
      green = g;
      blue = b;
      alpha = a;
   }

   /**
    * Creates a new RGB colour.
    * @param c RGB colour specification
    */
   public JDRColor(Color c)
   {
      red = c.getRed()/255.0;
      green = c.getGreen()/255.0;
      blue = c.getBlue()/255.0;
      alpha = c.getAlpha()/255.0;
   }

   /**
    * Creates a new RGB colour (black).
    */
   public JDRColor()
   {
      red   = 0;
      green = 0;
      blue  = 0;
      alpha = 1;
   }

   /**
    * Gets this.
    * @return this colour
    */
   public JDRColor getJDRColor()
   {
      return this;
   }

   public JDRGray getJDRGray()
   {
      double gray = (red+green+blue)/3;

      try
      {
         return new JDRGray(gray, alpha);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRGray();
   }

   public JDRColorCMYK getJDRColorCMYK()
   {
      double black   = Math.min(1.0-red,
                      Math.min(1.0-green,1.0-blue));
      double cyan    = 0;
      double magenta = 0;
      double yellow  = 0;

      if (black < 1)
      {
         cyan    = (1.0-red-black)/(1.0-black);
         magenta = (1.0-green-black)/(1.0-black);
         yellow  = (1.0-blue-black)/(1.0-black);
      }

      try
      {
         return new JDRColorCMYK(cyan,magenta,yellow,black,alpha);
      }
      catch (InvalidFormatException e)
      {
         //this shouldn't happen
      }

      return new JDRColorCMYK();
   }

   public JDRColorHSB getJDRColorHSB()
   {
      double max = Math.max(Math.max(red, green), blue);
      double min = Math.min(Math.min(red, green), blue);

      double hue=0, saturation, brightness;

      if (max == min)
      {
         hue = 0;
      }
      else if (max == red && green >= blue)
      {
         hue = 60*(green-blue)/(max-min);
      }
      else if (max == red && green < blue)
      {
         hue = 60*(green-blue)/(max-min) + 360;
      }
      else if (max == green)
      {
         hue = 60*(blue-red)/(max-min) + 120;
      }
      else if (max == blue)
      {
         hue = 60*(red-green)/(max-min) + 240;
      }

      saturation = (max == 0 ? 0 : 1- min/max);

      brightness = max;

      try
      {
         return new JDRColorHSB(hue, saturation, brightness);
      }
      catch (InvalidFormatException e)
      {
         // this shouldn't happen
      }

      return new JDRColorHSB();
   }

   public Paint getPaint(BBox box)
   {
      return getColor();
   }

   public Color getColor()
   {
      return new Color((float)red,(float)green,(float)blue,(float)alpha);
   }

   public String toString()
   {
      return new String("JDRColor@"+"R:" +red+"G:" +green+"B:"
                        +blue+"A:"+alpha);
   }

   public Object clone()
   {
      JDRColor c = null;

      try
      {
         c = new JDRColor(red,green,blue,alpha);
      }
      catch (InvalidFormatException ignore)
      {
      }

      return c;
   }

   public String pgf(BBox box)
   {
      return "\\color[rgb]{"+PGF.format(red)+","+PGF.format(green)
         +","+PGF.format(blue)+"}";
   }

   public String pgffillcolor(BBox box)
   {
      if (alpha == 1.0)
      {
         return  "\\color[rgb]{"+PGF.format(red)+","
                 +PGF.format(green)+","+PGF.format(blue)+"}";
      }
      else
      {
         return  "\\pgfsetfillopacity{"+PGF.format(alpha)+"}"+
                 "\\color[rgb]{"+PGF.format(red)+","
                 +PGF.format(green)+","+PGF.format(blue)+"}";
      }
   }

   public String pgfstrokecolor(BBox box)
   {
      if (alpha == 1.0)
      {
         return  "\\color[rgb]{"+red+","+green+","+blue+"}";
      }
      else
      {
         return  "\\pgfsetstrokeopacity{"+alpha+"}"+
                 "\\color[rgb]{"+red+","+green+","+blue+"}";
      }
   }

   public void saveEPS(PrintWriter out, BBox box)
      throws IOException
   {
      out.println(""+red+" "+green+" "+blue+" setrgbcolor");
   }

   public int psLevel()
   {
      return 1;
   }

   public double getAlpha()
   {
      return alpha;
   }

   /**
    * Gets the red component. This will be a value in the range
    * 0 to 1.
    * @return the red component of this colour
    */
   public double getRed()
   {
      return red;
   }

   /**
    * Gets the blue component. This will be a value in the range
    * 0 to 1.
    * @return the blue component of this colour
    */
   public double getBlue()
   {
      return blue;
   }

   /**
    * Gets the green component. This will be a value in the range
    * 0 to 1.
    * @return the green component of this colour
    */
   public double getGreen()
   {
      return green;
   }

   /**
    * Sets the red component for this colour. This value must be
    * in the range 0 to 1.
    * @param r the red component
    * @throws InvalidRedValueException if r &lt; 0 or r &gt; 1
    */
   public void setRed(double r)
      throws InvalidRedValueException
   {
      if (r < 0.0 || r > 1.0)
      {
         throw new InvalidRedValueException(r);
      }

      red = r;
   }

   /**
    * Sets the green component for this colour. This value must be
    * in the range 0 to 1.
    * @param g the green component
    * @throws InvalidGreenValueException if g &lt; 0 or g &gt; 1
    */
   public void setGreen(double g)
      throws InvalidGreenValueException
   {
      if (g < 0.0 || g > 1.0)
      {
         throw new InvalidGreenValueException(g);
      }

      green = g;
   }

   /**
    * Sets the blue component for this colour. This value must be
    * in the range 0 to 1.
    * @param b the blue component
    * @throws InvalidBlueValueException if b &lt; 0 or b &gt; 1
    */
   public void setBlue(double b)
      throws InvalidBlueValueException
   {
      if (b < 0.0 || b > 1.0)
      {
         throw new InvalidBlueValueException(b);
      }

      blue = b;
   }

   /**
    * Sets the alpha component for this colour. This value must be
    * in the range 0 to 1.
    * @param a the alpha component
    * @throws InvalidAlphaValueException if a &lt; 0 or a &gt; 1
    */
   public void setAlpha(double a)
      throws InvalidAlphaValueException
   {
      if (a < 0.0 || a > 1.0)
      {
         throw new InvalidAlphaValueException(a);
      }

      alpha = a;
   }

   public String svg()
   {
      return "rgb("+(100*red)+"%," +(100*green)+"%,"+(100*blue)+"%)";
   }

   public String svgFill()
   {
      return "fill=\""+svg()+"\" fill-opacity=\""
      +getAlpha()+"\"";
   }

   public String svgLine()
   {
      return "stroke=\""+svg()+"\" stroke-opacity=\""
      +getAlpha()+"\"";
   }

   public String getID()
   {
      Color c = getColor();

      return Integer.toHexString(c.getRed())
           + "." + Integer.toHexString(c.getGreen())
           + "." + Integer.toHexString(c.getBlue())
           + "." + Integer.toHexString(c.getAlpha());
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;

      if (obj == null)
      {
         return false;
      }

      if (!(obj instanceof JDRColor))
      {
         return false;
      }

      JDRColor c = (JDRColor)obj;

      return (getRed() == c.getRed()
           && getGreen() == c.getGreen()
           && getBlue() == c.getBlue()
           && getAlpha() == c.getAlpha());
   }

   public JDRPaintLoaderListener getListener()
   {
      return listener;
   }

   public void fade(double value)
   {
      if (value < 0)
      {
         throw new IllegalArgumentException
            ("illegal fade factor "+value+". (Negative factors not permitted)");
      }

      alpha *= value;

      if (alpha > 1.0)
      {
         alpha = 1.0;
      }
   }

   private double red, green, blue, alpha;

   private static JDRColorListener listener = new JDRColorListener();
}
