// File          : JDRBitmapListener.java
// Date          : 29th February 2008
// Last Modified : 29th February 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Loader listener for bitmaps.
 * @author Nicola L C Talbot
 */

public class JDRBitmapListener implements JDRObjectLoaderListener
{
   public char getJDRid(float version)
   {
      return 'I';
   }

   public void writeJDR(JDRObject object,
                    DataOutputStream dout,
                    float version)
   throws IOException,InvalidFormatException
   {
      JDRBitmap bitmap = (JDRBitmap)object;

      // filename
      String filename = bitmap.getFilename();

      dout.writeInt(filename.length());
      dout.writeChars(filename);

      // LaTeX flag
      dout.writeBoolean(true);
      // LaTeX link to file
      String latexlinkname = bitmap.getLaTeXLinkName();

      dout.writeInt(latexlinkname.length());
      dout.writeChars(latexlinkname);

      // LaTeX image command
      dout.writeInt(bitmap.latexCommand.length());
      dout.writeChars(bitmap.latexCommand);

      // affine transformation matrix
      double[] flatmatrix = new double[6];
      bitmap.getTransformation(flatmatrix);

      for (int i = 0; i < 6; i++)
      {
         dout.writeDouble(flatmatrix[i]);
      }
   }

   public JDRObject readJDR(DataInputStream din, float version)
   throws IOException,InvalidFormatException
   {
      int n = din.readInt();
      if (n <= 0)
      {
         throw new InvalidFilenameLengthException(n);
      }

      char[] name = new char[n];

      for (int i = 0; i < n; i++)
      {
         name[i] = din.readChar();
      }

      String filename = new String(name);

      String latexfilename = null;
      String imageCmd = "\\pgfimage";

      if (din.readBoolean())
      {
         n = din.readInt();
         if (n <= 0)
         {
            throw new InvalidLatexFilenameLengthException(n);
         }

         char[] latexname = new char[n];

         for (int i = 0; i < n; i++)
         {
            latexname[i] = din.readChar();
         }

         latexfilename = new String(latexname);

         n = din.readInt();

         if (n < 0)
         {
            throw new InvalidLatexImagCmdLengthException(n);
         }

         if (n > 0)
         {
            char[] cmd = new char[n];
            for (int i = 0; i < n; i++)
            {
               cmd[i] = din.readChar();
            }
            imageCmd = new String(cmd);
         }
      }

      // affine transformation matrix
      double[] matrix = new double[6];

      for (int i = 0; i < 6; i++)
      {
         matrix[i] = din.readDouble();
      }

      // check filename exists

      String oldFilename = filename;

      filename = JDRBitmap.checkFilename(filename);

      if (filename == null)
      {
         return null;
      }

      if (!oldFilename.equals(filename))
      {
         latexfilename = JDRBitmap.getLaTeXPath(filename);
      }

      JDRBitmap bitmap;

      if (latexfilename==null)
      {
         bitmap = new JDRBitmap(filename);
      }
      else
      {
         bitmap = new JDRBitmap(filename, latexfilename);
      }

      bitmap.transform(matrix);
      bitmap.latexCommand = imageCmd;

      return bitmap;
   }

   public char getAJRid(float version)
   {
      return 'I';
   }

   public void writeAJR(JDRObject object,
                        PrintWriter out,
                        float version)
   throws IOException,InvalidFormatException
   {
      JDRBitmap bitmap = (JDRBitmap)object;

      // filename
      String filename = bitmap.getFilename();

      AJR.writeInt(out, filename.length());
      out.println(filename);

      // LaTeX flag
      AJR.writeBoolean(out, true);

      // LaTeX link to file
      String latexlinkname = bitmap.getLaTeXLinkName();

      AJR.writeInt(out, latexlinkname.length());
      out.println(latexlinkname);

      // LaTeX image command
      AJR.writeInt(out, bitmap.latexCommand.length());
      out.println(bitmap.latexCommand);

      // affine transformation matrix
      double[] flatmatrix = new double[6];
      bitmap.getTransformation(flatmatrix);

      for (int i = 0; i < 6; i++)
      {
         AJR.writeDouble(out, flatmatrix[i]);
      }
   }

   public JDRObject readAJR(BufferedReader in, float version)
   throws IOException,InvalidFormatException,
          java.nio.BufferOverflowException,
          EOFException
   {
      int n = AJR.readInt(in);

      if (n <= 0)
      {
         throw new InvalidFilenameLengthException(n);
      }

      String filename = AJR.readString(in, n);

      String latexfilename = null;
      String imageCmd = "\\pgfimage";

      if (AJR.readBoolean(in))
      {
         n = AJR.readInt(in);

         if (n <= 0)
         {
            throw new InvalidLatexFilenameLengthException(n);
         }

         latexfilename = AJR.readString(in, n);

         n = AJR.readInt(in);

         if (n > 0)
         {
            imageCmd = AJR.readString(in, n);
         }
         else if (n < 0)
         {
            throw new InvalidLatexImagCmdLengthException(n);
         }
      }

      // affine transformation matrix
      double[] matrix = new double[6];

      for (int i = 0; i < 6; i++)
      {
         matrix[i] = AJR.readDouble(in);
      }

      // check filename exists

      filename = JDRBitmap.checkFilename(filename);

      if (filename == null)
      {
         return null;
      }

      JDRBitmap bitmap;

      if (latexfilename==null)
      {
         bitmap = new JDRBitmap(filename);
      }
      else
      {
         bitmap = new JDRBitmap(filename, latexfilename);
      }

      bitmap.transform(matrix);
      bitmap.latexCommand = imageCmd;

      return bitmap;
   }
}
