// File          : JDRBezier.java
// Date          : 1st February 2006
// Last Modified : 25th July 2010
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing a cubic B&eacute;zier curve. A cubic 
 * B&eacute;zier curve is a parametric curve <b>P</b>(<i>t</i>) 
 * described by four points:
 * the start point <b>P</b>(0) = <b>p</b><sub>0</sub>, 
 * the first curvature control point <b>c</b><sub>1</sub>,
 * the second curvature control point <b>c</b><sub>2</sub> and the end
 * point <b>P</b>(1) = <b>p</b><sub>1</sub>.
 *<p>
 * The parametric form is given by:
 * <br>
 * <b>P</b>(<i>t</i>) = <i>J</i><sub>0</sub> <b>p</b><sub>0</sub>
 * + <i>J</i><sub>1</sub> <b>c</b><sub>1</sub>
 * + <i>J</i><sub>2</sub> <b>c</b><sub>2</sub>
 * + <i>J</i><sub>3</sub> <b>p</b><sub>1</sub>
 * <br>
 * where
 * <br>
 * <i>J</i><sub>0</sub> = (1-<i>t</i>)<sup>3</sup><br>
 * <i>J</i><sub>1</sub> = 3<i>t</i>(1-<i>t</i>)<sup>2</sup><br>
 * <i>J</i><sub>2</sub> = 3<i>t</i><sup>2</sup>(1-<i>t</i>)<br>
 * <i>J</i><sub>3</sub> = <i>t</i><sup>3</sup><br>
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2>
 * <img src="images/bezier.png" alt="[annoted image of Bezier curve]">
 * </td></tr>
 * <tr><th valign=top>Figure&nbsp;1</th>
 * <td>A cubic B&eacute;zier curve is described by four points</td></tr>
 * </table>
 * </center>
 * @author Nicola L C Talbot
 */
public class JDRBezier extends JDRSegment
{
   /**
    * Creates a straight line B&eacute;zier segment.
    * The control points are set using {@link #flatten()}.
    * @param p0 the starting point
    * @param p1 the end point
    */
   public JDRBezier(Point p0, Point p1)
   {
      super(p0, p1);

      control1 = new JDRPoint();
      control2 = new JDRPoint();

      flatten();
   }

   /**
    * Creates a straight line B&eacute;zier segment.
    * The control points are set using {@link #flatten()}.
    * @param p0 the starting point
    * @param p1 the end point
    */
   public JDRBezier(Point2D p0, Point2D p1)
   {
      super(p0, p1);

      control1 = new JDRPoint();
      control2 = new JDRPoint();

      flatten();
   }

   /**
    * Creates a straight line B&eacute;zier segment.
    * The control points are set using {@link #flatten()}.
    * @param p0 the starting point
    * @param p1 the end point
    */
   public JDRBezier(JDRPoint p0, JDRPoint p1)
   {
      super(p0, p1);

      control1 = new JDRPoint();
      control2 = new JDRPoint();

      flatten();
   }

   /**
    * Creates a B&eacute;zier segment.
    * @param p0 the starting point
    * @param c1 the first control point
    * @param c2 the second control point
    * @param p1 the end point
    */
   public JDRBezier(Point p0, Point c1, Point c2, Point p1)
   {
      super(p0, p1);
      control1 = new JDRPoint(c1);
      control2 = new JDRPoint(c2);
   }

   /**
    * Creates a B&eacute;zier segment.
    * @param p0 the starting point
    * @param c1 the first control point
    * @param c2 the second control point
    * @param p1 the end point
    */
   public JDRBezier(Point2D p0, Point2D c1, Point2D c2, Point2D p1)
   {
      super(p0, p1);
      control1 = new JDRPoint(c1);
      control2 = new JDRPoint(c2);
   }

   /**
    * Creates a B&eacute;zier segment.
    * @param p0 the starting point
    * @param c1 the first control point
    * @param c2 the second control point
    * @param p1 the end point
    */
   public JDRBezier(JDRPoint p0, JDRPoint c1, JDRPoint c2, JDRPoint p1)
   {
      super(p0, p1);
      control1 = c1;
      control2 = c2;
   }

   /**
    * Creates a B&eacute;zier segment.
    * @param p0x the x co-ordinate of the starting point
    * @param p0y the y co-ordinate of the starting point
    * @param c1x the x co-ordinate of first control point
    * @param c1y the y co-ordinate of first control point
    * @param c2x the x co-ordinate of the second control point
    * @param c2y the y co-ordinate of the second control point
    * @param p1x the x co-ordinate of the end point
    * @param p1y the y co-ordinate of the end point
    */
   public JDRBezier(double p0x, double p0y,
                 double c1x, double c1y,
                 double c2x, double c2y,
                 double p1x, double p1y)
   {
      super(new JDRPoint(p0x, p0y), new JDRPoint(p1x, p1y));

      control1 = new JDRPoint(c1x, c1y);
      control2 = new JDRPoint(c2x, c2y);
   }

   /**
    * Shifts the control points of this segment to make it straight.
    */
   public void flatten()
   {
      Point2D dP = ((JDRSegment)this).getdP();
      setGradients(dP, dP);
   }

   /**
    * Reverses the direction of this segment.
    */
   public JDRBezier reverse()
   {
      JDRBezier curve = (JDRBezier)clone();

      curve.start.x = end.x;
      curve.start.y = end.y;
      curve.control1.x = control2.x;
      curve.control1.y = control2.y;
      curve.control2.x = control1.x;
      curve.control2.y = control1.y;
      curve.end.x = start.x;
      curve.end.y = start.y;

      return curve;
   }

   /**
    * Inverts this segment.
    */
   public JDRBezier invert()
   {
      double p1x = 2*start.x-end.x;
      double p1y = 2*start.y-end.y;
      double c2x = 2*start.x-control2.x;
      double c2y = 2*start.y-control2.y;
      double c1x = 2*start.x-control1.x;
      double c1y = 2*start.y-control1.y;

      return new JDRBezier(start.x,start.y, c1x, c1y, c2x, c2y, p1x, p1y);
   }

   /**
    * Creates a cubic B&eacute;zier segment from a quadratic 
    * B&eacute;zier.
    * @param b0x the x co-ordinate of the starting point
    * @param b0y the y co-ordinate of the starting point
    * @param b1x the x co-ordinate of the quadratic control point
    * @param b1y the y co-ordinate of the quadratic control point
    * @param b2x the x co-ordinate of the end point
    * @param b2y the y co-ordinate of the end point
    * @return the cubic equivalent of the quadratic B&eacute;zier curve
    */
   public static JDRBezier quadToCubic(double b0x, double b0y,
                             double b1x, double b1y,
                             double b2x, double b2y)
   {
      Point2D c0 = new Point2D.Double(b0x, b0y);

      Point2D c1 = new Point2D.Double((2*b1x+b0x)/3,(2*b1y+b0y)/3);
      Point2D c2 = new Point2D.Double((b2x+2*b1x)/3,(b2y+2*b1y)/3);

      Point2D c3 = new Point2D.Double(b2x, b2y);

      return new JDRBezier(c0,c1,c2,c3);
   }

   /**
    * Sets the points defining this B&eacute;zier curve.
    * @param p0 the starting point
    * @param c1 the first control point
    * @param c2 the second control point
    * @param p1 the end point
    */
   public void set(Point p0, Point c1, Point c2, Point p1)
   {
      start.x = p0.x;
      start.y = p0.y;
      control1.x = c1.x;
      control1.y = c1.y;
      control2.x = c2.x;
      control2.y = c2.y;
      end.x = p1.x;
      end.y = p1.y;
   }

   /**
    * Sets the points defining this B&eacute;zier curve.
    * @param p0 the starting point
    * @param c1 the first control point
    * @param c2 the second control point
    * @param p1 the end point
    */
   public void set(Point2D p0, Point2D c1, Point2D c2, Point2D p1)
   {
      start.x = p0.getX();
      start.y = p0.getY();
      control1.x = c1.getX();
      control1.y = c1.getY();
      control2.x = c2.getX();
      control2.y = c2.getY();
      end.x = p1.getX();
      end.y = p1.getY();
   }

   /**
    * Sets the start and end gradients of this segment.
    * Moves the control points of this segment so that gradients at the
    * start and end points are as specified.
    * @param dP0 the required gradient at the start of the curve
    * @param dP1 the required gradient at the end of the curve
    */
   public void setGradients(Point2D dP0, Point2D dP1)
   {
      control1.set(dP0.getX()/3+start.x,
                   dP0.getY()/3+start.y);
      control2.set(end.x-dP1.getX()/3,
                   end.y-dP1.getY()/3);
   }

   /**
    * Sets the start gradient of this segment.
    * Moves the control points of this segment so that gradient at the
    * start point is as specified.
    * @param dP0 the required gradient at the start of the curve
    */
   public void setStartGradient(JDRPoint dP0)
   {
      setStartGradient(dP0.x, dP0.y);
   }

   /**
    * Sets the start gradient of this segment.
    * Moves the control points of this segment so that gradient at the
    * start point is as specified.
    * @param dP0 the required gradient at the start of the curve
    */
   public void setStartGradient(Point2D dP0)
   {
      setStartGradient(dP0.getX(), dP0.getY());
   }

   /**
    * Sets the start gradient of this segment.
    * Moves the control points of this segment so that gradient at the
    * start point is as specified.
    * @param dP0x the x co-ordinate of the required gradient
    * @param dP0y the y co-ordinate of the required gradient
    */
   public void setStartGradient(double dP0x, double dP0y)
   {
      control1.x = dP0x/3+start.x;
      control1.y = dP0y/3+start.y;
   }

   /**
    * Sets the end gradient of this segment.
    * Moves the control points of this segment so that gradient at the
    * end point is as specified.
    * @param dP1 the required gradient at the end of the curve
    */
   public void setEndGradient(JDRPoint dP1)
   {
      setEndGradient(dP1.x, dP1.y);
   }

   /**
    * Sets the end gradient of this segment.
    * Moves the control points of this segment so that gradient at the
    * end point is as specified.
    * @param dP1 the required gradient at the end of the curve
    */
   public void setEndGradient(Point2D dP1)
   {
      setEndGradient(dP1.getX(), dP1.getY());
   }

   /**
    * Sets the end gradient of this segment.
    * Moves the control points of this segment so that gradient at the
    * end point is as specified.
    * @param dP1x the x co-ordinate of the required gradient
    * @param dP1y the y co-ordinate of the required gradient
    */
   public void setEndGradient(double dP1x, double dP1y)
   {
      control2.x = end.x-dP1x/3;
      control2.y = end.y-dP1y/3;
   }

   /**
    * Gets the gradient at the start of this segment.
    * @return the gradient at t=0
    * @see #getdP1()
    */
   public Point2D getdP0()
   {
      return new Point2D.Double(3*(control1.x-start.x), 
                                3*(control1.y-start.y));
   }

   /**
    * Gets the gradient at the end of this segment.
    * @return the gradient at t=1
    * @see #getdP0()
    */
   public Point2D getdP1()
   {
      return new Point2D.Double(3*(end.x-control2.x), 
                                3*(end.y-control2.y));
   }

   /**
    * Gets the point at t along this segment.
    * @return the required point
    */
   public Point2D getP(double t)
   {
      double one_minus_t = (1-t);
      double one_minus_t_sq = one_minus_t*one_minus_t;
      double t_sq = t*t;
      double J0 = one_minus_t_sq*one_minus_t;
      double J1 = 3*t*one_minus_t_sq;
      double J2 = 3*t_sq*one_minus_t;
      double J3 = t_sq*t;

      double x = J0*start.x + J1*control1.x + J2*control2.x + J3*end.x;
      double y = J0*start.y + J1*control1.y + J2*control2.y + J3*end.y;

      return new Point2D.Double(x, y);
   }

   /**
    * Gets the gradient at t for this segment.
    * @return the required gradient
    */
   public Point2D getdP(double t)
   {
      double dJ0 = - 3*(1-t)*(1-t);
      double dJ1 = 3*(1-t)*(1-3*t);
      double dJ2 = 3*t*(2-3*t);
      double dJ3 = 3*t*t;

      double x = dJ0*start.x+dJ1*control1.x+dJ2*control2.x+dJ3*end.x;
      double y = dJ0*start.y+dJ1*control1.y+dJ2*control2.y+dJ3*end.y;

      return new Point2D.Double(x,y);
   }

   /**
    * Gets the value of t for which p(t) is the closest point to 
    * (x0,y0) using t0 as starting value.
    * @param t0 starting value of t
    * @param x0 x co-ordinate of target point
    * @param y0 y co-ordinate of target point
    * @return value of t for which p(t) is closest to (x0,y0)
    */
   public double getT(double t0, double x0, double y0)
   {
      double t = t0;
      double oldE = 100000;

      for (int k = 0; k < 100; k++)
      {
         Point2D pt = getP(t);

         double r0 = pt.getX() - x0;
         double r1 = pt.getY() - y0;

         double E = r0*r0 + r1*r1;

         if (Math.abs(E-oldE) < 1e-6) break;

         Point2D dP = getdP(t);

         double g = 2*dP.getX()*r0 + 2*dP.getY()*r1;
         double h = 2*(dP.getX()*dP.getX() + dP.getY()*dP.getY());

         double deltaT = -g/h;

         t += deltaT;
      }

      return t;
   }

   /**
    * Gets the value of t for which p(t) is the closest point to 
    * (x0,y0) and p'(t) is closest to the given gradient.
    * @param x x co-ordinate of target point
    * @param y y co-ordinate of target point
    * @param dx x co-ordinate of target gradient
    * @param dy y co-ordinate of target gradient
    * @param tinc increment of t
    * @return value of t for which p(t) is closest to (x,y) and
    * p'(t) is closest to (dx,dy)
    */
   public double getT(double x, double y, double dx, double dy, double tinc)
   {
      double tval = 0;

      double oldDiffP = Double.MAX_VALUE;
      double oldDiffG = Double.MAX_VALUE;

      for (double t = 0; t < 1.0; t += tinc)
      {
         Point2D pt = getP(t);

         double r0 = pt.getX() - x;
         double r1 = pt.getY() - y;

         double diffP = r0*r0 + r1*r1;

         Point2D dp = getdP(t);

         r0 = dp.getX() - dx;
         r1 = dp.getY() - dy;

         double diffG = r0*r0 + r1*r1;

         if ((diffP < oldDiffP) && (diffG < oldDiffG))
         {
            oldDiffP = diffP;
            oldDiffG = diffG;
            tval = t;
         }
      }

      return tval;
   }

   /**
    * Transforms this curve. The transformation matrix should be
    * stored as the flat matrix [m00, m10, m01, m11, m02, m12].
    * @param matrix the transformation matrix
    */
   public void transform(double[] matrix)
   {
      start.transform(matrix);
      end.transform(matrix);
      control1.transform(matrix);
      control2.transform(matrix);
   }

   /**
    * Translates this curve. Translates the start, end and control
    * points defining this curve by (x,y).
    * @param x the x-shift
    * @param y the y-shift
    */
   public void translate(double x, double y)
   {
      start.translate(x,y);
      end.translate(x,y);
      control1.translate(x,y);
      control2.translate(x,y);
   }

   public void translate(double x, double y, boolean endPoint)
   {
      start.translate(x,y);
      control1.translate(x,y);
      control2.translate(x,y);
      if (endPoint) end.translate(x,y);
   }


   /**
    * Scales this curve.
    * @param factorX the x scale factor
    * @param factorY the y scale factor
    */
   public void scale(double factorX, double factorY)
   {
      start.scale(factorX, factorY);
      end.scale(factorX, factorY);
      control1.scale(factorX, factorY);
      control2.scale(factorX, factorY);
   }

   /**
    * Scales this curve about a given point. The end point will 
    * only be scaled if endPoint is true.
    * @param endPoint if true scale end point as well
    */
   public void scale(Point2D p, double factorX, double factorY,
                     boolean endPoint)
   {
      start.scale(p, factorX, factorY);
      control1.scale(p, factorX, factorY);
      control2.scale(p, factorX, factorY);
      if (endPoint) end.scale(p, factorX, factorY);
   }

   /**
    * Scales this curve along the x-axis.
    * @param factor the x scale factor
    */
   public void scaleX(double factor)
   {
      start.scaleX(factor);
      end.scaleX(factor);
      control1.scaleX(factor);
      control2.scaleX(factor);
   }

   /**
    * Scales this curve along the y-axis.
    * @param factor the y scale factor
    */
   public void scaleY(double factor)
   {
      start.scaleY(factor);
      end.scaleY(factor);
      control1.scaleY(factor);
      control2.scaleY(factor);
   }

   /**
    * Shears this curve.
    * @param factorX the x shear factor
    * @param factorY the y shear factor
    */
   public void shear(double factorX, double factorY)
   {
      start.shear(factorX, factorY);
      end.shear(factorX, factorY);
      control1.shear(factorX, factorY);
      control2.shear(factorX, factorY);
   }

   /**
    * Shears curve about a given point. The end point will only 
    * be sheared if endPoint is true.
    * @param endPoint if true shear end point as well
    */
   public void shear(Point2D p, double factorX, double factorY,
                     boolean endPoint)
   {
      start.shear(p, factorX, factorY);
      control1.shear(p, factorX, factorY);
      control2.shear(p, factorX, factorY);
      if (endPoint) end.shear(p, factorX, factorY);
   }

   /**
    * Rotates this curve.
    * @param angle the angle of rotation
    */
   public void rotate(double angle)
   {
      start.rotate(angle);
      end.rotate(angle);
      control1.rotate(angle);
      control2.rotate(angle);
   }

   /**
    * Rotates curve about a given point. The end point will only 
    * be rotated if endPoint is true.
    * @param p the point about which to rotate
    * @param angle the angle of rotation
    * @param endPoint if true rotate end point as well
    */
   public void rotate(Point2D p, double angle,
                     boolean endPoint)
   {
      start.rotate(p, angle);
      control1.rotate(p, angle);
      control2.rotate(p, angle);
      if (endPoint) end.rotate(p, angle);
   }

   /**
    * Draws the points defining this curve.
    * @param g the graphics device on which to draw
    * @param endPoint determines whether the end point should be drawn
    */
   public void drawControls(Graphics g, boolean endPoint)
   {
      Graphics2D g2 = (Graphics2D)g;

      Stroke oldStroke = g2.getStroke();
      Paint oldPaint = g2.getPaint();

      g2.setStroke(guideStroke);
      //g2.setPaint(draftColor.getColor());
      g2.setPaint(draftColor);
      g.drawLine((int)start.x, (int)start.y, 
                 (int)control1.x, (int)control1.y);
      g.drawLine((int)control1.x, (int)control1.y, 
                 (int)control2.x, (int)control2.y);
      g.drawLine((int)control2.x, (int)control2.y,
                 (int)end.x, (int)end.y);
      g2.setStroke(oldStroke);
      g2.setPaint(oldPaint);

      start.draw(g);
      control1.draw(g);
      control2.draw(g);
      if (endPoint) end.draw(g);
   }

   public void drawControls(Graphics g, boolean endPoint, double scale)
   {
      Graphics2D g2 = (Graphics2D)g;

      Stroke oldStroke = g2.getStroke();
      Paint oldPaint = g2.getPaint();

      g2.setStroke(guideStroke);
      //g2.setPaint(draftColor.getColor());
      g2.setPaint(draftColor);
      GeneralPath path = new GeneralPath();
      path.moveTo((float)(start.x*scale), (float)(start.y*scale));
      path.lineTo((float)(control1.x*scale), (float)(control1.y*scale));
      path.lineTo((float)(control2.x*scale), (float)(control2.y*scale));
      path.lineTo((float)(end.x*scale), (float)(end.y*scale));

      g2.draw(path);

      g2.setStroke(oldStroke);
      g2.setPaint(oldPaint);

      start.draw(g, scale);
      control1.draw(g, scale);
      control2.draw(g, scale);
      if (endPoint) end.draw(g, scale);
   }

   /**
    * Draws this curve.
    * @param g the graphics device on which to draw
    */
   public void draw(Graphics g)
   {
      Graphics2D g2 = (Graphics2D) g;

      g2.draw(new CubicCurve2D.Double(start.x, start.y,
                                      control1.x, control1.y,
                                      control2.x, control2.y,
                                      end.x, end.y));
   }

   public void draw(Graphics g, double scale)
   {
      Graphics2D g2 = (Graphics2D) g;

      g2.draw(new CubicCurve2D.Double(scale*start.x, scale*start.y,
                                      scale*control1.x, scale*control1.y,
                                      scale*control2.x, scale*control2.y,
                                      scale*end.x, scale*end.y));
   }

   public void drawSelectedNoControls(Graphics g, double scale)
   {
      Graphics2D g2 = (Graphics2D)g;
      Stroke oldStroke = g2.getStroke();
      Paint oldPaint = g2.getPaint();

      g2.setPaint(start.getSelectedPaint());
      g2.setStroke(guideStroke);

      draw(g, scale);

      g2.setPaint(oldPaint);
      g2.setStroke(oldStroke);
   }

   public void drawDraft(Graphics g, double scale, boolean drawEnd)
   {
      Graphics2D g2 = (Graphics2D)g;
      Stroke oldStroke = g2.getStroke();

      g2.setPaint(JDRObject.draftColor);

      draw(g, scale);

      // Draw guides
      
      GeneralPath path = new GeneralPath();
      path.moveTo((float)(start.x*scale), (float)(start.y*scale));
      path.lineTo((float)(control1.x*scale), (float)(control1.y*scale));
      path.lineTo((float)(control2.x*scale), (float)(control2.y*scale));
      path.lineTo((float)(end.x*scale), (float)(end.y*scale));

      if (isSelected())
      {
         g2.draw(path);

         g2.setPaint(start.getSelectedPaint());
         g2.setStroke(guideStroke);

         draw(g, scale);
         g2.draw(path);
      }
      else
      {
         g2.setStroke(guideStroke);
         g2.draw(path);
      }

      g2.setStroke(oldStroke);

      start.draw(g, scale);
      control1.draw(g, scale);
      control2.draw(g, scale);

      if (drawEnd) end.draw(g, scale);
   }

   /**
    * Makes adjacent curves continous. 
    * This assumes the end point of first segment coincides with start
    * point of second segment.
    * @param first the first B&eacute;zier curve
    * @param second the second B&eacute;zier curve
   */
   public static void makeContinuous(JDRBezier first, JDRBezier second)
   {
     // second.control1.x = 2*first.end.x - first.control2.x;
     // second.control1.y = 2*first.end.y - first.control2.y;
     first.control2.x = 2*first.end.x - second.control1.x;
     first.control2.y = 2*first.end.y - second.control1.y;
   }

   /**
    * Creates a new B&eacute;zier curve following on from given 
    * segment.
    * The given segment is not necessarily a B&eacute;zier curve.
    * @param previous the given segment
    * @param endP the end point of the new B&eacute;zier curve
    */
   public static JDRBezier constructBezier(JDRSegment previous, Point endP)
   {
      return constructBezier(previous,
                             new Point2D.Double(endP.x,endP.y));
   }

   /**
    * Creates a new B&eacute;zier curve following on from given 
    * segment.
    * The given segment is not necessarily a B&eacute;zier curve.
    * @param previous the given segment
    * @param endP the end point of the new B&eacute;zier curve
    */
   public static JDRBezier constructBezier(JDRSegment previous, Point2D endP)
   {
      JDRBezier curve = new JDRBezier(previous.end, new JDRPoint(endP));

      if (!(previous instanceof JDRBezier))
      {
         Point2D dP = previous.getdP();
         curve.control1 = new JDRPoint(dP.getX()/3+curve.start.x, 
                                    dP.getY()/3+curve.start.y);
         return curve;
      }

      JDRBezier prevCurve = (JDRBezier)previous;

      double c0x = prevCurve.start.x;
      double c0y = prevCurve.start.y;
      double c3x = prevCurve.end.x;
      double c3y = prevCurve.end.y;
      double d3x = endP.getX();
      double d3y = endP.getY();

      double b0x = (c0x-c3x)/3;
      double b0y = (c0y-c3y)/3;
      double b1x = (d3x+5*c3x)/3;
      double b1y = (d3y+5*c3y)/3;
      double b2x = -4*c3x;
      double b2y = -4*c3y;

      double d2x = (3*b1x-b0x+b2x)/2;
      double d2y = (3*b1y-b0y+b2y)/2;

      double c2x = b1x-d2x;
      double c2y = b1y-d2y;

      double c1x = b0x+c2x;
      double c1y = b0y+c2y;

      double d1x = 2*c3x-c2x;
      double d1y = 2*c3y-c2y;

      prevCurve.control1.x = c1x;
      prevCurve.control1.y = c1y;

      prevCurve.control2.x = c2x;
      prevCurve.control2.y = c2y;

      curve.control1.x = d1x;
      curve.control1.y = d1y;

      curve.control2.x = d2x;
      curve.control2.y = d2y;

      return curve;
   }

   public JDRPathSegment split()
   {
      double b0x = start.x;
      double b0y = start.y;
      double b1x = control1.x;
      double b1y = control1.y;
      double b2x = control2.x;
      double b2y = control2.y;
      double b3x = end.x;
      double b3y = end.y;

      JDRPoint midPt = new JDRPoint((b3x+3*b2x+3*b1x+b0x)/8,
                                   (b3y+3*b2y+3*b1y+b0y)/8);

      JDRPoint c1 = new JDRPoint(0.5*(b1x+b0x), 0.5*(b1y+b0y));
      JDRPoint c2 = new JDRPoint(0.25*(b2x+2*b1x+b0x),
                                 0.25*(b2y+2*b1y+b0y));

      JDRPoint d1 = new JDRPoint(0.25*(b3x+2*b2x+b1x),
                                 0.25*(b3y+2*b2y+b1y));
      JDRPoint d2 = new JDRPoint(0.5*(b3x+b2x),
                                 0.5*(b3y+b2y));

      JDRBezier newSegment = new JDRBezier(midPt, d1, d2, end);

      control1 = c1;
      control2 = c2;
      end = midPt;

      return newSegment;
   }

   /**
    * Creates a copy of this object.
    * @return copy of this object
    */
   public Object clone()
   {
      JDRBezier curve = new JDRBezier((JDRPoint)start.clone(), 
                                (JDRPoint)control1.clone(), 
                                (JDRPoint)control2.clone(), 
                                (JDRPoint)end.clone());
      return curve;
   }

   /**
    * Makes this curve equal to another curve.
    * @param curve the other curve
    */
   public void makeEqual(JDRBezier curve)
   {
      super.makeEqual((JDRSegment)curve);
      control1.makeEqual(curve.control1);
      control2.makeEqual(curve.control2);
   }

   public boolean equals(Object obj)
   {
      if (!super.equals(obj)) return false;

      if (!(obj instanceof JDRSegment)) return false;

      JDRBezier c = (JDRBezier)obj;

      // super.equals(obj) has already checked start and end

      return (control1.equals(c.control1)
           && control2.equals(c.control2));
   }

   /**
    * Sets the flag indicating whether the control points are 
    * being edited.
    * @param flag indicates whether control points are being edited
    */
   public void setEditedControls(boolean flag)
   {
      super.setEditedControls(flag);
      control1.selected = flag;
      control2.selected = flag;
   }

   /**
    * Returns control point that is currently being edited. If no
    * control points are being edited, null is return.
    * @return edited control point or null
    */
   public JDRPoint getEditedControl()
   {
      if (start.selected) return start;
      if (control1.selected) return control1;
      if (control2.selected) return control2;
      if (end.selected) return end;
      return null;
   }

   /**
    * Gets the control point in which given point is inside.
    * If more than one control point contains p, the control point
    * is chosen in the following order: start point, first 
    * curvature control, second curvature control. 
    * The dimensions of the control points are given
    * by {@link JDRPoint#pointSize}.
    * @param p the given point
    * @param endPoint if true include the end point in the search
    * otherwise don't check the end point.
    * @return the control point that contains the given point or
    * <code>null</code> if none of the control points contain that point
    */
   public JDRPoint getControl(Point p, boolean endPoint)
   {
      if (control2.contains(p)) return control2;
      if (control1.contains(p)) return control1;
      if (start.contains(p)) return start;
      if (endPoint && end.contains(p)) return end;

      return null;
   }

   /**
    * Gets the first curvature control point.
    * @return the first curvature control point
    * @see #setControl1(double,double)
    * @see #getControl2()
    */
   public JDRPoint getControl1()
   {
      return control1;
   }

   /**
    * Gets the second curvature control point.
    * @return the second curvature control point
    * @see #setControl2(double,double)
    * @see #getControl1()
    */
   public JDRPoint getControl2()
   {
      return control2;
   }

   /**
    * Sets the first curvature control point.
    * @param newX x co-ordinate
    * @param newY y co-ordinate
    * @see #getControl1()
    * @see #setControl2(double,double)
    */
   public void setControl1(double newX, double newY)
   {
      control1.x = newX;
      control1.y = newY;
   }

   /**
    * Sets the second curvature control point.
    * @param newX x co-ordinate
    * @param newY y co-ordinate
    * @see #getControl2()
    * @see #setControl1(double,double)
    */
   public void setControl2(double newX, double newY)
   {
      control2.x = newX;
      control2.y = newY;
   }

   /**
    * Sets the flag determining whether this curve is being edited.
    * @param flag indicates whether this curve is being edited
    */
   public void setSelected(boolean flag)
   {
      selected = flag;
      if (flag == false)
      {
         start.selected = false;
         control1.selected = false;
         control2.selected = false;
         end.selected = false;
      }
   }

   public void saveSVG(PrintWriter out) throws IOException
   {
      out.print("C ");
      control1.saveSVG(out);
      control2.saveSVG(out);
      end.saveSVG(out);
   }

   public void saveEPS(PrintWriter out) throws IOException
   {
      control1.saveEPS(out);
      control2.saveEPS(out);
      end.saveEPS(out);
      out.println("curveto");
   }

   public String pgf(AffineTransform af)
   {
      return "\\pgfpathcurveto{"
            + control1.pgf(af)
            + "}{"
            + control2.pgf(af)
            + "}{"
            + end.pgf(af)
            +"}";
   }

   public void appendToGeneralPath(GeneralPath path)
   {
      path.curveTo((float)control1.x, (float)control1.y,
                   (float)control2.x, (float)control2.y,
                   (float)end.x, (float)end.y);
   }

   public void appendReflectionToGeneralPath(GeneralPath path, JDRLine line)
   {
      Point2D c1 = control1.getReflection(line);
      Point2D c2 = control2.getReflection(line);
      Point2D p  = start.getReflection(line);

      path.curveTo((float)c2.getX(), (float)c2.getY(),
                   (float)c1.getX(), (float)c1.getY(),
                   (float)p.getX(), (float)p.getY());
   }

   public JDRPathSegment getReflection(JDRLine line)
   {
      Point2D p1 = start.getReflection(line);
      Point2D c1 = control1.getReflection(line);
      Point2D c2 = control2.getReflection(line);
      Point2D p2 = end.getReflection(line);

      return new JDRBezier(p1, c1, c2, p2);
   }

   /**
    * Gets string representation of this object.
    */
   public String toString()
   {
      return "JDRBezier:("+start.x+","+start.y+")("
          +control1.x+","+control1.y+")("
          +control2.x+","+control2.y+")("
          +end.x+","+end.y+"),startMarker="+startMarker+",endMarker="+endMarker;
   }

   public JDRObjectLoaderListener getListener()
   {
      return listener;
   }

   public String info()
   {
      return "bezier["+start.info()
      +","+control1+","+control2+","+end.info()+"]";
   }

   public int controlCount()
   {
      return 3;
   }

   public JDRPoint getControl(int index)
      throws IndexOutOfBoundsException
   {
      if (index == 0) return start;

      if (index == 1) return control1;

      if (index == 2) return control2;

      throw new IndexOutOfBoundsException("No control point at index "+index);
   }

   public int getControlIndex(JDRPoint point)
      throws NoSuchElementException
   {
      if (point == start) return 0;

      if (point == control1) return 1;

      if (point == control2) return 2;

      throw new NoSuchElementException();
   }

   public boolean isGap() {return false;}

   /**
    * The first curvature control point.
    */
   protected JDRPoint control1;
   /**
    * The second curvature control point.
    */
   protected JDRPoint control2;

   private static JDRBezierLoaderListener listener
      = new JDRBezierLoaderListener();
}
