// File          : JDRBasicStroke.java
// Date          : 1st February 2006
// Last Modified : 16th February 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import uk.ac.uea.cmp.nlct.jdr.marker.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing line styles used to draw a path.
 * The line style has the following attributes:
 * <ul>
 * <li> Cap style ({@link BasicStroke#CAP_SQUARE},
 * {@link BasicStroke#CAP_BUTT} or {@link BasicStroke#CAP_ROUND})
 * <li> Join style ({@link BasicStroke#JOIN_MITER},
 * {@link BasicStroke#JOIN_BEVEL} or {@link BasicStroke#JOIN_ROUND})
 * <li> Dash pattern
 * <li> Pen width (must be positive)
 * <li> Mitre limit (must be &gt;= 1)
 * <li> Winding rule ({@link GeneralPath#WIND_EVEN_ODD} or
 * {@link GeneralPath#WIND_NON_ZERO})
 * <li> Start marker
 * <li> End marker
 * <li> Mid-point marker
 * </ul>
 */

public class JDRBasicStroke implements JDRStroke
{
   /**
    * Creates default stroke. The default settings are: square
    * cap, mitre join, solid line, 1bp pen width, mitre limit of
    * 10.0, even-odd winding rule and no markers.
    */
   public JDRBasicStroke()
   {
      capStyle    = BasicStroke.CAP_SQUARE;
      joinStyle   = BasicStroke.JOIN_MITER;
      dashPattern = new DashPattern();
      penWidth    = 1.0;
      mitreLimit  = 10.0;
      windingRule = GeneralPath.WIND_EVEN_ODD;

      try
      {
         startMarker = new JDRMarker(1.0f, 1, false);
         midMarker = new JDRMarker(1.0f, 1, false);
         endMarker   = new JDRMarker(1.0f, 1, false);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }
   }

   /**
    * Creates a stroke with the given settings. The new stroke
    * has no markers and even-odd winding rule.
    * @param penwidth the pen width
    * @param capstyle the cap style
    * @param joinstyle the join style
    * @param mitrelimit the mitre limit
    * @param dashpattern the dash pattern
    * @throws InvalidMitreLimitException if the mitre limit is less
    * than 1
    * @throws InvalidJoinStyleException if the join style is not
    * one of:{@link BasicStroke#JOIN_MITER},
    * {@link BasicStroke#JOIN_BEVEL} or {@link BasicStroke#JOIN_ROUND}
    * @throws InvalidCapStyleException if the cap style is not one of:
    * {@link BasicStroke#CAP_SQUARE}, {@link BasicStroke#CAP_BUTT}
    * or {@link BasicStroke#CAP_ROUND}
    * @throws InvalidPenWidthException if the pen width is less than 0
    */
   public JDRBasicStroke(double penwidth, int capstyle, int joinstyle,
                    double mitrelimit, DashPattern dashpattern)
     throws InvalidMitreLimitException,
            InvalidJoinStyleException,
            InvalidCapStyleException,
            InvalidPenWidthException
   {
      setCapStyle(capstyle);
      setJoinStyle(joinstyle);
      dashPattern = dashpattern;
      setMitreLimit(mitrelimit);
      windingRule = GeneralPath.WIND_EVEN_ODD;

      try
      {
         startMarker = new JDRMarker(1.0f, 1, false);
         midMarker = new JDRMarker(1.0f, 1, false);
         endMarker   = new JDRMarker(1.0f, 1, false);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      setPenWidth(penwidth);
   }

   /**
    * Creates a stroke with the given settings. The new stroke
    * has no markers.
    * @param penwidth the pen width
    * @param capstyle the cap style
    * @param joinstyle the join style
    * @param mitrelimit the mitre limit
    * @param dashpattern the dash pattern
    * @param windingrule the winding rule
    * @throws InvalidMitreLimitException if the mitre limit is less
    * than 1
    * @throws InvalidJoinStyleException if the join style is not
    * one of:{@link BasicStroke#JOIN_MITER},
    * {@link BasicStroke#JOIN_BEVEL} or {@link BasicStroke#JOIN_ROUND}
    * @throws InvalidCapStyleException if the cap style is not one of:
    * {@link BasicStroke#CAP_SQUARE}, {@link BasicStroke#CAP_BUTT}
    * or {@link BasicStroke#CAP_ROUND}
    * @throws InvalidPenWidthException if the pen width is less than 0
    * @throws InvalidWindingRuleException if the winding rule is not
    * one of: {@link GeneralPath#WIND_EVEN_ODD} or
    * {@link GeneralPath#WIND_NON_ZERO}
    */
   public JDRBasicStroke(double penwidth, int capstyle, int joinstyle,
                      double mitrelimit, DashPattern dashpattern,
                      int windingrule)
     throws InvalidMitreLimitException,
            InvalidJoinStyleException,
            InvalidCapStyleException,
            InvalidPenWidthException,
            InvalidWindingRuleException
   {
      setCapStyle(capstyle);
      setJoinStyle(joinstyle);
      dashPattern = dashpattern;
      setMitreLimit(mitrelimit);
      setWindingRule(windingrule);

      try
      {
         startMarker = new JDRMarker(1.0f, 1, false);
         midMarker = new JDRMarker(1.0f, 1, false);
         endMarker   = new JDRMarker(1.0f, 1, false);
      }
      catch (InvalidRepeatValueException ignore)
      {
      }

      setPenWidth(penwidth);
   }

   /**
    * Sets the pen width for this stroke.
    * @param width the pen width
    * @throws InvalidPenWidthException if the pen width is less than 0
    */
   public void setPenWidth(double width)
      throws InvalidPenWidthException
   {
      if (width < 0)
      {
         throw new InvalidPenWidthException(width);
      }

      penWidth = width;
      startMarker.setPenWidth(width);
      midMarker.setPenWidth(width);
      endMarker.setPenWidth(width);
   }

   /**
    * Gets the pen width for this stroke.
    * @return the pen width for this stroke
    */
   public double getPenWidth()
   {
      return penWidth;
   }

   public void fade(double factor)
   {
      startMarker.fade(factor);
      midMarker.fade(factor);
      endMarker.fade(factor);
   }

   /**
    * Gets the colour of the start marker for this stroke.
    * @return the start marker paint or null if the marker paint
    * should be the same as the path
    * @see #getMidArrowColour()
    * @see #getEndArrowColour()
    * @see #setStartArrowColour(JDRPaint)
    * @see #setMidArrowColour(JDRPaint)
    * @see #setEndArrowColour(JDRPaint)
    */
   public JDRPaint getStartArrowColour()
   {
      return startMarker.fillPaint;
   }

   /**
    * Gets the colour of the mid markers for this stroke.
    * @return the mid marker paint or null if the marker paint
    * should be the same as the path
    * @see #getStartArrowColour()
    * @see #getEndArrowColour()
    * @see #setStartArrowColour(JDRPaint)
    * @see #setMidArrowColour(JDRPaint)
    * @see #setEndArrowColour(JDRPaint)
    */
   public JDRPaint getMidArrowColour()
   {
      return midMarker.fillPaint;
   }

   /**
    * Gets the colour of the end marker for this stroke.
    * @return the end marker paint or null if the marker paint
    * should be the same as the path
    * @see #getStartArrowColour()
    * @see #getMidArrowColour()
    * @see #setStartArrowColour(JDRPaint)
    * @see #setMidArrowColour(JDRPaint)
    * @see #setEndArrowColour(JDRPaint)
    */
   public JDRPaint getEndArrowColour()
   {
      return endMarker.fillPaint;
   }

   /**
    * Sets the paint for the start marker. If the specified paint
    * is null or an instance of {@link JDRTransparent}, the marker
    * will use the same colour as the path to which it is attached.
    * @param paint the marker paint or null if the marker should
    * use the same paint as the path
    * @see #getStartArrowColour()
    * @see #getMidArrowColour()
    * @see #getEndArrowColour()
    * @see #setMidArrowColour(JDRPaint)
    * @see #setEndArrowColour(JDRPaint)
    */
   public void setStartArrowColour(JDRPaint paint)
   {
      startMarker.setFillPaint(paint);
   }

   /**
    * Sets the paint for the mid markers. If the specified paint
    * is null or an instance of {@link JDRTransparent}, the markers
    * will use the same colour as the path to which they are attached.
    * @param paint the marker paint or null if the markers should
    * use the same paint as the path
    * @see #getStartArrowColour()
    * @see #getMidArrowColour()
    * @see #getEndArrowColour()
    * @see #setStartArrowColour(JDRPaint)
    * @see #setEndArrowColour(JDRPaint)
    */
   public void setMidArrowColour(JDRPaint paint)
   {
      midMarker.setFillPaint(paint);
   }

   /**
    * Sets the paint for the end marker. If the specified paint
    * is null or an instance of {@link JDRTransparent}, the marker
    * will use the same colour as the path to which it is attached.
    * @param paint the marker paint or null if the marker should
    * use the same paint as the path
    * @see #getStartArrowColour()
    * @see #getMidArrowColour()
    * @see #getEndArrowColour()
    * @see #setStartArrowColour(JDRPaint)
    * @see #setMidArrowColour(JDRPaint)
    */
   public void setEndArrowColour(JDRPaint paint)
   {
      endMarker.setFillPaint(paint);
   }

   /**
    * Gets the ID associated with the start marker for this stroke.
    * @return the start marker's ID
    * @see JDRMarker#getType()
    * @see #getMidArrowType()
    * @see #getEndArrowType()
    */
   public int getStartArrowType()
   {
      return startMarker.getType();
   }

   /**
    * Gets the ID associated with the mid markers for this stroke.
    * @return the ID of the mid markers
    * @see JDRMarker#getType()
    * @see #getStartArrowType()
    * @see #getEndArrowType()
    */
   public int getMidArrowType()
   {
      return midMarker.getType();
   }

   /**
    * Gets the ID associated with the end marker for this stroke.
    * @return the end marker's ID
    * @see JDRMarker#getType()
    * @see #getStartArrowType()
    * @see #getMidArrowType()
    */
   public int getEndArrowType()
   {
      return endMarker.getType();
   }

   /**
    * Gets the repeat count of the start marker for this stroke.
    * A value of 1 indicates a single marker.
    * @return the repeat count of the start marker
    * @see #getMidArrowRepeated()
    * @see #getEndArrowRepeated()
    */
   public int getStartArrowRepeated()
   {
      return startMarker.getRepeated();
   }

   /**
    * Gets the repeat count of the mid markers for this stroke.
    * A value of 1 indicates a single marker at each mid point
    * vertex.
    * @return the repeat count of the mid markers
    * @see #getStartArrowRepeated()
    * @see #getEndArrowRepeated()
    */
   public int getMidArrowRepeated()
   {
      return midMarker.getRepeated();
   }

   /**
    * Gets the repeat count of the end marker for this stroke.
    * A value of 1 indicates a single marker.
    * @return the repeat count of the end marker
    * @see #getStartArrowRepeated()
    * @see #getMidArrowRepeated()
    */
   public int getEndArrowRepeated()
   {
      return endMarker.getRepeated();
   }

   /**
    * Determines if the start marker should be reversed.
    * @return true if the start marker should be reversed
    * @see #getMidArrowReverse()
    * @see #getEndArrowReverse()
    */
   public boolean getStartArrowReverse()
   {
      return startMarker.isReversed();
   }

   /**
    * Determines if the mid markers should be reversed.
    * @return true if the mid markers should be reversed
    * @see #getStartArrowReverse()
    * @see #getEndArrowReverse()
    */
   public boolean getMidArrowReverse()
   {
      return midMarker.isReversed();
   }

   /**
    * Determines if the end marker should be reversed.
    * @return true if the end marker should be reversed
    * @see #getStartArrowReverse()
    * @see #getMidArrowReverse()
    */
   public boolean getEndArrowReverse()
   {
      return endMarker.isReversed();
   }

   /**
    * Determines if the start marker should be oriented so that
    * its x axis is aligned with the path's gradient vector.
    * @return true if the start marker should be oriented so that
    * its x axis is aligned with the path's gradient vector
    * @see #getMidArrowAutoOrient()
    * @see #getEndArrowAutoOrient()
    * @see #getStartArrowAngle()
    * @see #getMidArrowAngle()
    * @see #getEndArrowAngle()
    */
   public boolean getStartArrowAutoOrient()
   {
      return startMarker.getAutoOrient();
   }

   /**
    * Gets the start marker's angle of orientation in the event that
    * {@link #getStartArrowAutoOrient()} returns false.
    * @return the start marker's angle of orientation
    * @see #getStartArrowAutoOrient()
    * @see #getMidArrowAutoOrient()
    * @see #getEndArrowAutoOrient()
    * @see #getMidArrowAngle()
    * @see #getEndArrowAngle()
    */
   public double getStartArrowAngle()
   {
      return startMarker.getAngle();
   }

   /**
    * Determines if the mid markers should be oriented so that
    * the marker x axis is aligned with the path's gradient vector.
    * @return true if the mid markers should be oriented so that
    * the marker x axis is aligned with the path's gradient vector
    * @see #getStartArrowAutoOrient()
    * @see #getEndArrowAutoOrient()
    * @see #getStartArrowAngle()
    * @see #getMidArrowAngle()
    * @see #getEndArrowAngle()
    */
   public boolean getMidArrowAutoOrient()
   {
      return midMarker.getAutoOrient();
   }

   /**
    * Gets the angle of orientation for the mid markers in the event 
    * that {@link #getMidArrowAutoOrient()} returns false.
    * @return the angle of orientation for the mid markers
    * @see #getStartArrowAutoOrient()
    * @see #getMidArrowAutoOrient()
    * @see #getEndArrowAutoOrient()
    * @see #getStartArrowAngle()
    * @see #getEndArrowAngle()
    */
   public double getMidArrowAngle()
   {
      return midMarker.getAngle();
   }

   /**
    * Determines if the end marker should be oriented so that
    * its x axis is aligned with the path's gradient vector.
    * @return true if the end marker should be oriented so that
    * its x axis is aligned with the path's gradient vector
    * @see #getStartArrowAutoOrient()
    * @see #getMidArrowAutoOrient()
    * @see #getStartArrowAngle()
    * @see #getMidArrowAngle()
    * @see #getEndArrowAngle()
    */
   public boolean getEndArrowAutoOrient()
   {
      return endMarker.getAutoOrient();
   }

   /**
    * Gets the end marker's angle of orientation in the event that
    * {@link #getEndArrowAutoOrient()} returns false.
    * @return the end marker's angle of orientation
    * @see #getStartArrowAutoOrient()
    * @see #getMidArrowAutoOrient()
    * @see #getEndArrowAutoOrient()
    * @see #getStartArrowAngle()
    * @see #getMidArrowAngle()
    */
   public double getEndArrowAngle()
   {
      return endMarker.getAngle();
   }

   /**
    * Gets the start marker's size.
    * @return the start marker's size
    * @see #getMidArrowSize()
    * @see #getEndArrowSize()
    */
   public double getStartArrowSize()
   {
      return startMarker.getSize();
   }

   /**
    * Gets the size of the mid markers.
    * @return the size of the mid markers
    * @see #getStartArrowSize()
    * @see #getEndArrowSize()
    */
   public double getMidArrowSize()
   {
      return midMarker.getSize();
   }

   /**
    * Gets the end marker's size.
    * @return the sendmarker's size
    * @see #getStartArrowSize()
    * @see #getMidArrowSize()
    */
   public double getEndArrowSize()
   {
      return endMarker.getSize();
   }

   /**
    * Determines if the start marker has user offset enabled.
    * @return true if the start marker has user offset enabled
    * @see #getMidUserOffsetEnabled()
    * @see #getEndUserOffsetEnabled()
    */
   public boolean getStartUserOffsetEnabled()
   {
      return startMarker.isUserOffsetEnabled();
   }

   /**
    * Determines if the mid markers have user offset enabled.
    * @return true if the mid markers have user offset enabled
    * @see #getStartUserOffsetEnabled()
    * @see #getEndUserOffsetEnabled()
    */
   public boolean getMidUserOffsetEnabled()
   {
      return midMarker.isUserOffsetEnabled();
   }

   /**
    * Determines if the end marker has user offset enabled.
    * @return true if the end marker has user offset enabled
    * @see #getStartUserOffsetEnabled()
    * @see #getMidUserOffsetEnabled()
    */
   public boolean getEndUserOffsetEnabled()
   {
      return endMarker.isUserOffsetEnabled();
   }

   /**
    * Sets whether the start marker has user offset enabled.
    * @param enabled true if user offset is to be enabled
    * @see #getStartUserOffsetEnabled()
    * @see #setMidUserOffsetEnabled(boolean)
    * @see #setEndUserOffsetEnabled(boolean)
    */
   public void setStartUserOffsetEnabled(boolean enabled)
   {
      startMarker.enableUserOffset(enabled);
   }

   public void setStartOverlay(boolean overlaid)
   {
      startMarker.setOverlay(overlaid);
   }

   public void setMidOverlay(boolean overlaid)
   {
      midMarker.setOverlay(overlaid);
   }

   public void setEndOverlay(boolean overlaid)
   {
      endMarker.setOverlay(overlaid);
   }

   /**
    * Sets whether the mid markers have user offset enabled.
    * @param enabled true if user offset is to be enabled
    * @see #getMidUserOffsetEnabled()
    * @see #setStartUserOffsetEnabled(boolean)
    * @see #setEndUserOffsetEnabled(boolean)
    */
   public void setMidUserOffsetEnabled(boolean enabled)
   {
      midMarker.enableUserOffset(enabled);
   }

   /**
    * Sets whether the end marker has user offset enabled.
    * @param enabled true if user offset is to be enabled
    * @see #setStartUserOffsetEnabled(boolean)
    * @see #setMidUserOffsetEnabled(boolean)
    * @see #getEndUserOffsetEnabled()
    */
   public void setEndUserOffsetEnabled(boolean enabled)
   {
      endMarker.enableUserOffset(enabled);
   }

   /**
    * Gets start marker offset.
    * @return start marker offset
    * @see #getMidOffset()
    * @see #getEndOffset()
    */
   public double getStartOffset()
   {
      return startMarker.getOffset();
   }

   /**
    * Gets mid marker offset.
    * @return mid marker offset
    * @see #getStartOffset()
    * @see #getEndOffset()
    */
   public double getMidOffset()
   {
      return midMarker.getOffset();
   }

   /**
    * Gets end marker offset.
    * @return end marker offset
    * @see #getStartOffset()
    * @see #getMidOffset()
    */
   public double getEndOffset()
   {
      return endMarker.getOffset();
   }

   /**
    * Sets start marker offset.
    * @param offset start marker offset
    * @see #getStartOffset()
    * @see #setMidOffset(double)
    * @see #setEndOffset(double)
    */
   public void setStartOffset(double offset)
   {
      startMarker.setOffset(offset);
   }

   /**
    * Sets mid marker offset.
    * @param offset mid marker offset
    * @see #getMidOffset()
    * @see #setStartOffset(double)
    * @see #setEndOffset(double)
    */
   public void setMidOffset(double offset)
   {
      midMarker.setOffset(offset);
   }

   /**
    * Sets end marker offset.
    * @param offset end marker offset
    * @see #getEndOffset()
    * @see #setStartOffset(double)
    * @see #setMidOffset(double)
    */
   public void setEndOffset(double offset)
   {
      endMarker.setOffset(offset);
   }

   /**
    * Determines if the start marker has repeat offset enabled.
    * @return true if the start marker has repeat offset enabled
    * @see #getMidRepeatOffsetEnabled()
    * @see #getEndRepeatOffsetEnabled()
    */
   public boolean getStartRepeatOffsetEnabled()
   {
      return startMarker.isUserRepeatOffsetEnabled();
   }

   /**
    * Determines if the mid markers have repeat offset enabled.
    * @return true if the mid markers have repeat offset enabled
    * @see #getStartRepeatOffsetEnabled()
    * @see #getEndRepeatOffsetEnabled()
    */
   public boolean getMidRepeatOffsetEnabled()
   {
      return midMarker.isUserRepeatOffsetEnabled();
   }

   /**
    * Determines if the end marker has repeat offset enabled.
    * @return true if the end marker has repeat offset enabled
    * @see #getStartRepeatOffsetEnabled()
    * @see #getMidRepeatOffsetEnabled()
    */
   public boolean getEndRepeatOffsetEnabled()
   {
      return endMarker.isUserRepeatOffsetEnabled();
   }

   /**
    * Sets whether the start marker has repeat offset enabled.
    * @param enabled true if repeat offset is to be enabled
    * @see #getStartRepeatOffsetEnabled()
    * @see #setMidRepeatOffsetEnabled(boolean)
    * @see #setEndRepeatOffsetEnabled(boolean)
    */
   public void setStartRepeatOffsetEnabled(boolean enabled)
   {
      startMarker.enableUserRepeatOffset(enabled);
   }

   /**
    * Sets whether the mid markers have repeat offset enabled.
    * @param enabled true if repeat offset is to be enabled
    * @see #getMidRepeatOffsetEnabled()
    * @see #setStartRepeatOffsetEnabled(boolean)
    * @see #setEndRepeatOffsetEnabled(boolean)
    */
   public void setMidRepeatOffsetEnabled(boolean enabled)
   {
      midMarker.enableUserRepeatOffset(enabled);
   }

   /**
    * Sets whether the end marker has repeat offset enabled.
    * @param enabled true if repeat offset is to be enabled
    * @see #setStartRepeatOffsetEnabled(boolean)
    * @see #setMidRepeatOffsetEnabled(boolean)
    * @see #getEndRepeatOffsetEnabled()
    */
   public void setEndRepeatOffsetEnabled(boolean enabled)
   {
      endMarker.enableUserRepeatOffset(enabled);
   }

   /**
    * Gets start marker repeat offset.
    * @return start marker repeat offset
    * @see #getMidRepeatOffset()
    * @see #getEndRepeatOffset()
    */
   public double getStartRepeatOffset()
   {
      return startMarker.getRepeatOffset();
   }

   /**
    * Gets mid marker repeat offset.
    * @return mid marker repeat offset
    * @see #getStartRepeatOffset()
    * @see #getEndRepeatOffset()
    */
   public double getMidRepeatOffset()
   {
      return midMarker.getRepeatOffset();
   }

   /**
    * Gets end marker repeat offset.
    * @return end marker repeat offset
    * @see #getStartRepeatOffset()
    * @see #getMidRepeatOffset()
    */
   public double getEndRepeatOffset()
   {
      return endMarker.getOffset();
   }

   /**
    * Sets start marker repeat offset.
    * @param offset start marker repeat offset
    * @see #getStartRepeatOffset()
    * @see #setMidRepeatOffset(double)
    * @see #setEndRepeatOffset(double)
    */
   public void setStartRepeatOffset(double offset)
   {
      startMarker.setRepeatOffset(offset);
   }

   /**
    * Sets mid marker repeat offset.
    * @param offset mid marker repeat offset
    * @see #getMidRepeatOffset()
    * @see #setStartRepeatOffset(double)
    * @see #setEndRepeatOffset(double)
    */
   public void setMidRepeatOffset(double offset)
   {
      midMarker.setRepeatOffset(offset);
   }

   /**
    * Sets end marker repeat offset.
    * @param offset end marker repeat offset
    * @see #getEndRepeatOffset()
    * @see #setStartRepeatOffset(double)
    * @see #setMidRepeatOffset(double)
    */
   public void setEndRepeatOffset(double offset)
   {
      endMarker.setRepeatOffset(offset);
   }

   /**
    * Sets the repeat factor for the start marker.
    * @param repeat the repeat factor
    * @throws InvalidRepeatValueException if the repeat factor is less
    * than 1
    */
   public void setStartArrowRepeat(int repeat)
      throws InvalidRepeatValueException
   {
      startMarker.setRepeated(repeat);
   }

   /**
    * Sets whether the start marker should be reversed.
    * @param isReversed true if the start marker should be reversed
    */
   public void setStartArrowReverse(boolean isReversed)
   {
      startMarker.setReversed(isReversed);
   }

   /**
    * Sets the size of the start marker.
    * @param size the size to set the start marker
    */
   public void setStartArrowSize(double size)
   {
      startMarker.setSize(size);
   }

   /**
    * Sets whether the start marker should be oriented so that its
    * x axis lies along the path's gradient vector.
    * @param orient true if the start marker should be oriented so
    * that its x axis lies along the path's gradient vector
    */
   public void setStartArrowAutoOrient(boolean orient)
   {
      startMarker.setOrient(orient);
   }

   /**
    * Sets the angle of orientation for the start marker in the event
    * that {@link #getStartArrowAutoOrient()} returns false.
    * @param angle the angle of orientation
    */
   public void setStartArrowAngle(double angle)
   {
      startMarker.setAngle(angle);
   }

   /**
    * Sets the start marker.
    * @param marker the marker to use at the start of the path
    * @see #setStartArrow(int)
    * @see #setStartArrow(int,double,int,boolean)
    * @see #setMidArrow(JDRMarker)
    * @see #setEndArrow(JDRMarker)
    */
   public void setStartArrow(JDRMarker marker)
   {
      startMarker = marker;
      startMarker.setPenWidth(penWidth);
   }

   /**
    * Sets the start marker.
    * @param type the marker ID
    * @throws InvalidMarkerTypeException  if the marker ID doesn't
    * correspond to any known marker
    * @see #setStartArrow(JDRMarker)
    * @see #setStartArrow(int,double,int,boolean)
    * @see #setMidArrow(int)
    * @see #setEndArrow(int)
    */
   public void setStartArrow(int type)
      throws InvalidMarkerTypeException
   {
      try
      {
         setStartArrow(type, getStartArrowSize(),
            getStartArrowRepeated(),
            getStartArrowReverse());
      }
      catch (InvalidRepeatValueException ignore)
      {
      }
   }

   /**
    * Sets the start marker.
    * @param type the marker ID
    * @param size the marker's size
    * @param repeat the repeat factor
    * @param isReversed true if the marker should be reversed
    * @throws InvalidMarkerTypeException  if the marker ID doesn't
    * correspond to any known marker
    * @throws InvalidRepeatValueException if the repeat factor is
    * less than 1
    * @see #setStartArrow(int)
    * @see #setStartArrow(JDRMarker)
    * @see #setMidArrow(int,double,int,boolean)
    * @see #setEndArrow(int,double,int,boolean)
    */
   public void setStartArrow(int type, double size,
                             int repeat,
                             boolean isReversed)
      throws InvalidMarkerTypeException,
             InvalidRepeatValueException
   {
      startMarker = JDRMarker.getPredefinedMarker(
         type, penWidth, repeat, 
         isReversed, size);
   }

   /**
    * Sets the repeat factor for the mid markers.
    * @param repeat the repeat factor
    * @throws InvalidRepeatValueException if the repeat factor is less
    * than 1
    */
   public void setMidArrowRepeat(int repeat)
      throws InvalidRepeatValueException
   {
      midMarker.setRepeated(repeat);
   }

   /**
    * Sets whether the mid markers should be reversed.
    * @param isReversed true if the mid markers should be reversed
    */
   public void setMidArrowReverse(boolean isReversed)
   {
      midMarker.setReversed(isReversed);
   }

   /**
    * Sets whether the mid markers should be oriented so that the
    * x axis lies along the path's gradient vector.
    * @param orient true if the mid markers should be oriented so
    * that the x axis lies along the path's gradient vector
    */
   public void setMidArrowAutoOrient(boolean orient)
   {
      midMarker.setOrient(orient);
   }

   /**
    * Sets the angle of orientation for the mid markers in the event
    * that {@link #getMidArrowAutoOrient()} returns false.
    * @param angle the angle of orientation
    */
   public void setMidArrowAngle(double angle)
   {
      midMarker.setAngle(angle);
   }

   /**
    * Sets the size of the mid markers.
    * @param size the size to set the mid marker
    */
   public void setMidArrowSize(double size)
   {
      midMarker.setSize(size);
   }

   /**
    * Sets the mid markers.
    * @param marker the marker to use at the mid point vertices
    * of the path
    * @see #setMidArrow(int)
    * @see #setMidArrow(int,double,int,boolean)
    * @see #setStartArrow(JDRMarker)
    * @see #setEndArrow(JDRMarker)
    */
   public void setMidArrow(JDRMarker marker)
   {
      midMarker = marker;
      midMarker.setPenWidth(penWidth);
   }

   /**
    * Sets the mid markers.
    * @param type the marker ID
    * @throws InvalidMarkerTypeException  if the marker ID doesn't
    * correspond to any known marker
    * @see #setMidArrow(JDRMarker)
    * @see #setMidArrow(int,double,int,boolean)
    * @see #setStartArrow(int)
    * @see #setEndArrow(int)
    */
   public void setMidArrow(int type)
      throws InvalidMarkerTypeException
   {
      try
      {
         setMidArrow(type, getMidArrowSize(),
            getMidArrowRepeated(),
            getMidArrowReverse());
      }
      catch (InvalidRepeatValueException ignore)
      {
      }
   }

   /**
    * Sets the mid markers.
    * @param type the marker ID
    * @param size the size of the markers
    * @param repeat the repeat factor
    * @param isReversed true if the markers should be reversed
    * @throws InvalidMarkerTypeException  if the marker ID doesn't
    * correspond to any known marker
    * @throws InvalidRepeatValueException if the repeat factor is
    * less than 1
    * @see #setMidArrow(int)
    * @see #setMidArrow(JDRMarker)
    * @see #setStartArrow(int,double,int,boolean)
    * @see #setEndArrow(int,double,int,boolean)
    */
   public void setMidArrow(int type, double size,
                             int repeat,
                             boolean isReversed)
      throws InvalidMarkerTypeException,
             InvalidRepeatValueException
   {
      midMarker = JDRMarker.getPredefinedMarker(
         type, penWidth, repeat, 
         isReversed, size);
   }

   /**
    * Sets the repeat factor for the end marker.
    * @param repeat the repeat factor
    * @throws InvalidRepeatValueException if the repeat factor is less
    * than 1
    */
   public void setEndArrowRepeat(int repeat)
      throws InvalidRepeatValueException
   {
      endMarker.setRepeated(repeat);
   }

   /**
    * Sets whether the end marker should be reversed.
    * @param isReversed true if the end marker should be reversed
    */
   public void setEndArrowReverse(boolean isReversed)
   {
      endMarker.setReversed(isReversed);
   }

   /**
    * Sets whether the end marker should be oriented so that its
    * x axis lies along the path's gradient vector.
    * @param orient true if the end marker should be oriented so
    * that its x axis lies along the path's gradient vector
    */
   public void setEndArrowAutoOrient(boolean orient)
   {
      endMarker.setOrient(orient);
   }

   /**
    * Sets the angle of orientation for the end marker in the event
    * that {@link #getEndArrowAutoOrient()} returns false.
    * @param angle the angle of orientation
    */
   public void setEndArrowAngle(double angle)
   {
      endMarker.setAngle(angle);
   }

   /**
    * Sets the size of the end marker.
    * @param size the size to set the end marker
    */
   public void setEndArrowSize(double size)
   {
      endMarker.setSize(size);
   }

   /**
    * Sets the end marker.
    * @param marker the marker to use at the end of the path
    * @see #setEndArrow(int)
    * @see #setEndArrow(int,double,int,boolean)
    * @see #setStartArrow(JDRMarker)
    * @see #setMidArrow(JDRMarker)
    */
   public void setEndArrow(JDRMarker marker)
   {
      endMarker = marker;
      endMarker.setPenWidth(penWidth);
   }

   /**
    * Sets the end marker.
    * @param type the marker ID
    * @throws InvalidMarkerTypeException  if the marker ID doesn't
    * correspond to any known marker
    * @see #setEndArrow(JDRMarker)
    * @see #setEndArrow(int,double,int,boolean)
    * @see #setStartArrow(int)
    * @see #setMidArrow(int)
    */
   public void setEndArrow(int type)
      throws InvalidMarkerTypeException
   {
      try
      {
         setEndArrow(type, getEndArrowSize(), getEndArrowRepeated(),
            getEndArrowReverse());
      }
      catch (InvalidRepeatValueException ignore)
      {
      }
   }

   /**
    * Sets the end marker.
    * @param type the marker ID
    * @param size the marker's size
    * @param repeat the repeat factor
    * @param isReversed true if the marker should be reversed
    * @throws InvalidMarkerTypeException  if the marker ID doesn't
    * correspond to any known marker
    * @throws InvalidRepeatValueException if the repeat factor is
    * less than 1
    * @see #setEndArrow(int)
    * @see #setEndArrow(JDRMarker)
    * @see #setStartArrow(int,double,int,boolean)
    * @see #setMidArrow(int,double,int,boolean)
    */
   public void setEndArrow(int type, double size,
                           int repeat,
                           boolean isReversed)
      throws InvalidMarkerTypeException,
             InvalidRepeatValueException
   {
      endMarker = JDRMarker.getPredefinedMarker(type,
         penWidth, repeat, isReversed,
         size);
   }

   /**
    * Creates the stroked shape of the given path. Note that
    * this does not include the markers as they may be a different
    * colour to the path, so they need to be drawn separately.
    * @param p the path to stroke
    * @return the outline of the stroked path
    */
   public Shape createStrokedShape(Shape p)
   {
      // this was changed in version 0.1.8b
      // the markers are no longer included in 
      // createStrokedShape to allow for different
      // coloured markers. So this now just
      // uses the BasicStroke createStrokedShape function
      BasicStroke stroke = new BasicStroke((float)penWidth,
            capStyle, joinStyle, (float)mitreLimit,
            dashPattern.pattern, dashPattern.offset);

      Shape shape = stroke.createStrokedShape(p);

      return shape;
   }

   /**
    * Returns the PGF commands to set this line style. Note that
    * this does not include the markers as the pgf package does not
    * support mid markers or markers that have a fixed orientation.
    * @return string containing PGF commands that set this line style
    */
   public String pgf()
   {
      String eol = System.getProperty("line.separator", "\n");
      String str = new String("\\pgfsetlinewidth{"+penWidth+"bp}"+eol);

      switch (capStyle)
      {
         case BasicStroke.CAP_SQUARE :
            str += "\\pgfsetrectcap ";
         break;
         case BasicStroke.CAP_BUTT :
            str += "\\pgfsetbuttcap ";
         break;
         case BasicStroke.CAP_ROUND :
            str += "\\pgfsetroundcap ";
         break;
      }

      str += eol;

      switch (joinStyle)
      {
         case BasicStroke.JOIN_MITER :
            str += "\\pgfsetmiterjoin ";
            str += "\\pgfsetmiterlimit{"+mitreLimit+"}";
         break;
         case BasicStroke.JOIN_BEVEL :
            str += "\\pgfsetbeveljoin ";
         break;
         case BasicStroke.JOIN_ROUND :
            str += "\\pgfsetroundjoin ";
         break;
      }

      //str += eol;

      //str += getStartArrow().pgf(true);
      //str += getEndArrow().pgf(false);

      if (dashPattern.pattern != null)
      {
         str += eol;

         str += "\\pgfsetdash{";
         for (int i = 0; i < dashPattern.pattern.length; i++)
         {
            str += "{"+dashPattern.pattern[i]+"bp}";
         }
         str += "}{"+dashPattern.offset+"bp}";
      }

      return str+eol;
   }

   /**
    * Creates a copy of this line style.
    * @return a copy of this line style
    */
   public Object clone()
   {
      JDRBasicStroke s = new JDRBasicStroke();

      s.penWidth    = penWidth;
      s.capStyle    = capStyle;
      s.joinStyle   = joinStyle;
      s.mitreLimit  = mitreLimit;
      s.dashPattern = (DashPattern)dashPattern.clone();
      s.windingRule = windingRule;

      s.startMarker = (JDRMarker)startMarker.clone();
      s.midMarker = (JDRMarker)midMarker.clone();
      s.endMarker = (JDRMarker)endMarker.clone();

      return s;
   }

   /**
    * Determines if this object is the same as another object.
    * @param o the object with which to compare this object
    * @return true if this object is the equivalent to the other object
    */
   public boolean equals(Object o)
   {
      if (this == o) return true;

      if (o == null) return false;

      if (!(o instanceof JDRBasicStroke)) return false;

      JDRBasicStroke s = (JDRBasicStroke)o;

      if (penWidth != s.penWidth) return false;
      if (capStyle != s.capStyle) return false;
      if (joinStyle != s.joinStyle) return false;
      if (mitreLimit != s.mitreLimit) return false;
      if (windingRule != s.windingRule) return false;
      if (!dashPattern.equals(s.dashPattern)) return false;
      if (!startMarker.equals(s.startMarker)) return false;
      if (!midMarker.equals(s.midMarker)) return false;
      if (!endMarker.equals(s.endMarker)) return false;

      return true;
   }

   /**
    * Saves this line style in the given version of JDR format.
    * @param dout the output stream
    * @param version the JDR version
    * @throws IOException if I/O error occurs
    * @see #read(DataInputStream,float)
    * @see #saveAJR(PrintWriter,float)
    */
   public void save(DataOutputStream dout, float version)
      throws IOException
   {
      dout.writeFloat((float)penWidth);
      dashPattern.save(dout, version);
      dout.writeByte((byte)capStyle);
      dout.writeByte((byte)joinStyle);
      if (joinStyle==BasicStroke.JOIN_MITER)
      {
         dout.writeFloat((float)mitreLimit);
      }
      dout.writeByte((byte)windingRule);
      startMarker.save(dout, version);
      if (version >= 1.1f)
      {
         midMarker.save(dout, version);
      }
      endMarker.save(dout, version);
   }

   /**
    * Reads line style data from the input stream in the given 
    * version of JDR file format.
    * @param din the input stream
    * @param version the JDR version
    * @return the line style data read from the input stream
    * @throws IOException if I/O error occurs
    * @throws EOFException if the end of file occurs unexpectedly
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @see #save(DataOutputStream,float)
    * @see #readAJR(BufferedReader,float)
    */
   public static JDRBasicStroke read(DataInputStream din, float version)
      throws IOException,InvalidFormatException,EOFException
   {
      double pen_width = din.readFloat();

      if (pen_width < 0)
      {
         throw new InvalidPenWidthException(pen_width);
      }

      DashPattern pattern = DashPattern.read(din, version);

      int cap_style = (int)din.readByte();

      if (cap_style < 0 || cap_style > 2)
      {
         throw new InvalidCapStyleException(cap_style);
      }

      int join_style = (int)din.readByte();

      if (join_style < 0 || join_style > 2)
      {
         throw new InvalidJoinStyleException(join_style);
      }

      double mitre_limit = 10.0F;

      if (join_style==BasicStroke.JOIN_MITER)
      {
         mitre_limit = din.readFloat();
      }

      int winding_rule = (int)din.readByte();

      if (winding_rule < 0 || winding_rule > 1)
      {
         throw new InvalidWindingRuleException(winding_rule);
      }

      JDRBasicStroke stroke = new JDRBasicStroke(pen_width, cap_style,
                                       join_style, mitre_limit,
                                       pattern, winding_rule);

      stroke.startMarker = JDRMarker.read(din, version);
      stroke.startMarker.setPenWidth(pen_width);
      if (version >= 1.1f)
      {
         stroke.midMarker = JDRMarker.read(din, version);
         stroke.midMarker.setPenWidth(pen_width);
      }
      stroke.endMarker = JDRMarker.read(din, version);
      stroke.endMarker.setPenWidth(pen_width);

      return stroke;
   }

   /**
    * Saves this line style in the given version of AJR format.
    * @param out the output stream
    * @param version the AJR version
    * @throws IOException if I/O error occurs
    * @see #readAJR(BufferedReader,float)
    * @see #save(DataOutputStream,float)
    */
   public void saveAJR(PrintWriter out, float version)
      throws IOException,
        InvalidFormatException
   {
      AJR.writeFloat(out, (float)penWidth);
      dashPattern.saveAJR(out, version);
      AJR.writeInt(out, capStyle);
      AJR.writeInt(out, joinStyle);

      if (joinStyle==BasicStroke.JOIN_MITER)
      {
         AJR.writeFloat(out, (float)mitreLimit);
      }

      AJR.writeInt(out, windingRule);
      startMarker.saveAJR(out, version);
      if (version >= 1.1f)
      {
         midMarker.saveAJR(out, version);
      }
      endMarker.saveAJR(out, version);
   }

   /**
    * Reads line style data from the input stream in the given 
    * version of AJR file format.
    * @param in the input stream
    * @param version the AJR version
    * @return the line style data read from the input stream
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if there is something wrong
    * with the format
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * is exceeded
    * @throws EOFException if the end of file occurs unexpectedly
    * @see #saveAJR(PrintWriter,float)
    * @see #read(DataInputStream,float)
    */
   public static JDRBasicStroke readAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      double pen_width = AJR.readFloat(in);

      if (pen_width < 0)
      {
         throw new InvalidPenWidthException(pen_width, AJR.getLineNum());
      }

      DashPattern pattern = DashPattern.readAJR(in, version);

      int cap_style = AJR.readInt(in);

      if (cap_style < 0 || cap_style > 2)
      {
         throw new InvalidCapStyleException(cap_style, AJR.getLineNum());
      }

      int join_style = AJR.readInt(in);

      if (join_style < 0 || join_style > 2)
      {
         throw new InvalidJoinStyleException(join_style, AJR.getLineNum());
      }

      double mitre_limit = 10.0F;

      if (join_style==BasicStroke.JOIN_MITER)
      {
         mitre_limit = AJR.readFloat(in);
      }

      int winding_rule = AJR.readInt(in);

      if (winding_rule < 0 || winding_rule > 1)
      {
         throw new InvalidWindingRuleException(winding_rule,
            AJR.getLineNum());
      }

      JDRBasicStroke stroke = new JDRBasicStroke(pen_width, cap_style,
                                       join_style, mitre_limit,
                                       pattern, winding_rule);

      stroke.startMarker = JDRMarker.readAJR(in, version);
      stroke.startMarker.setPenWidth(pen_width);
      if (version >= 1.1f)
      {
         stroke.midMarker = JDRMarker.readAJR(in, version);
         stroke.midMarker.setPenWidth(pen_width);
      }
      stroke.endMarker = JDRMarker.readAJR(in, version);
      stroke.endMarker.setPenWidth(pen_width);

      return stroke;
   }

   /**
    * Gets the SVG syntax for this line style.
    * @param p the line paint of the path to which this
    * line style should be applied
    * @return string containing SVG syntax for this line style
    * @see #svgStartMarker(JDRPaint)
    * @see #svgMidMarker(JDRPaint)
    * @see #svgEndMarker(JDRPaint)
    * @see #svgDefs(PrintWriter,JDRGroup)
    */
   public String svg(JDRPaint p)
   {
      String rule = (windingRule==GeneralPath.WIND_EVEN_ODD ?
                     "evenodd" : "nonzero");
      String cap = "square";

      switch (capStyle)
      {
          case BasicStroke.CAP_BUTT :
             cap = "butt";
             break;
          case BasicStroke.CAP_ROUND :
             cap = "round";
      }

      String join = "miter";

      switch (joinStyle)
      {
         case BasicStroke.JOIN_BEVEL :
            join = "bevel";
            break;
         case BasicStroke.JOIN_ROUND :
            join = "round";
      }

      String eol = System.getProperty("line.separator", "\n");

      return "fill-rule=\"" +rule+"\"" +eol
         +"      stroke-width=\""+penWidth+"\"" +eol
         +"      stroke-linecap=\""+cap+"\""+eol
         +"      stroke-linejoin=\""+join+"\""+eol
         +"      stroke-miterlimit=\""+mitreLimit+"\""+eol
         +"      "+dashPattern.svg() + eol
         +"      "+svgStartMarker(p) +eol
         +"      "+svgMidMarker(p) +eol
         +"      "+svgEndMarker(p);
   }

   /**
    * Gets the SVG syntax for the start marker.
    * @param p the line paint of the path to which this
    * line style should be applied
    * @return string containing SVG syntax for the start marker
    * @see #svg(JDRPaint)
    * @see #svgMidMarker(JDRPaint)
    * @see #svgEndMarker(JDRPaint)
    * @see #svgDefs(PrintWriter,JDRGroup)
    */
   public String svgStartMarker(JDRPaint p)
   {
      return startMarker.svgStartMarker(p);
   }

   /**
    * Gets the SVG syntax for the mid markers.
    * @param p the line paint of the path to which this
    * line style should be applied
    * @return string containing SVG syntax for the mid markers
    * @see #svg(JDRPaint)
    * @see #svgStartMarker(JDRPaint)
    * @see #svgEndMarker(JDRPaint)
    * @see #svgDefs(PrintWriter,JDRGroup)
    */
   public String svgMidMarker(JDRPaint p)
   {
      return midMarker.svgMidMarker(p);
   }

   /**
    * Gets the SVG syntax for the end marker.
    * @param p the line paint of the path to which this
    * line style should be applied
    * @return string containing SVG syntax for the end marker
    * @see #svg(JDRPaint)
    * @see #svgStartMarker(JDRPaint)
    * @see #svgMidMarker(JDRPaint)
    * @see #svgDefs(PrintWriter,JDRGroup)
    */
   public String svgEndMarker(JDRPaint p)
   {
      return endMarker.svgEndMarker(p);
   }

   /**
    * Writes the SVG definitions for all the markers used in the
    * given group.
    * @param out the output stream
    * @param group the group containing all the objects that will
    * be saved in SVG file
    * @see #svg(JDRPaint)
    * @see #svgStartMarker(JDRPaint)
    * @see #svgMidMarker(JDRPaint)
    * @see #svgEndMarker(JDRPaint)
    * @see JDRMarker#svgDefs(PrintWriter,JDRGroup)
    */
   public static void svgDefs(PrintWriter out, JDRGroup group)
      throws IOException
   {
      JDRMarker.svgDefs(out, group);
   }

   /**
    * Draws the start marker.
    * @param segment the first segment in the path that has this
    * line style
    * @param g the graphics device
    * @see #drawMidArrowShape(JDRPathSegment,Graphics)
    * @see #drawEndArrowShape(JDRPathSegment,Graphics)
    */
   public void drawStartArrowShape(JDRPathSegment segment, Graphics g)
   {
      getStartArrow().draw(g, penWidth, segment, true);
   }

   /**
    * Draws the mid marker that should be placed at the end of the
    * given segment.
    * @param segment the segment on whose end point the mid marker
    * should be placed (this should not include the last segment in
    * the path)
    * @param g the graphics device
    * @see #drawStartArrowShape(JDRPathSegment,Graphics)
    * @see #drawEndArrowShape(JDRPathSegment,Graphics)
    */
   public void drawMidArrowShape(JDRPathSegment segment, Graphics g)
   {
      getMidArrow().draw(g, penWidth, segment, false);
   }

   /**
    * Draws the end marker.
    * @param segment the last segment in the path that has this
    * line style
    * @param g the graphics device
    * @see #drawStartArrowShape(JDRPathSegment,Graphics)
    * @see #drawMidArrowShape(JDRPathSegment,Graphics)
    */
   public void drawEndArrowShape(JDRPathSegment segment, Graphics g)
   {
      getEndArrow().draw(g, penWidth, segment, false);
   }

   /**
    * Writes the shape of the start marker in EPS format.
    * @param pathPaint the line paint applied to the path that has
    * this line style
    * @param pathBBox the bounding box of the path that has this
    * line style
    * @param segment the first segment in the path that has this
    * line style
    * @param out the output stream
    * @throws IOException if I/O error occurs
    * @see #saveEPSMidArrowShape(JDRPaint,BBox,JDRPathSegment,PrintWriter)
    * @see #saveEPSEndArrowShape(JDRPaint,BBox,JDRPathSegment,PrintWriter)
    */
   public void saveEPSStartArrowShape(JDRPaint pathPaint,
      BBox pathBBox, JDRPathSegment segment, PrintWriter out)
      throws IOException
   {
      getStartArrow().saveEPS(penWidth, pathPaint, pathBBox,
         segment, true, out);
   }

   /**
    * Writes the shape of the mid marker in EPS format.
    * @param pathPaint the line paint applied to the path that has
    * this line style
    * @param pathBBox the bounding box of the path that has this
    * line style
    * @param segment the segment on whose end point the mid marker
    * should be placed (this should not include the last segment in
    * the path)
    * @param out the output stream
    * @throws IOException if I/O error occurs
    * @see #saveEPSStartArrowShape(JDRPaint,BBox,JDRPathSegment,PrintWriter)
    * @see #saveEPSEndArrowShape(JDRPaint,BBox,JDRPathSegment,PrintWriter)
    */
   public void saveEPSMidArrowShape(JDRPaint pathPaint,
      BBox pathBBox, JDRPathSegment segment, PrintWriter out)
      throws IOException
   {
      getMidArrow().saveEPS(penWidth, pathPaint, pathBBox,
         segment, false, out);
   }

   /**
    * Writes the shape of the end marker in EPS format.
    * @param pathPaint the line paint applied to the path that has
    * this line style
    * @param pathBBox the bounding box of the path that has this
    * line style
    * @param segment the last segment in the path that has this
    * line style
    * @param out the output stream
    * @throws IOException if I/O error occurs
    * @see #saveEPSStartArrowShape(JDRPaint,BBox,JDRPathSegment,PrintWriter)
    * @see #saveEPSMidArrowShape(JDRPaint,BBox,JDRPathSegment,PrintWriter)
    */
   public void saveEPSEndArrowShape(JDRPaint pathPaint,
      BBox pathBBox, JDRPathSegment segment, PrintWriter out)
      throws IOException
   {
      getEndArrow().saveEPS(penWidth, pathPaint, pathBBox,
         segment, false, out);
   }

   /**
    * Draws the given path using this line style.
    * @param path the path to draw
    * @param g the graphics device
    */
   public void drawPath(JDRShape path, Graphics g)
   {
      drawPath(path, path.getGeneralPath(), g);
   }

   /**
    * Draws the given path using this line style. This method is
    * like {@link #drawPath(JDRShape, Graphics)} but the GeneralPath
    * is passed as a parameter.
    * @param shape the path to draw
    * @param generalPath the shape described as a GeneralPath
    * @param g the graphics device
    */
   public void drawPath(JDRShape shape, GeneralPath generalPath, Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;
      g2.fill(createStrokedShape(generalPath));

      int startType = getStartArrowType();
      int midType = getMidArrowType();
      int endType = getEndArrowType();

      if (startType == JDRMarker.ARROW_NONE
        && midType == JDRMarker.ARROW_NONE
        && endType == JDRMarker.ARROW_NONE)
      {
         return;
      }

      JDRPathIterator iterator = shape.getIterator();

      while (iterator.hasNext())
      {
         JDRPathSegment segment = iterator.next();

         JDRMarker marker = segment.getStartMarker();

         if (marker != null)
         {
            marker.draw(g, penWidth, segment, true);
         }

         marker = segment.getEndMarker();

         if (marker != null)
         {
            marker.draw(g, penWidth, segment, false);
         }
      }

   }

   public void drawMarkers(JDRShape shape, Graphics g)
   {
      Graphics2D g2 = (Graphics2D)g;

      int startType = getStartArrowType();
      int midType = getMidArrowType();
      int endType = getEndArrowType();

      if (startType == JDRMarker.ARROW_NONE
        && midType == JDRMarker.ARROW_NONE
        && endType == JDRMarker.ARROW_NONE)
      {
         return;
      }

      JDRPathIterator iterator = shape.getIterator();

      while (iterator.hasNext())
      {
         JDRPathSegment segment = iterator.next();

         JDRMarker marker = segment.getStartMarker();

         if (marker != null && marker.getFillPaint() != null)
         {
            marker.draw(g, penWidth, segment, true);
         }

         marker = segment.getEndMarker();

         if (marker != null && marker.getFillPaint() != null)
         {
            marker.draw(g, penWidth, segment, false);
         }
      }

   }

   /**
    * Writes the given path using this line style in EPS format.
    * (Includes markers).
    * @param path the path to save
    * @param out the output stream
    * @throws IOException if I/O error occurs
    */
   public void saveEPS(JDRShape path, PrintWriter out)
      throws IOException
   {
      JDRPaint paint = path.getLinePaint();
      BBox pathBBox = path.getBBox();

      out.println("gsave");

      if ((paint instanceof JDRGradient)
        || (paint instanceof JDRRadial))
      {
         Shape shape = getStrokedArea(path);

         EPS.fillPath(shape, paint, out);
      }
      else
      {
         out.println(""+penWidth+" setlinewidth");
         switch (capStyle)
         {
            case BasicStroke.CAP_BUTT :
               out.println("0 setlinecap");
            break;
            case BasicStroke.CAP_ROUND :
               out.println("1 setlinecap");
            break;
            case BasicStroke.CAP_SQUARE :
               out.println("2 setlinecap");
            break;
         }
         switch (joinStyle)
         {
            case BasicStroke.JOIN_MITER :
               out.println("0 setlinejoin");
            break;
            case BasicStroke.JOIN_ROUND :
               out.println("1 setlinejoin");
            break;
            case BasicStroke.JOIN_BEVEL :
               out.println("2 setlinejoin");
            break;
         }
         out.println(""+mitreLimit+" setmiterlimit");
         dashPattern.saveEPS(out);
         EPS.drawPath(path.getGeneralPath(), paint, out);
      }
      out.println("grestore");

      if (getStartArrowType() == JDRMarker.ARROW_NONE
        && getMidArrowType() == JDRMarker.ARROW_NONE
        && getEndArrowType() == JDRMarker.ARROW_NONE)
      {
         return;
      }

      JDRPathIterator pi = path.getIterator();

      while (pi.hasNext())
      {
         JDRPathSegment segment = pi.next();

         JDRMarker marker = segment.getStartMarker();

         if (marker != null)
         {
            saveEPSStartArrowShape(paint, pathBBox, segment, out);
         }

         marker = segment.getEndMarker();

         if (marker != null)
         {
            saveEPSEndArrowShape(paint, pathBBox, segment, out);
         }
      }

   }

   /**
    * Gets the stroked outline of the given path including the markers.
    * This method is not used for drawing the path, as the markers
    * may use a different colour to the path's line colour, but
    * is used to determine the bounds of the stroked path.
    * @param path the path to which this line style applies
    * @return the path outlining the stroked shape including the
    * marker outlines
    * @see #getStrokedArea(JDRShape)
    */
   public Shape getStrokedPath(JDRShape path)
   {
      GeneralPath shape = new GeneralPath(
         createStrokedShape(path.getGeneralPath()));

      int startType = getStartArrowType();
      int midType = getMidArrowType();
      int endType = getEndArrowType();

      if (startType == JDRMarker.ARROW_NONE
        && midType == JDRMarker.ARROW_NONE
        && endType == JDRMarker.ARROW_NONE)
      {
         return shape;
      }

      JDRPathIterator iterator = path.getIterator();

      while (iterator.hasNext())
      {
         JDRPathSegment segment = iterator.next();

         JDRMarker marker = segment.getStartMarker();

         if (marker != null)
         {
            shape.append(marker.getCompleteShape(segment, true), false);
         }

         marker = segment.getEndMarker();

         if (marker != null)
         {
            shape.append(marker.getCompleteShape(segment, false), false);
         }
      }

      return shape;
   }

   /**
    * Gets the stroked outline of the given path including the markers.
    * This method is not used for drawing the path, as the markers
    * may use a different colour to the path's line colour, but
    * is used for {@link JDRPath#outlineToPath()}, 
    * {@link JDRPath#parshape(Graphics2D,double,boolean)} and
    * {@link JDRPath#shapepar(Graphics2D,double,boolean)}.
    * @param path the path to which this line style applies
    * @return the path outlining the stroked shape including the
    * marker outlines
    * @see #getStrokedPath(JDRShape)
    */
   public Area getStrokedArea(JDRShape path)
   {
      Area area = new Area(createStrokedShape(path.getGeneralPath()));

      int startType = getStartArrowType();
      int midType = getMidArrowType();
      int endType = getEndArrowType();

      if (startType == JDRMarker.ARROW_NONE
        && midType == JDRMarker.ARROW_NONE
        && endType == JDRMarker.ARROW_NONE)
      {
         return area;
      }

      JDRPathIterator iterator = path.getIterator();

      while (iterator.hasNext())
      {
         JDRPathSegment segment = iterator.next();

         JDRMarker marker = segment.getStartMarker();

         if (marker != null)
         {
            area.add(new Area(marker.getCompleteShape(segment, true)));
         }

         marker = segment.getEndMarker();

         if (marker != null)
         {
            area.add(new Area(marker.getCompleteShape(segment, false)));
         }
      }

      return area;
   }

   /**
    * Gets the outline of the start marker's primary shape.
    * @param segment the first segment in the path to which this
    * line style applies
    * @see #getMidArrowShape(JDRPathSegment)
    * @see #getEndArrowShape(JDRPathSegment)
    * @see JDRMarker#getShape(JDRPathSegment,boolean)
    */
   public Shape getStartArrowShape(JDRPathSegment segment)
   {
      return getStartArrow().getCompleteShape(segment, true);
   }

   /**
    * Gets the outline of the mid marker's primary shape.
    * @param segment the segment on whose end point the mid marker
    * should be placed (this should not include the last segment in
    * the path)
    * @see #getStartArrowShape(JDRPathSegment)
    * @see #getEndArrowShape(JDRPathSegment)
    * @see JDRMarker#getShape(JDRPathSegment,boolean)
    */
   public Shape getMidArrowShape(JDRPathSegment segment)
   {
      return getMidArrow().getCompleteShape(segment, false);
   }

   /**
    * Gets the outline of the end marker's primary shape.
    * @param segment the last segment in the path to which this
    * line style applies
    * @see #getStartArrowShape(JDRPathSegment)
    * @see #getMidArrowShape(JDRPathSegment)
    * @see JDRMarker#getShape(JDRPathSegment,boolean)
    */
   public Shape getEndArrowShape(JDRPathSegment segment)
   {
      return getEndArrow().getCompleteShape(segment, false);
   }

   /**
    * Gets the start marker.
    * @return the start marker
    * @see #getMidArrow()
    * @see #getEndArrow()
    */
   public JDRMarker getStartArrow()
   {
      return startMarker;
   }

   /**
    * Gets the mid marker.
    * @return the mid marker
    * @see #getStartArrow()
    * @see #getEndArrow()
    */
   public JDRMarker getMidArrow()
   {
      return midMarker;
   }

   /**
    * Gets the end marker.
    * @return the end marker
    * @see #getStartArrow()
    * @see #getMidArrow()
    */
   public JDRMarker getEndArrow()
   {
      return endMarker;
   }

   /**
    * Gets the winding rule.
    * @return the winding rule
    */
   public int getWindingRule()
   {
      return windingRule;
   }

   /**
    * Sets the winding rule.
    * @param rule the winding rule
    * @throws InvalidWindingRuleException if the winding rule is not
    * one of: {@link GeneralPath#WIND_EVEN_ODD} or
    * {@link GeneralPath#WIND_NON_ZERO}
    */
   public void setWindingRule(int rule)
      throws InvalidWindingRuleException
   {
      if (rule == GeneralPath.WIND_EVEN_ODD
        ||rule == GeneralPath.WIND_NON_ZERO)
      {
         windingRule = rule;
      }
      else
      {
         throw new InvalidWindingRuleException(rule);
      }
   }

   /**
    * Sets the mitre limit.
    * @param limit the mitre limit
    * @throws InvalidMitreLimitException if the mitre limit is
    * less than 1
    */
   public void setMitreLimit(double limit)
      throws InvalidMitreLimitException
   {
      if (limit < 1.0)
      {
         throw new InvalidMitreLimitException(limit);
      }

      mitreLimit = limit;
   }

   /**
    * Gets the mitre limit.
    * @return the mitre limit
    */
   public double getMitreLimit()
   {
      return mitreLimit;
   }

   /**
    * Gets the cap style.
    * @return the cap style
    */
   public int getCapStyle()
   {
      return capStyle;
   }

   /**
    * Sets the cap style.
    * @param style the cap style
    * @throws InvalidCapStyleException if the cap style is not one of:
    * {@link BasicStroke#CAP_SQUARE}, {@link BasicStroke#CAP_BUTT}
    * or {@link BasicStroke#CAP_ROUND}
    */
   public void setCapStyle(int style)
      throws InvalidCapStyleException
   {
      if (style == BasicStroke.CAP_BUTT
        ||style == BasicStroke.CAP_ROUND
        ||style == BasicStroke.CAP_SQUARE)
      {
         capStyle = style;
      }
      else
      {
         throw new InvalidCapStyleException(style);
      }
   }

   /**
    * Gets the join style.
    * @return the join style
    */
   public int getJoinStyle()
   {
      return joinStyle;
   }

   /**
    * Sets the join style.
    * @param style the join style
    * @throws InvalidJoinStyleException if the join style is not one
    * of: {@link BasicStroke#JOIN_MITER},
    * {@link BasicStroke#JOIN_BEVEL} or {@link BasicStroke#JOIN_ROUND}
    */
   public void setJoinStyle(int style)
      throws InvalidJoinStyleException
   {
      if (style == BasicStroke.JOIN_MITER
        ||style == BasicStroke.JOIN_ROUND
        ||style == BasicStroke.JOIN_BEVEL)
      {
         joinStyle = style;
      }
      else
      {
         throw new InvalidJoinStyleException(style);
      }
   }

   public JDRPathStyleListener getPathStyleListener()
   {
      return pathStyleListener;
   }

   public String info()
   {
      String str = "Basic stroke: dash="+dashPattern;

      str += ", cap="+capStyle
           + ", join="+joinStyle
           + ", mitre="+mitreLimit
           + ", winding rule="+windingRule
           + ", pen width="+penWidth
           + ", start marker="+startMarker
           + ", mid marker="+midMarker
           + ", end marker="+endMarker;

      return str;
   }

   /**
    * The dash pattern.
    */
   public DashPattern dashPattern;

   /**
    * The cap style.
    */
   protected int capStyle;
   /**
    * The join style.
    */
   protected int joinStyle;
   /**
    * The winding rule.
    */
   protected int windingRule;
   /**
    * The mitre limit.
    */
   protected double mitreLimit;

   private double penWidth;
   private JDRMarker startMarker, midMarker, endMarker;

   private static JDRPathStyleListener pathStyleListener
      = new JDRBasicPathStyleListener();
}

