// File          : FlowFrame.java
// Purpose       : provides flowframe information for JpgfDraw
// Date          : 5th June 2006
// Last Modified : 16th February 2007
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.util.*;
import java.lang.*;
import java.awt.geom.*;
import java.awt.font.*;
import javax.swing.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing <a target="_top" href="http://theoval.cmp.uea.ac.uk/~nlct/latex/packages/index.html#flowfram">flowframe</a> information.
 * @author Nicola L C Talbot
 */
public class FlowFrame implements Cloneable,Serializable
{
   /**
    * Constructs a frame of the given type. The type may be one of:
    * {@link #STATIC}, {@link #FLOW}, {@link #DYNAMIC} or 
    * {@link #TYPEBLOCK}. Note that the border, label and page list
    * properties are ignored if the frame type is 
    * {@link #TYPEBLOCK}. The margins are all initialised to 0 and
    * the shape is set to {@link #STANDARD}. The vertical alignment
    * is set to {@link #CENTER} if the frame type is {@link #STATIC},
    * otherwise it is set to {@link #TOP}.
    * @param frameType the type of frame
    * @param hasBorder flag to indicate whether or not this frame has
    * a border
    * @param idl the label to assign to this frame
    * @param pageList the list of pages for which this frame is defined
    */
   public FlowFrame(int frameType, boolean hasBorder, String idl,
                    String pageList)
   {
      type   = frameType;
      border = hasBorder;
      label  = idl;
      pages  = pageList;
      top    = 0.0f;
      bottom = 0.0f;
      left   = 0.0f;
      right  = 0.0f;
      shape  = STANDARD;
      valign = (frameType == STATIC ? CENTER : TOP);
      if (label == "") label = ""+maxid;
      maxid++;
   }

   /**
    * Constructs a default frame of the given type. The frame is set
    * to have no border, a default label and to be shown on all pages.
    * @param frameType the type of frame
    * @see #FlowFrame(int,boolean,String,String)
    */
   public FlowFrame(int frameType)
   {
      this(frameType, false, "", "all");
   }

   /**
    * Draws the labelled text area for the given bounding box. 
    * This is the area in which the text will be placed. A rectangle 
    * will always been drawn regardless of the {@link #shape} 
    * specification. The margins between the text area and the 
    * bounding box are given by {@link #top}, {@link #bottom}, 
    * {@link #left} and {@link #right}.
    * @param g graphics device on which to draw
    * @param bbox bounding box
    */
   public void draw(Graphics g, BBox bbox)
   {
      double x = bbox.getMinX()+left;
      double y = bbox.getMinY()+top;
      double width = bbox.getWidth()-(left+right);
      double height = bbox.getHeight()-(top+bottom);

      Rectangle2D rect = new Rectangle2D.Double(x, y,
         width, height);

      Graphics2D g2 = (Graphics2D)g;

      Stroke oldStroke = g2.getStroke();
      Font oldFont = g2.getFont();
      g2.setStroke(new BasicStroke());
      g2.setFont(JDRCompleteObject.annoteFont);

      g2.draw(rect);

      String str = getDisplayLabel();

      g2.drawString(str, (int)x, (int)(y+height));

      g2.setStroke(oldStroke);
      g2.setFont(oldFont);
   }

   /**
    * Gets the annotation text.
    * @return annotation text
    */
   public String getDisplayLabel()
   {
      String str="";

      switch (type)
      {
         case STATIC :
            str = "static:"+label+":"+pages;
         break;
         case FLOW :
            str = "flow:"+label+":"+pages;
         break;
         case DYNAMIC :
            str = "dynamic:"+label+":"+pages;
         break;
         case TYPEBLOCK :
            str = "typeblock";
      }

      return str;
   }

   /**
    * Gets the bounds of the annotation text.
    * @return annotation text bounds
    */
   public Rectangle2D getLabelBounds(BBox bbox)
   {
      FontRenderContext frc = new FontRenderContext(null,true,true);
      TextLayout layout = new TextLayout(getDisplayLabel(),
         JDRCompleteObject.annoteFont, frc);

      double x = bbox.getMinX()+left;
      double y = bbox.getMinY()+top;
      double height = bbox.getHeight()-(top+bottom);

      Rectangle2D bounds = layout.getBounds();

      bounds.setRect(bounds.getX()+x, bounds.getY()+y+height,
                     bounds.getWidth(), bounds.getHeight());

      return bounds;
   }

   /**
    * Writes the flowframe information in TeX format.
    * The flowfram package requires the frames to be positioned
    * relative to the typeblock while JpgfDraw positions objects
    * relative to the top left corner of the canvas so the typeblock
    * is required to determine the correct co-ordinates.
    * @param object the object to which this frame belongs
    * @param typeblock the typeblock for the LaTeX document
    * @param out the output stream
    * @param baselineskip the value of \baselineskip for the LaTeX
    * document
    * @see #tex(JDRObject,Rectangle2D,PrintWriter,String,double)
    * @throws IOException if I/O error occurs
    * @throws InvalidShapeException if frame has a nonstandard
    * shape but the required shape command can't reproduce the
    * required shape
    */
   public void tex(JDRObject object, Rectangle2D typeblock,
                   PrintWriter out, double baselineskip)
      throws IOException,InvalidShapeException
   {
      tex(object, typeblock, out, "Border command for frame",
          baselineskip);
   }

   // borderCommand -> tex.comment.border_command

   /**
    * Writes the flowframe information in TeX format.
    * The flowfram package requires the frames to be positioned
    * relative to the typeblock while JpgfDraw positions objects
    * relative to the top left corner of the canvas so the typeblock
    * is required to determine the correct co-ordinates.
    * @param object the object to which this frame belongs
    * @param typeblock the typeblock for the LaTeX document
    * @param out the output stream
    * @param borderCommand comment for the definition of the border 
    * command
    * @param baselineskip the value of \baselineskip for the LaTeX
    * document
    * @throws IOException if I/O error occurs
    * @throws InvalidShapeException if frame has a nonstandard
    * shape but the required shape command can't reproduce the
    * required shape
    */
   public void tex(JDRObject object, Rectangle2D typeblock,
                   PrintWriter out, String borderCommand,
                   double baselineskip)
      throws IOException,InvalidShapeException
   {
      BBox bbox = object.getBBox();

      switch (type)
      {
         case STATIC :
            out.print("\\newstaticframe");
         break;
         case FLOW :
            out.print("\\newflowframe");
         break;
         case DYNAMIC :
            out.print("\\newdynamicframe");
         break;
         case TYPEBLOCK :
            out.println("\\geometry{lmargin="
              +PGF.format(left)+"bp,rmargin="
              +PGF.format(right)+"bp,tmargin="
              +PGF.format(top)+"bp,bmargin="
              +PGF.format(bottom)+"bp}");
            return;
      }

      double x0 = typeblock.getX();
      double x  = bbox.getMinX()-x0+left;
      double y0 = typeblock.getY();
      double y  = y0+typeblock.getHeight()-bbox.getMaxY()+bottom;
      double width = bbox.getWidth()-(left+right);
      double height = bbox.getHeight()-(bottom+top);

      out.println("["+pages+"]{"+PGF.format(width)+"bp}{"
         + PGF.format(height)+"bp}{"
         + PGF.format(x) +"bp}{" 
         + PGF.format(y) +"bp}["+label+"]");
      out.println();

      if (border)
      {
         out.println("%"+borderCommand+ " '"+label+"'");
         out.println("\\expandafter\\def\\csname @flf@border@"+label+"\\endcsname#1{%");
         out.println("\\begin{pgfpicture}{0bp}{0bp}{"
            +PGF.format(bbox.getWidth())+"bp}{"
            +PGF.format(bbox.getHeight())+"bp}");

         AffineTransform af = new AffineTransform(1, 0, 0, -1,
            -bbox.getMinX(), bbox.getMaxY());

         out.println("\\pgfputat{"
            + PGF.point(-left, -bottom) +"}{%");

         String pgf = object.pgf(af);

         out.print(pgf);
         out.println("}");
         out.println("\\pgfputat{\\pgfpoint{0bp}{0bp}}{\\pgftext[left,bottom]{#1}}");
         out.println("\\end{pgfpicture}}");

         switch (type)
         {
            case STATIC :
               out.println("\\setstaticframe*");
            break;
            case FLOW :
               out.println("\\setflowframe*");
            break;
            case DYNAMIC :
               out.println("\\setdynamicframe*");
            break;
         }

         out.println("{"+label+"}{offset=0pt,border={@flf@border@"+label+"}}");
         out.println();
      }

      if (shape != STANDARD
       && (type == STATIC || type == DYNAMIC)
       && (object instanceof JDRPath))
      {
         JDRPath path = (JDRPath)object;
         Parshape parshape;

         if (shape == PARSHAPE)
         {
            parshape = path.parshape(null, baselineskip, false);
         }
         else
         {
            parshape = path.shapepar(null, baselineskip, false);
         }

         String shapecmd = parshape.string;

         if (type == STATIC)
         {
            out.println("\\setstaticframe*{"+label+"}{shape={"+shapecmd+"}}");
         }
         else
         {
            out.println("\\setdynamicframe*{"+label+"}{shape={"+shapecmd+"}}");
         }
      }

      if (type == STATIC || type == DYNAMIC)
      {
         out.print("\\set"+(type==STATIC?"static":"dynamic")
                   +"frame*{"+label+"}{valign=");
         switch (valign)
         {
            case TOP :
               out.print("t");
            break;
            case CENTER :
               out.print("c");
            break;
            case BOTTOM :
               out.print("b");
            break;
         }
         out.println("}");
      }
   }

   /**
    * Saves the information for this frame in JDR format.
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    */
   public void save(DataOutputStream dout, float version)
      throws IOException
   {
      dout.writeByte((byte)type);

      if (type != TYPEBLOCK)
      {
         dout.writeBoolean(border);
         dout.writeInt(label.length());
         dout.writeChars(label);
         dout.writeInt(pages.length());
         dout.writeChars(pages);
      }
      dout.writeFloat((float)top);
      dout.writeFloat((float)bottom);
      dout.writeFloat((float)left);
      dout.writeFloat((float)right);

      if (version >= 1.2f)
      {
         if (type == STATIC || type == DYNAMIC)
         {
            dout.writeByte((byte)shape);

            if (version >= 1.3f)
            {
               dout.writeByte((byte)valign);
            }
         }
      }

   }

   /**
    * Reads frame information stored in JDR format.
    * @param din the input stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if data stored incorrectly
    * @return the frame defined by the given information
    */
   public static FlowFrame read(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      int frameType = (int)din.readByte();

      if (frameType < 0 || frameType > TYPEBLOCK)
      {
         throw new InvalidFrameTypeException(frameType);
      }

      boolean hasBorder=false;
      String idl = "";
      String pageList="all";
      double topMargin=0.0f;
      double bottomMargin=0.0f;
      double leftMargin=0.0f;
      double rightMargin=0.0f;

      if (frameType != TYPEBLOCK)
      {
         hasBorder = din.readBoolean();
         int n = din.readInt();
         char[] chars;
         if (n > 0)
         {
            chars = new char[n];

            for (int i = 0; i < n; i++)
            {
               chars[i] = din.readChar();
            }
            idl = new String(chars);
         }
         else if (n < 0)
         {
            throw new InvalidIdlLengthException(n);
         }

         n = din.readInt();
         if (n > 0)
         {
            chars = new char[n];

            for (int i = 0; i < n; i++)
            {
               chars[i] = din.readChar();
            }
            pageList = new String(chars);
         }
         else if (n < 0)
         {
            throw new InvalidPageListLengthException(n);
         }
      }

      topMargin = din.readFloat();
      bottomMargin = din.readFloat();
      leftMargin = din.readFloat();
      rightMargin = din.readFloat();

      FlowFrame f = new FlowFrame(frameType, hasBorder, idl, pageList);

      f.top = topMargin;
      f.bottom = bottomMargin;
      f.left = leftMargin;
      f.right = rightMargin;

      if (version >= 1.2f)
      {
         if (f.getType() == STATIC || f.getType() == DYNAMIC)
         {
            f.setShape((int)din.readByte());
            if (version >= 1.3f)
            {
               f.setVAlign((int)din.readByte());
            }
         }
      }

      return f;
   } 

   /**
    * Saves frame information in AJR format.
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    */
   public void saveAJR(PrintWriter out, float version)
      throws IOException
   {
      AJR.writeInt(out, type);

      if (type != TYPEBLOCK)
      {
         AJR.writeBoolean(out, border);
         AJR.writeInt(out, label.length());
         out.print(label+" ");
         AJR.writeInt(out, pages.length());
         out.print(pages+" ");
      }
      AJR.writeFloat(out, (float)top);
      AJR.writeFloat(out, (float)bottom);
      AJR.writeFloat(out, (float)left);
      AJR.writeFloat(out, (float)right);

      if (version >= 1.2f)
      {
         if (type == STATIC || type == DYNAMIC)
         {
            AJR.writeInt(out, shape);

            if (version >= 1.3f)
            {
               AJR.writeInt(out, valign);
            }
         }
      }
   }

   /**
    * Reads frame information stored in AJR format.
    * @param in the input stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if data is stored incorrectly
    * @throws java.nio.BufferOverflowException if AJR buffer overflows
    * @throws EOFException if end of file occurs unexpectedly
    * @return the frame defined by the given information
    */
   public static FlowFrame readAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      int frameType = AJR.readInt(in);

      if (frameType < 0 || frameType > TYPEBLOCK)
      {
         throw new InvalidFrameTypeException(frameType, AJR.getLineNum());
      }

      boolean hasBorder=false;
      String idl = "";
      String pageList="all";
      double topMargin=0.0f;
      double bottomMargin=0.0f;
      double leftMargin=0.0f;
      double rightMargin=0.0f;

      if (frameType != TYPEBLOCK)
      {
         hasBorder = AJR.readBoolean(in);

         int n = AJR.readInt(in);

         if (n < 0)
         {
            throw new InvalidIdlLengthException(n, AJR.getLineNum());
         }
         else if (n > 0)
         {
            idl = AJR.readString(in, n);
         }

         n = AJR.readInt(in);

         if (n < 0)
         {
            throw new InvalidPageListLengthException(n, AJR.getLineNum());
         }
         else if (n > 0)
         {
            pageList = AJR.readString(in, n);
         }
      }

      topMargin = AJR.readFloat(in);
      bottomMargin = AJR.readFloat(in);
      leftMargin = AJR.readFloat(in);
      rightMargin = AJR.readFloat(in);

      FlowFrame f = new FlowFrame(frameType, hasBorder, idl, pageList);

      f.top = topMargin;
      f.bottom = bottomMargin;
      f.left = leftMargin;
      f.right = rightMargin;

      if (version >= 1.2f)
      {
         if (f.getType() == STATIC || f.getType() == DYNAMIC)
         {
            int frameShape = AJR.readInt(in);

            if (frameShape < 0 || frameShape > 2)
            {
               throw new InvalidShapeIDException(frameShape, AJR.getLineNum());
            }

            f.setShape(frameShape);

            if (version >= 1.3f)
            {
               int align = AJR.readInt(in);

               if (align < 0 || align > 2)
               {
                  throw new InvalidVAlignException(align, AJR.getLineNum());
               }

               f.setVAlign(align);
            }
         }
      }

      return f;
   } 

   /**
    * Returns a copy of this object.
    * @return a copy of this object
    */
   public Object clone()
   {
      FlowFrame f = new FlowFrame(type, border, label, pages);

      f.top = top;
      f.bottom = bottom;
      f.left = left;
      f.right = right;
      f.shape = shape;

      return f;
   }

   /**
    * Makes this frame the same as another frame.
    * @param f the other frame
    */
   public void makeEqual(FlowFrame f)
   {
      type   = f.type;
      border = f.border;
      label  = f.label;
      pages  = f.pages;
      top    = f.top;
      bottom = f.bottom;
      left   = f.left;
      right  = f.right;
      shape  = f.shape;
   }

   /**
    * Determines if this object is equal to another object.
    * @param obj the other object
    * @return true if this object is consider equal to another object
    */
   public boolean equals(Object obj)
   {
      if (this == obj) return true;

      if (obj == null) return false;
      if (!(obj instanceof FlowFrame)) return false;

      FlowFrame f = (FlowFrame)obj;

      if (type != f.type) return false;
      if (!label.equals(f.label)) return false;
      if (border != f.border) return false;
      if (!pages.equals(f.pages)) return false;
      if (top != f.top) return false;
      if (bottom != f.bottom) return false;
      if (left != f.left) return false;
      if (right != f.right) return false;
      if (shape != f.shape) return false;

      return true;
   }

   /**
    * Gets the shape assigned to this frame.
    * The shape may be one of: {@link #STANDARD} (use standard
    * rectangular paragraphs), {@link #PARSHAPE} (use 
    * <code>\parshape</code> to define the paragraph shape) or
    * {@link #SHAPEPAR} (use <code>\shapepar</code> to define the
    * paragraph shape).
    * @return shape identifier
    * @see #setShape(int)
    */
   public int getShape()
   {
      return shape;
   }

   /**
    * Gets the type assigned to this frame.
    * The type may be one of: {@link #STATIC}, {@link #FLOW}, 
    * {@link #DYNAMIC} or {@link #TYPEBLOCK}.
    * @return frame type identifier
    * @see #setType(int)
    */
   public int getType()
   {
      return type;
   }

   /**
    * Sets this frame's type.
    * The type may be one of: {@link #STATIC}, {@link #FLOW}, 
    * {@link #DYNAMIC} or {@link #TYPEBLOCK}. Note that if the
    * type is neither {@link #STATIC} nor {@link #DYNAMIC}, the
    * shape is automatically set to {@link #STANDARD}, since only
    * static or dynamic frames may have a nonstandard shape.
    * @param frameType the type to set this frame
    * @throws InvalidFrameTypeException if the specified type isn't
    * one of: {@link #STATIC}, {@link #FLOW}, 
    * {@link #DYNAMIC} or {@link #TYPEBLOCK}
    * @see #getType()
    */
   public void setType(int frameType)
      throws InvalidFrameTypeException
   {
      if (frameType < 0 || frameType > TYPEBLOCK)
      {
         throw new InvalidFrameTypeException(frameType);
      }

      type = frameType;

      if (!(frameType == STATIC || frameType == DYNAMIC))
      {
         shape = STANDARD;
      }
   }

   /**
    * Sets this frame's shape.
    * The shape may be one of: {@link #STANDARD} (use standard
    * rectangular paragraphs), {@link #PARSHAPE} (use 
    * <code>\parshape</code> to define the paragraph shape) or
    * {@link #SHAPEPAR} (use <code>\shapepar</code> to define the
    * paragraph shape). Note that only static or dynamic frames
    * may have nonstandard shapes.
    * @param frameShape the shape to use for this frame
    * @throws InvalidShapeException if this frame's type doesn't
    * support the requested shape
    * @throws InvalidShapeIDException if the specified shape isn't
    * one of: {@link #STANDARD}, {@link #PARSHAPE} or {@link #SHAPEPAR}
    * @see #getShape()
    */
   public void setShape(int frameShape)
      throws InvalidShapeException,InvalidShapeIDException
   {
      if (frameShape < 0 || frameShape > 2)
      {
         throw new InvalidShapeIDException(frameShape);
      }

      if (type == STATIC || type == DYNAMIC)
      {
         shape = frameShape;
      }
      else if (frameShape == STANDARD)
      {
         shape = STANDARD;
      }
      else
      {
         throw new InvalidShapeException("Non standard shape only avaiblable for static or dynamic frames");
      }
   }

   /**
    * Sets the vertical alignment for this frame.
    * The alignment may be one of: {@link #TOP}, {@link #CENTER} or
    * {@link #BOTTOM}. Note that this setting is only available for
    * static or dynamic frames.
    * @param align the vertical alignment for this frame
    * @throws InvalidShapeException if this frame's type doesn't
    * support vertical alignment setting
    * @throws InvalidVAlignException if the specified alignment isn't
    * one of: {@link #TOP}, {@link #CENTER} or {@link #BOTTOM}
    * @see #getVAlign()
    */
   public void setVAlign(int align)
      throws InvalidShapeException,InvalidVAlignException
   {
      if (align < 0 || align > 2)
      {
         throw new InvalidVAlignException(align);
      }

      if (type == STATIC || type == DYNAMIC)
      {
         valign = align;
      }
      else
      {
         throw new InvalidShapeException("Vertical alignment only avaiblable for static or dynamic frames");
      }
   }

   /**
    * Gets the vertical alignment for this frame.
    * @return the vertical alignment for this frame
    * @see #setVAlign(int)
    */
   public int getVAlign()
   {
      return valign;
   }

   /**
    * Determines whether this frame is defined on all even pages.
    * @return true if this frame's page list is either "all" or "even"
    * otherwise false
    */
   public boolean isDefinedOnEvenPages()
   {
      if (pages.equals("all") || pages.equals("even"))
      {
         return true;
      }

      return false;
   }

   /**
    * Determines whether this frame is defined on all odd pages.
    * @return true if this frame's page list is either "all" or "odd"
    * otherwise false
    */
   public boolean isDefinedOnOddPages()
   {
      if (pages.equals("all") || pages.equals("odd"))
      {
         return true;
      }

      return false;
   }

   /**
    * Determines whether this frame is defined on the given page.
    * @param page the specified page number
    * @return true if this frame's page list includes the specified
    * page otherwise false
    */
   public boolean isDefinedOnPage(int page)
   {
      if (pages.equals("none"))
      {
         return (page == 0);
      }

      if (page <= 0)
      {
         return false;
      }

      if (pages.equals("all"))
      {
         return true;
      }

      boolean isEven = (page%2==0 ? true: false);

      if (pages.equals("odd"))
      {
         return !isEven;
      }

      if (pages.equals("even"))
      {
         return isEven;
      }

      StringTokenizer st = new StringTokenizer(pages, ",");

      while (st.hasMoreTokens())
      {
         String token = st.nextToken();

         int idx=-1;
         int n = token.length();
         if (n == 0) return false;

         if ((idx=token.indexOf('<')) != -1)
         {
            if (idx == n-1) return false;
            String subStr = token.substring(idx+1);
            try
            {
               int i = Integer.parseInt(subStr);

               if (page < i) return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
         else if ((idx=token.indexOf('>')) != -1)
         {
            if (idx == n-1) return false;
            String subStr = token.substring(idx+1);
            try
            {
               int i = Integer.parseInt(subStr);

               if (page > i) return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
         else if ((idx=token.indexOf('-')) != -1)
         {
            if (idx == n-1 || idx == 0) return false;

            String first = token.substring(0, idx);
            String last  = token.substring(idx+1);

            try
            {
               int firstNum = Integer.parseInt(first);
               int lastNum = Integer.parseInt(last);

               if (page >= firstNum && page <= lastNum) return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
         else
         {
            try
            {
               int i = Integer.parseInt(token);
               if (i == page) return true;
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
      }
      return false;
   }

   /**
    * Determines if the given page list is in a valid format.
    * @param pageList the page list to test
    * @return true if the given page list is valid otherwise false
    */
   public static boolean isValidPageList(String pageList)
   {
      if (pageList.equals("all") || pageList.equals("odd")
        || pageList.equals("even") || pageList.equals("none"))
      {
         return true;
      }

      StringTokenizer st = new StringTokenizer(pageList, ",");

      while (st.hasMoreTokens())
      {
         String token = st.nextToken();

         int idx=-1;
         int n = token.length();
         if (n == 0) return false;

         if ((idx=token.indexOf('<')) != -1)
         {
            if (idx == n-1) return false;
            String subStr = token.substring(idx+1);
            try
            {
               int i = Integer.parseInt(subStr);
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
         else if ((idx=token.indexOf('>')) != -1)
         {
            if (idx == n-1) return false;
            String subStr = token.substring(idx+1);
            try
            {
               int i = Integer.parseInt(subStr);
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
         else if ((idx=token.indexOf('-')) != -1)
         {
            if (idx == n-1 || idx == 0) return false;

            String first = token.substring(0, idx);
            String last  = token.substring(idx+1);

            try
            {
               int firstNum = Integer.parseInt(first);
               int lastNum = Integer.parseInt(last);
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
         else
         {
            try
            {
               int i = Integer.parseInt(token);
            }
            catch (NumberFormatException e)
            {
                return false;
            }
         }
      }

      return true;
   }

   public String toString()
   {
      return "FlowFrame["
        + "label="+label
        + ",border="+border
        + ",type="+type
        + ",pages="+pages
        + ",top="+top
        + ",bottom="+bottom
        + ",left="+left
        + ",right="+right
        + ",shape="+shape
        + ",valign="+valign
        + "]";
   }

   /**
    * Indicates that a frame is a static frame.
    */
   public static final int STATIC=0;
   /**
    * Indicates that a frame is a flow frame.
    */
   public static final int FLOW=1;
   /**
    * Indicates that a frame is a dynamic frame.
    */
   public static final int DYNAMIC=2;
   /**
    * Indicates that a frame represents the typeblock.
    */
   public static final int TYPEBLOCK=3;

   /**
    * Indicates that a frame should use a standard paragraph shape.
    */
   public static final int STANDARD=0;
   /**
    * Indicates that a frame should use <code>\parshape</code>
    * to define the paragraph shape. Only available for static or
    * dynamic frames.
    */
   public static final int PARSHAPE=1;
   /**
    * Indicates that a frame should use <code>\shapepar</code>
    * to define the paragraph shape. Only available for static or
    * dynamic frames.
    */
   public static final int SHAPEPAR=2;

   /**
    * Indicates that a frame should be aligned along the top.
    */
   public static final int TOP=0;
   /**
    * Indicates that a frame should be aligned along the middle.
    */
   public static final int CENTER=1;
   /**
    * Indicates that a frame should be aligned along the bottom.
    */
   public static final int BOTTOM=2;

   /**
    * Indicates whether this frame has a border.
    */
   public boolean border=false;

   /**
    * The type assigned to this frame.
    */
   private int type;

   /**
    * The label identifying this frame.
    */
   public String label;
   /**
    * Indicates the pages on which this frame is defined.
    */
   public String pages;

   /**
    * This frame's top margin.
    */
   public double top;
   /**
    * This frame's bottom margin.
    */
   public double bottom;
   /**
    * This frame's left margin.
    */
   public double left;
   /**
    * This frame's right margin.
    */
   public double right;

   private static int maxid=0;
   private int shape=STANDARD;
   private int valign=CENTER;
}
