// File          : DashPattern.java
// Date          : 1st February 2006
// Last Modified : 16th February 2007
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing dash pattern.
 * @author Nicola L C Talbot
 */
public class DashPattern implements Serializable,Cloneable
{
   /**
    * Initialises using default settings: solid line.
    */
   public DashPattern()
   {
      pattern = null;
      offset = 0.0F;
   }

   /**
    * Initialises using the specified dash pattern with 0 offset.
    * A solid line is indicated by setting dashPattern to null.
    * @param dashPattern the dash pattern (alternating dash length
    * and gap length)
    * @see #DashPattern()
    */
   public DashPattern(float[] dashPattern)
   {
      pattern = dashPattern;
      offset = 0.0F;
   }

   /**
    * Initialises using the specified dash pattern and offset.
    * A solid line is indicated by setting <code>dashPattern</code> to 
    * <code>null</code>.
    * @param dashPattern the dash pattern (alternating dash length
    * and gap length)
    * @param dashOffset the offset
    * @see #DashPattern()
    */
   public DashPattern(float[] dashPattern, float dashOffset)
   {
      pattern = dashPattern;
      offset = dashOffset;
   }

   /**
    * Returns a copy of this object.
    * @return a copy of this object
    */
   public Object clone()
   {
      if (pattern == null)
      {
         return new DashPattern(null, offset);
      }
      float[] dashPattern = new float[pattern.length];
      for (int i=0; i < pattern.length; i++) dashPattern[i]=pattern[i];
      return new DashPattern(dashPattern, offset);
   }

   /**
    * Saves this object in JDR format.
    * @param dout JDR output stream
    * @param version JDR version number
    */
   public void save(DataOutputStream dout, float version)
      throws IOException
   {
      int n = (pattern == null? 0 : pattern.length);

      dout.writeInt(n);
      for (int i = 0; i < n; i++)
      {
         dout.writeFloat(pattern[i]);
      }
      if (n > 0) dout.writeFloat(offset);
   }

   /**
    * Reads a dash pattern from a JDR input stream.
    * @param din JDR input stream
    * @param version JDR version number
    * @throws IOException if there is an I/O error
    * @throws InvalidFormatException if the data in the file is
    * not correctly formatted
    * @return the specified dash pattern
    */
   public static DashPattern read(DataInputStream din, float version)
      throws IOException,InvalidFormatException
   {
      int n = din.readInt();
      if (n == 0)
      {
         return new DashPattern(null, 0.0F);
      }
      else if (n < 0)
      {
         throw new NegativeDashException();
      }
      else
      {
         float[] pat = new float[n];
         for (int i = 0; i < n; i++)
         {
            pat[i] = din.readFloat();
         }
         float off = din.readFloat();

         return new DashPattern(pat, off);
      }
   }

   /**
    * Saves this dash pattern in AJR format.
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    */
   public void saveAJR(PrintWriter out, float version)
      throws IOException
   {
      int n = (pattern == null? 0 : pattern.length);

      AJR.writeInt(out, n);

      for (int i = 0; i < n; i++)
      {
         AJR.writeFloat(out, pattern[i]);
      }

      if (n > 0) AJR.writeFloat(out, offset);
   }

   /**
    * Reads a dash pattern stored in AJR format.
    * @param in the input stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    * @throws java.nio.BufferOverflowException if AJR buffer overflow
    * occurs
    * @throws EOFException if the file ends unexpectedly
    * @throws InvalidFormatException if the data is not stored
    * correctly
    * @return the specified dash pattern
    */
   public static DashPattern readAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      int n = AJR.readInt(in);

      if (n == 0)
      {
         return new DashPattern(null, 0.0F);
      }
      else if (n < 0)
      {
         throw new NegativeDashException();
      }
      else
      {
         float[] pat = new float[n];

         for (int i = 0; i < n; i++)
         {
            pat[i] = AJR.readFloat(in);
         }

         float off = AJR.readFloat(in);

         return new DashPattern(pat, off);
      }
   }

   /**
    * Converts this dash pattern to SVG format.
    * @return the equivalent SVG format representing this dash pattern
    */
   public String svg()
   {
      if (pattern == null) return "stroke-dasharray=\"none\"";

      String str="stroke-dasharray=\""+pattern[0]+"pt";

      for (int i = 1, n=pattern.length; i < n; i++)
      {
         str += ","+pattern[i]+"pt";
      }

      str += "\" stroke-dashoffset=\""+offset+"pt\"";
      return str;
   }

   /**
    * Writes PostScript command to set this dash pattern.
    * @param out the output stream
    * @throws IOException if I/O error occurs
    */
   public void saveEPS(PrintWriter out)
      throws IOException
   {
      if (pattern == null)
      {
         out.println("[] 0 setdash");
      }
      else
      {
         out.print("[");
         for (int i = 0; i < pattern.length; i++)
         {
            out.print(""+pattern[i]+" ");
         }
         out.println("] "+offset+" setdash");
      }
   }

   /**
    * Stores the alternating dash length and gap length for this
    * pattern.
    */
   public float[] pattern;
   /**
    * Stores the offset to the start of the first dash.
    */
   public float offset;
}

