// File          : CanvasSettings.java
// Purpose       : Canvas settings
// Date          : 1st February 2006
// Last Modified : 16th August 2010
// Author        : Nicola L.C. Talbot
//               http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.geom.*;

import java.awt.print.PageFormat;

import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Class representing jpgfdraw's canvas settings.
 * @author Nicola L C Talbot
 */
public class CanvasSettings implements Cloneable
{
   /**
    * Initialises using default settings. The defaults are: no grid, 
    * no grid lock, rectangular grid, select tool, show rulers and 
    * A4 paper.
    */
   public CanvasSettings()
   {
      display_grid         = false;
      grid_lock    = false;
      tool         = ACTION_SELECT;
      showRulers   = true;
      paper        = JDRPaper.A4;

      grid = new JDRRectangularGrid();
   }

   /**
    * Initialises with default settings, except for the display grid
    * setting which is as given.
    * @param showGrid indicates whether or not to display the
    * grid
    * @see #CanvasSettings()
    */
   public CanvasSettings(boolean showGrid)
   {
      this();
      display_grid = showGrid;
   }

   /**
    * Initialises with default settings, except for the display grid
    * and lock grid settings which are as given.
    * @param showGrid indicates whether or not to display the
    * grid
    * @param lockGrid indicates whether or not to lock the grid
    * @see #CanvasSettings()
    */
   public CanvasSettings(boolean showGrid, boolean lockGrid)
   {
      this();
      display_grid = showGrid;
      grid_lock    = lockGrid;
   }

   /**
    * Initialises with default settings, except for the display grid,
    * lock grid and display ruler settings which are as given.
    * @param showGrid indicates whether or not to display the
    * grid
    * @param lockGrid indicates whether or not to lock the grid
    * @param displayRulers indicates whether or not to display the
    * rulers
    * @see #CanvasSettings()
    */

   public CanvasSettings(boolean showGrid, boolean lockGrid,
                        boolean displayRulers)
   {
      this();
      display_grid = showGrid;
      grid_lock    = lockGrid;
      showRulers   = displayRulers;
   }

   /**
    * Initialises with default settings, except for the tool setting
    * which is as given.
    * @param initialTool the current tool setting
    * @see #CanvasSettings()
    * @throws InvalidToolException if the tool setting is invalid
    */

   public CanvasSettings(int initialTool)
     throws InvalidToolException
   {
      this();

      setTool(initialTool);
   }

   /**
    * Initialises with default settings, except for the tool
    * and display grid settings which are as given.
    * @param initialTool the current tool setting
    * @param showGrid determines whether the grid should be displayed
    * @see #CanvasSettings()
    * @throws InvalidToolException if the tool setting is invalid
    */
   public CanvasSettings(int initialTool, boolean showGrid)
     throws InvalidToolException
   {
      this(initialTool);
      display_grid = showGrid;
   }

   /**
    * Initialises with default settings, except for the tool,
    * display grid and lock grid settings which are as given.
    * @param initialTool the current tool setting
    * @param showGrid determines whether the grid should be displayed
    * @param lockGrid determines whether the grid should be locked
    * @see #CanvasSettings()
    * @throws InvalidToolException if the tool setting is invalid
    */
   public CanvasSettings(int initialTool,
                        boolean showGrid, boolean lockGrid)
     throws InvalidToolException
   {
      this(initialTool);
      display_grid = showGrid;
      grid_lock    = lockGrid;
   }

   /**
    * Initialises with the given settings.
    * @param initialTool the tool setting
    * @param showGrid determines whether to display the grid
    * @param lockGrid determines whether to lock the grid
    * @param displayRulers determines whether to display the rulers
    * @param papersize the paper size setting
    * @param normalSize the normal font size
    * @param theGrid the grid
    * @throws InvalidToolException if the tool setting is invalid
    */
   public CanvasSettings(int initialTool, boolean showGrid, 
                        boolean lockGrid,
                        boolean displayRulers,
                        JDRPaper papersize, int normalSize,
                        JDRGrid theGrid)
     throws InvalidToolException
   {
      setTool(initialTool);
      display_grid = showGrid;
      grid_lock    = lockGrid;
      showRulers   = displayRulers;
      paper        = papersize;
      normalsize   = normalSize;
      grid         = theGrid;
   }

   private CanvasSettings(boolean showGrid, 
                        boolean lockGrid,
                        boolean displayRulers,
                        JDRPaper papersize, int normalSize,
                        int initialTool, JDRGrid theGrid)
   {
      display_grid = showGrid;
      grid_lock    = lockGrid;
      showRulers   = displayRulers;
      paper        = papersize;
      normalsize   = normalSize;
      tool         = initialTool;
      grid         = theGrid;
   }

   /**
    * Returns a copy of this object.
    * @return a copy of this object
    */
   public Object clone()
   {
      CanvasSettings s = new CanvasSettings(display_grid,grid_lock,
         showRulers, paper, normalsize, tool, (JDRGrid)grid.clone());

      return s;
   }

   /**
    * Writes this object to the given output stream for the given JDR
    * format.
    * @param dout the output stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    */
   public void save(DataOutputStream dout, float version)
      throws IOException,InvalidFormatException
   {
      dout.writeBoolean(display_grid);
      dout.writeBoolean(grid_lock);
      dout.writeBoolean(showRulers);
      dout.writeInt(tool);
      dout.writeInt(normalsize);
      paper.save(dout, version);

      if (version < 1.6f)
      {
         grid.getListener().writeJDR(grid, dout, version);
      }
      else
      {
         JDRGridLoader loader = JDR.getGridLoader();
         loader.saveJDR(grid, dout, version);
      }
   }

   /**
    * Reads settings from the given input stream for the given JDR
    * format, and returns.
    * @param din the input stream
    * @param version the JDR version number
    * @return the canvas settings read from the input stream
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if the input stream is 
    * incorrectly formatted
    */
   public static CanvasSettings read(DataInputStream din,
      float version)
      throws IOException,InvalidFormatException
   {
      CanvasSettings s = new CanvasSettings();

      read(s, din, version);

      return s;
   }

   /**
    * Reads settings from the given input stream for the given JDR
    * format, and stores in the given <code>CanvasSettings</code> 
    * object.
    * @param s the <code>CanvasSettings</code> object in which to
    * store the data
    * @param din the input stream
    * @param version the JDR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if the input stream is 
    * incorrectly formatted
    */
   public static void read(CanvasSettings s, DataInputStream din,
      float version)
      throws IOException,InvalidFormatException
   {
      s.display_grid  = din.readBoolean();
      s.grid_lock     = din.readBoolean();
      s.showRulers    = din.readBoolean();
      s.tool          = din.readInt();

      if (s.tool < 0 || s.tool >= MAX_TOOLS)
      {
         throw new InvalidToolException(s.tool);
      }

      s.normalsize    = din.readInt();
      s.paper         = JDRPaper.read(din, version);

      if (version < 1.6f)
      {
         byte unitID = din.readByte();

         JDRUnit unit = JDRUnit.getUnit((int)unitID);

         if (unit == null)
         {
            throw new InvalidUnitIDException(unitID);
         }

         int majorDivisions = din.readInt();
         int subDivisions   = din.readInt();
       
         s.grid = new JDRRectangularGrid(unit, majorDivisions,
            subDivisions);
      }
      else
      {
         JDRGridLoader loader = JDR.getGridLoader();
         s.grid = loader.loadJDR(din, version);
      }
   }

   /**
    * Writes this object to the given output stream for the given AJR
    * format.
    * @param out the output stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    */
   public void saveAJR(PrintWriter out, float version)
      throws IOException,InvalidFormatException
   {
      AJR.writeBoolean(out, display_grid);
      AJR.writeBoolean(out, grid_lock);
      AJR.writeBoolean(out, showRulers);
      AJR.writeInt(out, tool);
      AJR.writeInt(out, normalsize);
      paper.saveAJR(out, version);

      if (version < 1.6f)
      {
         grid.getListener().writeAJR(grid, out, version);
      }
      else
      {
         JDRGridLoader loader = AJR.getGridLoader();
         loader.saveAJR(grid, out, version);
      }
   }

   /**
    * Reads settings from the given input stream for the given AJR
    * format, and returns.
    * @param in the input stream
    * @param version the AJR version number
    * @return the canvas settings read from the input stream
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if the input stream is 
    * incorrectly formatted
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * is exceeded
    * @see AJR#buffLength
    */
   public static CanvasSettings readAJR(BufferedReader in, float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      CanvasSettings s = new CanvasSettings();

      readAJR(s, in, version);

      return s;
   }

   /**
    * Reads settings from the given input stream for the given AJR
    * format, and stores in the given <code>CanvasSettings</code>
    * object.
    * @param s the <code>CanvasSettings</code> object in which to
    * store the data
    * @param in the input stream
    * @param version the AJR version number
    * @throws IOException if I/O error occurs
    * @throws InvalidFormatException if the input stream is 
    * incorrectly formatted
    * @throws java.nio.BufferOverflowException if the AJR buffer
    * is exceeded
    * @see AJR#buffLength
    */
   public static void readAJR(CanvasSettings s, BufferedReader in,
      float version)
      throws IOException,InvalidFormatException,
             java.nio.BufferOverflowException,
             EOFException
   {
      s.display_grid          = AJR.readBoolean(in);
      s.grid_lock     = AJR.readBoolean(in);
      s.showRulers    = AJR.readBoolean(in);
      s.tool          = AJR.readInt(in);

      if (s.tool < 0 || s.tool >= MAX_TOOLS)
      {
         throw new InvalidToolException(s.tool, AJR.getLineNum());
      }

      s.normalsize    = AJR.readInt(in);

      s.paper = JDRPaper.readAJR(in, version);

      if (version < 1.6f)
      {
         int unitID = AJR.readInt(in);

         JDRUnit unit = JDRUnit.getUnit(unitID);

         if (unit == null)
         {
            throw new InvalidUnitIDException(unitID, AJR.getLineNum());
         }

         int majorDivisions = AJR.readInt(in);
         int subDivisions   = AJR.readInt(in);
       
         s.grid = new JDRRectangularGrid(unit, majorDivisions,
            subDivisions);
      }
      else
      {
         JDRGridLoader loader = AJR.getGridLoader();
         s.grid = loader.loadAJR(in, version);
      }
   }

   /**
    * Sets the current tool.
    * @param toolSetting the new tool
    * @throws InvalidToolException if the given tool setting is invalid
    */
   public void setTool(int toolSetting)
      throws InvalidToolException
   {
      if (toolSetting < 0 || toolSetting >= MAX_TOOLS)
      {
         throw new InvalidToolException(toolSetting);
      }

      tool = toolSetting;
   }

   /**
    * Sets the current tool using the given string. The given
    * string must be one of the following: "select", "open_line",
    * "closed_line", "open_curve", "closed_curve", "rectangle",
    * "ellipse" or "text".
    * @param string the string identifying the tool
    * @throws InvalidToolException if the identifying string is 
    * not one of those listed
    */
   public void setTool(String string)
      throws InvalidToolException
   {
      if (string.equals("select"))
      {
         tool = ACTION_SELECT;
      }
      else if (string.equals("open_line"))
      {
         tool = ACTION_OPEN_LINE;
      }
      else if (string.equals("closed_line"))
      {
         tool = ACTION_CLOSED_LINE;
      }
      else if (string.equals("open_curve"))
      {
         tool = ACTION_OPEN_CURVE;
      }
      else if (string.equals("closed_curve"))
      {
         tool = ACTION_CLOSED_CURVE;
      }
      else if (string.equals("rectangle"))
      {
         tool = ACTION_RECTANGLE;
      }
      else if (string.equals("ellipse"))
      {
         tool = ACTION_ELLIPSE;
      }
      else if (string.equals("text"))
      {
         tool = ACTION_TEXT;
      }
      else
      {
         throw new InvalidToolException(string);
      }
   }

   /**
    * Gets the current tool.
    * @return the current tool
    */
   public int getTool()
   {
      return tool;
   }

   /**
    * Gets the string identifying the current tool.
    * @return the string identifying the current tool
    * @see #setTool(String)
    */
   public String getToolString()
   {
      switch (tool)
      {
         case ACTION_SELECT : return "select";
         case ACTION_OPEN_LINE : return "open_line";
         case ACTION_CLOSED_LINE : return "closed_line";
         case ACTION_OPEN_CURVE : return "open_curve";
         case ACTION_CLOSED_CURVE : return "closed_curve";
         case ACTION_RECTANGLE : return "rectangle";
         case ACTION_ELLIPSE : return "ellipse";
         case ACTION_TEXT : return "text";
      }

      return "";
   }

   public void setGrid(JDRGrid theGrid)
   {
      grid = theGrid;
   }

   public JDRGrid getGrid()
   {
      return grid;
   }

   public boolean isGridDisplayed()
   {
      return display_grid;
   }

   public void setDisplayGrid(boolean showGrid)
   {
      display_grid = showGrid;
   }

   public void setGridLock(boolean lockGrid)
   {
      grid_lock = lockGrid;
   }

   public boolean isGridLocked()
   {
      return grid_lock;
   }

   public void setMargins(PageFormat pf)
   {
      paper.setImageableArea(pf);
   }

   public Rectangle2D getImageableArea()
   {
      return paper.getImageableArea();
   }

   /**
    * Indicates whether to show the grid.
    */
   public boolean display_grid;
   /**
    * Indicates whether to lock the grid.
    */
   public boolean grid_lock;
   /**
    * Indicates whether to show the rulers.
    */
   public boolean showRulers;

   /**
    * Indicates the current tool.
    */
   private int tool;

   /**
    * Stores the current paper.
    */
   public JDRPaper paper;

   /**
    * Stores the value of the LaTeX normal font size.
    */
   public int normalsize=10;

   /**
    * Stores the grid.
    */
   private JDRGrid grid;

   /**
    * Indicates the select tool.
    */
   public static final int ACTION_SELECT=0;
   /**
    * Indicates the open line tool.
    */
   public static final int ACTION_OPEN_LINE=1;
   /**
    * Indicates the closed line tool.
    */
   public static final int ACTION_CLOSED_LINE=2;
   /**
    * Indicates the open curve tool.
    */
   public static final int ACTION_OPEN_CURVE=3;
   /**
    * Indicates the closed curve tool.
    */
   public static final int ACTION_CLOSED_CURVE=4;
   /**
    * Indicates the rectangle tool.
    */
   public static final int ACTION_RECTANGLE=5;
   /**
    * Indicates the ellipse tool.
    */
   public static final int ACTION_ELLIPSE=6;
   /**
    * Indicates the text tool.
    */
   public static final int ACTION_TEXT=7;
   /**
    * Indicates the total number of tool settings.
    */
   public static final int MAX_TOOLS=8;
}

