// File          : BBox.java
// Date          : 1st February 2006
// Last Modified : 13th March 2008
// Author        : Nicola L.C. Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

package uk.ac.uea.cmp.nlct.jdr;

import java.io.*;
import java.awt.*;
import java.awt.geom.*;

/**
 * Bounding box information.
 * Stores the bounding box information for a given {@link JDRObject}.
 * A bounding box has eight "hotspots" given by a {@link JDRPoint}.
 * These regions are shown in red in <a href="#fig1">Figure 1</a>.
 * <p>
 * <center>
 * <table width=60%>
 * <tr align=center><td colspan=2>
 * <img src="images/bbox.png" alt="[image of box with red square regions at the 8 compass points]"></td></tr>
 * <tr><th valign=top>Figure&nbsp;1</th>
 * <td>A bounding box has eight "hotspot" regions (shown in red)</td>
 * </table>
 * </center>
 * JpgfDraw only uses six of these eight hotspots.
 *
 * @version 0.3.1b 13 March 2008
 * @author Nicola L C Talbot
 */
public class BBox implements Cloneable,Serializable
{
   /**
    * Construct bounding box from two opposing points.
    * @param p1 first point
    * @param p2 diagonally opposite point
    */
   public BBox(Point p1, Point p2)
   {
      minX_ = p1.x < p2.x ? p1.x : p2.x;
      minY_ = p1.y < p2.y ? p1.y : p2.y;
      maxX_ = p1.x < p2.x ? p2.x : p1.x; 
      maxY_ = p1.y < p2.y ? p2.y : p1.y;

      initHotspots();
   }

   /**
    * Construct bounding box from two opposing points.
    * @param p1 first point
    * @param p2 diagonally opposite point
    */
   public BBox(Point2D p1, Point2D p2)
   {
      minX_ = p1.getX() < p2.getX() ?  p1.getX() : p2.getX();
      minY_ = p1.getY() < p2.getY() ?  p1.getY() : p2.getY();
      maxX_ = p1.getX() < p2.getX() ?  p2.getX() : p1.getX(); 
      maxY_ = p1.getY() < p2.getY() ?  p2.getY() : p1.getY();

      initHotspots();
   }

   /**
    * Construct bounding box from two opposing points.
    * @param p1 first point
    * @param p2 diagonally opposite point
    */
   public BBox(JDRPoint p1, JDRPoint p2)
   {
      minX_ = p1.x < p2.x ? p1.x : p2.x;
      minY_ = p1.y < p2.y ? p1.y : p2.y;
      maxX_ = p1.x < p2.x ? p2.x : p1.x; 
      maxY_ = p1.y < p2.y ? p2.y : p1.y;

      initHotspots();
   }

   /**
    * Construct bounding box from minimum and maximum values.
    * @param minX minimum horizontal extent
    * @param minY minimum vertical extent
    * @param maxX maximum horizontal extent
    * @param maxY maximum vertical extent
    */
   public BBox(int minX, int minY, int maxX, int maxY)
   {
      minX_ = minX;
      minY_ = minY;
      maxX_ = maxX;
      maxY_ = maxY;

      initHotspots();
   }

   /**
    * Construct bounding box from minimum and maximum values.
    * @param minX minimum horizontal extent
    * @param minY minimum vertical extent
    * @param maxX maximum horizontal extent
    * @param maxY maximum vertical extent
    */
   public BBox(double minX, double minY, double maxX, double maxY)
   {
      minX_ = minX;
      minY_ = minY;
      maxX_ = maxX;
      maxY_ = maxY;

      initHotspots();
   }

   /**
    * Construct bounding box from given rectangle.
    * @param r rectangle describing bounding box extent
    */
   public BBox(Rectangle2D r)
   {
      this(r.getX(), r.getY(),
           r.getX()+r.getWidth(), r.getY()+r.getHeight());
   }

   /**
    * Initialise the hotspots.
    */
   private void initHotspots()
   {
      double midX = getMidX();
      double midY = getMidY();

      hotspotS  = new JDRPoint(midX, maxY_);
      hotspotSE = new JDRPoint(maxX_, maxY_);
      hotspotE  = new JDRPoint(maxX_, midY);
      hotspotNE = new JDRPoint(maxX_, minY_);
      hotspotN  = new JDRPoint(midX, minY_);
      hotspotNW = new JDRPoint(minX_, minY_);
      hotspotW  = new JDRPoint(minX_, midY);
      hotspotSW = new JDRPoint(minX_, maxY_);
   }

   /**
    * Reset the hotspots. (Required when bounding box changes size.
    */
   private void resetHotspots()
   {
      double midX = getMidX();
      double midY = getMidY();

      hotspotS.set(midX, maxY_);
      hotspotSE.set(maxX_, maxY_);
      hotspotE.set(maxX_, midY);
      hotspotNE.set(maxX_, minY_);
      hotspotN.set(midX, minY_);
      hotspotNW.set(minX_, minY_);
      hotspotW.set(minX_, midY);
      hotspotSW.set(minX_, maxY_);
   }

   /**
    * Determines if this bounding box intersects the given 
    * rectangle.
    * @param rect the rectangle
    * @return true if this bounding box intersects the given
    * rectangle
    */
   public boolean intersects(Rectangle rect)
   {
      double x0 = rect.getX();
      double y0 = rect.getY();
      double x1 = x0 + rect.getWidth();
      double y1 = y0 + rect.getHeight();

      if (x1 < minX_) return false;

      if (maxX_ < x0) return false;

      if (y1 < minY_) return false;

      if (maxY_ < y0) return false;

      return true;
   }

   /**
    * Determines if this bounding box intersects the given 
    * rectangle.
    * @param rect the rectangle
    * @return true if this bounding box intersects the given
    * rectangle
    */
   public boolean intersects(Rectangle2D rect)
   {
      double x0 = rect.getX();
      double y0 = rect.getY();
      double x1 = x0 + rect.getWidth();
      double y1 = y0 + rect.getHeight();

      if (x1 < minX_) return false;

      if (maxX_ < x0) return false;

      if (y1 < minY_) return false;

      if (maxY_ < y0) return false;

      return true;
   }

   /**
    * Determines if given point is inside this bounding box.
    * @return <code>true</code> if point inside this bounding box, 
    * otherwise <code>false</code>
    * @param p point in question
    */
   public boolean contains(Point p)
   {
      if (p.x >= minX_-1 && p.x <= maxX_
       && p.y >= minY_-1 && p.y <= maxY_)
      {
         return true;
      }

      return false;
   }

   /**
    * Determines if given point is inside this bounding box.
    * @return <code>true</code> if point inside this bounding box, 
    * otherwise <code>false</code>
    * @param p point in question
    */
   public boolean contains(Point2D p)
   {
      if (p.getX() >= minX_-1 && p.getX() <= maxX_
       && p.getY() >= minY_-1 && p.getY() <= maxY_)
      {
         return true;
      }

      return false;
   }

   /**
    * Gets the height of this bounding box.
    * @return the height of this bounding box
    */
   public double getHeight()
   {
      return Math.abs(maxY_-minY_);
   }

   /**
    * Gets the width of this bounding box.
    * @return the width of this bounding box
    */
   public double getWidth()
   {
      return Math.abs(maxX_-minX_);
   }

   /**
    * Gets the minimum horizontal extent of this bounding box.
    * @return the minimum horizontal extent
    */
   public double getMinX()
   {
      return minX_;
   }

   /**
    * Gets the minimum vertical extent of this bounding box.
    * @return the minimum vertical extent
    */
   public double getMinY()
   {
      return minY_;
   }

   /**
    * Gets the maximum horizontal extent of this bounding box.
    * @return the maximum horizontal extent
    */
   public double getMaxX()
   {
      return maxX_;
   }

   /**
    * Gets the maximum vertical extent of this bounding box.
    * @return the maximum vertical extent
    */
   public double getMaxY()
   {
      return maxY_;
   }

   /**
    * Gets the horizontal mid point of this bounding box.
    * @return the horizontal mid point
    */
   public double getMidX()
   {
      return minX_+0.5*getWidth();
   }

   /**
    * Gets the vertical mid point of this bounding box.
    * @return the horizontal mid point
    */
   public double getMidY()
   {
      return minY_+0.5*getHeight();
   }

   /**
    * Translates this bounding box.
    * @param x horizontal shift
    * @param y vertical shift
    */
   public void translate(double x, double y)
   {
      minX_ -= x;
      maxX_ -= x;
      minY_ -= y;
      maxY_ -= y;
   }

   /**
    * Resets the extent of this bounding box.
    * @param minX minimum horizontal extent
    * @param minY minimum vertical extent
    * @param maxX maximum horizontal extent
    * @param maxY maximum vertical extent
    */
   public void reset(double minX, double minY, double maxX, double maxY)
   {
      minX_ = minX;
      minY_ = minY;
      maxX_ = maxX;
      maxY_ = maxY;

      resetHotspots();
   }

   /**
    * Enlarges this box so that it encompasses another bounding
    * box.
    * @param bbox the other bounding box
    * @see #add(BBox)
    */
   public void encompass(BBox bbox)
   {
      double minX = bbox.getMinX();
      double minY = bbox.getMinY();
      double maxX = bbox.getMaxX();
      double maxY = bbox.getMaxY();

      double thisminX = getMinX();
      double thisminY = getMinY();
      double thismaxX = getMaxX();
      double thismaxY = getMaxY();

      if (minX > thisminX) minX = thisminX;
      if (minY > thisminY) minY = thisminY;
      if (maxX < thismaxX) maxX = thismaxX;
      if (maxY < thismaxY) maxY = thismaxY;

      minX_ = minX;
      minY_ = minY;
      maxX_ = maxX;
      maxY_ = maxY;
   }

   /**
    * Gets the smallest bounding box that encompasses both this
    * bounding box and another bounding box.
    * @param bbox the other bounding box
    * @return bounding box encompassing both this and the other 
    * bounding box
    */
   public BBox add(BBox bbox)
   {
      double minX = bbox.getMinX();
      double minY = bbox.getMinY();
      double maxX = bbox.getMaxX();
      double maxY = bbox.getMaxY();

      double thisminX = getMinX();
      double thisminY = getMinY();
      double thismaxX = getMaxX();
      double thismaxY = getMaxY();

      if (minX > thisminX) minX = thisminX;
      if (minY > thisminY) minY = thisminY;
      if (maxX < thismaxX) maxX = thismaxX;
      if (maxY < thismaxY) maxY = thismaxY;

      return new BBox(minX, minY, maxX, maxY);
   }

   /**
    * Sets this bounding box so that it is the smallest bounding box
    * that encompasses both the original version of this bounding
    * box and another bounding box. This is like add(BBox) but
    * doesn't make a new object.
    * @param bbox the other bounding box
    */
   public void merge(BBox bbox)
   {
      merge(bbox.getMinX(), bbox.getMinY(), bbox.getMaxX(), bbox.getMaxY());
   }

   public void merge(double minX, double minY, double maxX, double maxY)
   {
      double thisminX = getMinX();
      double thisminY = getMinY();
      double thismaxX = getMaxX();
      double thismaxY = getMaxY();

      if (minX > thisminX) minX = thisminX;
      if (minY > thisminY) minY = thisminY;
      if (maxX < thismaxX) maxX = thismaxX;
      if (maxY < thismaxY) maxY = thismaxY;
      
      reset(minX, minY, maxX, maxY);
   }

   /**
    * Sets this bounding box so that it is the smallest bounding box
    * that encompasses both this bounding
    * box and the given rectangle.
    * @param rect the given rectangle
    */
   public void merge(Rectangle2D rect)
   {
      double minX = rect.getX();
      double minY = rect.getY();
      double maxX = minX + rect.getWidth();
      double maxY = minY + rect.getHeight();

      merge(minX, minY, maxX, maxY);
   }

   /**
    * If the given point isn't contained in this bounding box, this
    * box is expanded to contain the point.
    * @param point the point that needs to be added to this bounding
    * box
    */
   public void merge(Point2D point)
   {
      merge(point.getX(), point.getY(),
            point.getX(), point.getY());
   }

   /**
    * Gets this bounding box as a rectangle, including area
    * taken up by hotspots.
    * @return rectangle encompassing this bounding box and its
    * hotspots
    */
   public Rectangle getRectangle()
   {
      double d = JDRPoint.pointSize/2+1;

      return new Rectangle((int)Math.floor(getMinX()-d),
                           (int)Math.floor(getMinY()-d),
                           (int)Math.ceil(getWidth()+2*d+2),
                           (int)Math.ceil(getHeight()+2*d+2));
   }

   /**
    * Gets this bounding box as a rectangle, 
    * including hotspots, and scaled.
    * @param scale scaling factor
    * @return rectangle encompassing this bounding box and its
    * hotspots
    */
   public Rectangle getRectangle(double scale)
   {
      double d = JDRPoint.pointSize/2+1;

      return new Rectangle((int)Math.floor((getMinX()-d-1)*scale),
                           (int)Math.floor((getMinY()-d-1)*scale),
                           (int)Math.ceil((getWidth()+2*d+2)*scale),
                           (int)Math.ceil((getHeight()+2*d+2)*scale));
   }

   /**
    * Draws this bounding box on given graphics object.
    * (The hotspots are not drawn.)
    * @param g Graphics object
    */
   public void draw(Graphics g)
   {
      draw(g,(short)0);
   }

   /**
    * Draws this bounding box on given graphics object
    * possibly including its hotspots. This just calls {@link
    * #draw(Graphics,double,short)} with the scale factor set to
    * 1.0.
    */
   public void draw(Graphics g, short hotspotFlags)
   {
      draw(g, 1.0, hotspotFlags);
   }

   public void draw(Graphics g, double scale)
   {
      draw(g, scale, (short)0);
   }

   /**
    * Draws this bounding box on given graphics object
    * possibly including its hotspots. The hotspots to be
    * drawn are given in the bits of <code>hotspotFlags</code>.
    * For example, JpgfDraw uses the value
    * <pre>
    * hotspotFlags = BBox.SOUTH
    *              | BBox.SOUTH_EAST
    *              | BBox.EAST
    *              | BBox.NORTH_EAST
    *              | BBox.NORTH_WEST
    *              | BBox.SOUTH_WEST
    * </pre>
    * when it is configured to show the hotspots, and uses 
    * <code>hotspotFlags=0</code> otherwise. The co-ordinates of the
    * bounding box are scaled by the given factor. Note that this
    * doesn't alter the graphics device transformation matrix.
    * @param g Graphics object
    * @param scale the scaling factor
    * @param hotspotFlags which hotspots should also be drawn
    */
   public void draw(Graphics g, double scale, short hotspotFlags)
   {
      Graphics2D g2 = (Graphics2D)g;
      Stroke oldS = g2.getStroke();
      Paint oldPaint = g2.getPaint();
      g2.setPaint(outlineColour);

      Rectangle2D rect = new Rectangle2D.Double
         (minX_*scale-0.5, minY_*scale-0.5, 
          getWidth()*scale+1, getHeight()*scale+1);

      g2.draw(rect);

      float[] dashPattern = new float[2];

      dashPattern[0] = 6.0f;
      dashPattern[1] = 6.0f;
      g2.setPaint(dashColour);
      BasicStroke s = new BasicStroke(1.0f,
                          BasicStroke.CAP_BUTT,
                          BasicStroke.JOIN_MITER,
                          10.0f,dashPattern,5.0f);
      g2.setStroke(s);
      g2.draw(rect);

      g2.setStroke(oldS);

      if ((hotspotFlags & SOUTH)==SOUTH)
      {
         hotspotS.draw(g, scale);
      }

      if ((hotspotFlags & SOUTH_EAST)==SOUTH_EAST)
      {
         hotspotSE.draw(g, scale);
      }

      if ((hotspotFlags & EAST)==EAST)
      {
         hotspotE.draw(g, scale);
      }

      if ((hotspotFlags & NORTH_EAST)==NORTH_EAST)
      {
         hotspotNE.draw(g, scale);
      }

      if ((hotspotFlags & NORTH)==NORTH)
      {
         hotspotN.draw(g, scale);
      }

      if ((hotspotFlags & NORTH_WEST)==NORTH_WEST)
      {
         hotspotNW.draw(g, scale);
      }

      if ((hotspotFlags & WEST)==WEST)
      {
         hotspotW.draw(g, scale);
      }

      if ((hotspotFlags & SOUTH_WEST)==SOUTH_WEST)
      {
         hotspotSW.draw(g, scale);
      }

      g2.setPaint(oldPaint);
   }

   /**
    * Returns a copy of this object.
    * @return copy of this object
    */
   public Object clone()
   {
      return new BBox(minX_, minY_, maxX_, maxY_);
   }

   /**
    * Makes this bounding box equal to the other.
    * @param bbox the other bounding box
    */
   public void makeEqual(BBox bbox)
   {
      minX_ = bbox.getMinX();
      minY_ = bbox.getMinY();
      maxX_ = bbox.getMaxX();
      maxY_ = bbox.getMaxY();
   }

   public boolean equals(Object obj)
   {
      if (this == obj) return true;
      if (obj == null) return false;
      if (!(obj instanceof BBox)) return false;

      BBox b = (BBox) obj;

      return (minX_ == b.minX_
           && minY_ == b.minY_
           && maxX_ == b.maxX_
           && maxY_ == b.maxY_);
   }

   /**
    * Gets the top left hotspot.
    * @return the top left hotspot
    */
   public JDRPoint getTopLeft()
   {
      return hotspotNW;
   }

   /**
    * Gets the bottom left hotspot.
    * @return the bottom left hotspot
    */
   public JDRPoint getBottomLeft()
   {
      return hotspotSW;
   }

   /**
    * Gets the bottom right hotspot.
    * @return the bottom right hotspot
    */
   public JDRPoint getBottomRight()
   {
      return hotspotSE;
   }

   /**
    * Gets the top right hotspot.
    * @return the top right hotspot
    */
   public JDRPoint getTopRight()
   {
      return hotspotNE;
   }

   /**
    * Gets the mid point of this bounding box.
    * @return mid point
    */
   public JDRPoint getCentre()
   {
      return new JDRPoint(getMidX(), getMidY());
   }

   /**
    * Gets number identifying hotspot that contains the given point.
    * Uses {@link JDRPoint#contains(Point2D)} to determine whether
    * the given point is inside a hotspot. Possible values are:
    * <code>HOTSPOT_SE</code>, <code>HOTSPOT_SW</code>,
    * <code>HOTSPOT_NE</code>, <code>HOTSPOT_NW</code>,
    * <code>HOTSPOT_SW</code>, <code>HOTSPOT_E</code>,
    * <code>HOTSPOT_S</code> or <code>HOTSPOT_NONE</code>.
    * @return the identifier of the hotspot that contains given point
    * or <code>HOTSPOT_NONE</code> if none of this bounding box's
    * hotspots contain given point.
    * @param p point in question
    */
   public int getHotspot(Point2D p)
   {
      if (hotspotSE.contains(p))
      {
         return HOTSPOT_SE;
      }
      else if (hotspotSW.contains(p))
      {
         return HOTSPOT_SW;
      }
      else if (hotspotNE.contains(p))
      {
         return HOTSPOT_NE;
      }
      else if (hotspotNW.contains(p))
      {
         return HOTSPOT_NW;
      }
      else if (hotspotNW.contains(p))
      {
         return HOTSPOT_SW;
      }
      else if (hotspotE.contains(p))
      {
         return HOTSPOT_E;
      }
      else if (hotspotS.contains(p))
      {
         return HOTSPOT_S;
      }

      return HOTSPOT_NONE;
   }

   /**
    * Gets a string representation of this bounding box.
    * This is of the form BBox:[minX, minY, maxX, maxY].
    */
   public String toString()
   {
      return new String("BBox:["+minX_+" "+minY_+" "
                      +maxX_+" "+maxY_+"]");
   }

   public String info()
   {
      return new String("minX="+minX_+", minY="+minY_+", maxX="
                      +maxX_+", maxY="+maxY_);
   }

   /**
    * This bounding box's extents.
    */
   private double minX_, minY_, maxX_, maxY_;

   //public static int hotspotD=1;

   /**
    * No hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point does not 
    * lie in any of the hotspot regions.
    */
   public static final int HOTSPOT_NONE=0;
   /**
    * East hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the East (centre right) hotspot region.
    */
   public static final int HOTSPOT_E=1;
   /**
    * South-East hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the South-East (lower right) hotspot region.
    */
   public static final int HOTSPOT_SE=2;
   /**
    * South hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the South (lower middle) hotspot region.
    */
   public static final int HOTSPOT_S=3;
   /**
    * South-West hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the South-West (lower left) hotspot region.
    */
   public static final int HOTSPOT_SW=4;
   /**
    * West hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the West (centre left) hotspot region.
    */
   public static final int HOTSPOT_W=5;
   /**
    * North-West hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the North-West (upper left) hotspot region.
    */
   public static final int HOTSPOT_NW=6;
   /**
    * North hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the North (upper middle) hotspot region.
    */
   public static final int HOTSPOT_N=7;
   /**
    * North-East hotspot. This value is returned by
    * {@link #getHotspot(Point2D)} if the given point 
    * lies in the North-East (upper right) hotspot region.
    */
   public static final int HOTSPOT_NE=8;

   /**
    * Hotspot region.
    */
   private JDRPoint hotspotS, hotspotSE, hotspotE,
                    hotspotNE, hotspotNW, hotspotSW,
                    hotspotN, hotspotW;

   /**
    * Hotspot flag.
    * @see #draw(Graphics, short)
    */
   public static final short NORTH      = 0x80,
                             NORTH_EAST = 0x40,
                             EAST       = 0x20,
                             SOUTH_EAST = 0x10,
                             SOUTH      = 0x8,
                             SOUTH_WEST = 0x4,
                             WEST       = 0x2,
                             NORTH_WEST = 0x1;

//   private static Color dashColour = new Color(192,192,192,100);
   private static Color dashColour = Color.white;
   private static Color outlineColour = new Color(255,0,0,200);
}

