// File          : Eps2Jdr.java
// Description   : command line eps to jdr converter
// Date          : 5 March 2007
// Last Modified : 1 October 2009
// Author        : Nicola L C Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.eps2jdr;

import java.io.*;
import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

import uk.ac.uea.cmp.nlct.jdrresources.*;

/**
 * EPS to JDR command line converter.
 * v1.1: add -normalsize switch
*/
public class Eps2Jdr
{
   /**
    * Prints version information to STDERR.
    */
   public static void appVersion()
   {
      System.err.println("eps2jdr 1.5");
      System.err.println("Copyright (C) 2007 Nicola L C Talbot");
      System.err.println("This is free software distributed under the GNU General Public License.");
      System.err.println("There is NO WARRANTY. See accompanying licence file for details.");
   }

   /**
    * Prints syntax to STDERR and exits.
    */
   public static void syntax()
   {
         System.err.println("Syntax: eps2jdr [options] <eps file> <jdr file>");
         System.err.println("Options:");
         System.err.println("-v1.0\tSave as JDR version 1.0");
         System.err.println("-v1.1\tSave as JDR version 1.1");
         System.err.println("-v1.2\tSave as JDR version 1.2");
         System.err.println("-v1.3\tSave as JDR version 1.3");
         System.err.println("-v1.4\tSave as JDR version 1.4");
         System.err.println("-v1.4\tSave as JDR version 1.5 (default)");
         System.err.println("-bitmap <basename>\tExtracted bitmaps to be saved as <basename><n>.png\n");
         System.err.println("-normalsize <n>\tSet normal LaTeX font size to <n>\n");
         System.err.println("-verbose\tVerbose output");
         System.err.println("-version\tPrint current version and exit");
         System.exit(0);
   }

   public static void syntaxError(String message)
   {
      System.err.println(message);
      System.err.println("Use '-help' for help");

      System.exit(1);
   }

   public static void main(String[] args)
   {
      String epsFile = null;
      String jdrFile = null;
      String bitmapBase = null;
      int verbose = 0;
      float version = JDR.currentVersion;
      int normalsize=10;

      for (int i = 0; i < args.length; i++)
      {
         if (args[i].equals("-v1.0"))
         {
            version = 1.0f;
         }
         else if (args[i].equals("-v1.1"))
         {
            version = 1.1f;
         }
         else if (args[i].equals("-v1.2"))
         {
            version = 1.2f;
         }
         else if (args[i].equals("-v1.3"))
         {
            version = 1.3f;
         }
         else if (args[i].equals("-v1.4"))
         {
            version = 1.4f;
         }
         else if (args[i].equals("-v1.5"))
         {
            version = 1.5f;
         }
         else if (args[i].equals("-version"))
         {
            appVersion();
            System.exit(0);
         }
         else if (args[i].equals("-help"))
         {
            syntax();
         }
         else if (args[i].equals("-bitmap"))
         {
            i++;
            if (i >= args.length)
            {
               syntaxError("missing -bitmap basename");
            }
            bitmapBase = args[i]; 
         }
         else if (args[i].equals("-normalsize"))
         {
            i++;
            if (i >= args.length)
            {
               syntaxError("missing -normalsize value");
            }
            try
            {
               normalsize = Integer.parseInt(args[i]); 
            }
            catch (NumberFormatException e)
            {
               System.err.println(
                  "Integer expected for -normalsize, '"+args[i]+"' found");
               System.exit(1);
            }
         }
         else if (args[i].equals("-verbose"))
         {
            verbose = 1;
         }
         else if (args[i].startsWith("-"))
         {
            syntaxError("Unknown switch '"+args[i]+"'");
         }
         else if (epsFile == null)
         {
            epsFile = args[i];
         }
         else if (jdrFile == null)
         {
            jdrFile = args[i];
         }
         else
         {
            syntaxError("Too many file names specified");
         }
      }

      if (verbose == 0)
      {
         JDR.messageSystem.suspend();
      }
      else
      {
         JDR.messageSystem.displayMessages();
      }

      if (epsFile == null)
      {
         syntaxError("Missing input file");
      }
      else if (jdrFile == null)
      {
         syntaxError("Missing output file");
      }

      if (bitmapBase == null)
      {
         int idx = jdrFile.lastIndexOf(".jdr");

         if (idx == -1)
         {
            bitmapBase = jdrFile;
         }
         else
         {
            bitmapBase = jdrFile.substring(0, idx);
         }
      }

      String userconfigdir = JDRResources.getUserConfigDir();

      if (userconfigdir != null)
      {
         // load PostScript font mappings

         String psmappings=null;

         File file = new File(userconfigdir, "psfontmap");

         try
         {
            psmappings = file.getCanonicalPath();

            if (psmappings != null && file.exists()
                && file.isFile())
            {
               try
               {
                  LaTeXFont.loadPostScriptMappings(file);
               }
               catch (InvalidFormatException e)
               {
                  JDRResources.error(
                     new String[]{"'"+psmappings+"':",
                                  e.getMessage()});
               }
            }
         }
         catch (IOException e)
         {
            JDRResources.error(
               new String[]{"'"+psmappings+"':",
                            e.getMessage()});
         }
      }

      BufferedReader in=null;

      try
      {
         in = new BufferedReader(new FileReader(epsFile));
      }
      catch (IOException e)
      {
         System.err.println("Unable to open '"+epsFile
            +"' "+e.getMessage());
         System.exit(1);
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      JDR.messageSystem.messageln("Reading '"+epsFile+"'");

      JDRGroup paths=null;

      try
      {
         paths = EPS.load(in, bitmapBase, verbose, normalsize);
      }
      catch (InvalidFormatException e)
      {
         System.err.println(e.getMessage());
      }
      catch (IOException e)
      {
         System.err.println(e.getMessage());
      }
      catch (Exception e)
      {
         System.err.println(e.getMessage());
         e.printStackTrace();
      }

      try
      {
         in.close();
      }
      catch (Exception e)
      {
         //System.err.println(e);
         e.printStackTrace();
         System.exit(1);
      }

      if (paths == null || paths.size() == 0)
      {
         System.err.println("no image created: nothing to save");
         System.exit(1);
      }

      CanvasSettings settings = new CanvasSettings();

      settings.normalsize = normalsize;

      BBox box = paths.getBBox();

      settings.paper = JDRPaper.getClosestEnclosingPredefinedPaper(
         box.getWidth(), box.getHeight(), version);

      DataOutputStream out=null;

      try
      {
         out = new DataOutputStream(new FileOutputStream(jdrFile));
      }
      catch (IOException e)
      {
         System.err.println("Unable to open '"+jdrFile
            +"' "+e.getMessage());
         System.exit(1);
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      JDR.messageSystem.messageln("");
      JDR.messageSystem.messageln("writing '"+jdrFile+"'");

      try
      {
         JDR.save(paths, settings, out, version, JDR.ALL_SETTINGS);
      }
      catch (Exception e)
      {
         //System.err.println(e);
         e.printStackTrace();
         System.exit(1);
      }

      try
      {
         out.close();
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      JDR.messageSystem.messageln("");
   }
}
