// File          : Ajr2Jdr.java
// Description   : command line ajr to jdr converter
// Date          : 6 February 2007
// Last Modified : 1 October 2009
// Author        : Nicola L C Talbot
//                 http://theoval.cmp.uea.ac.uk/~nlct/

/*
    Copyright (C) 2006 Nicola L.C. Talbot

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
package uk.ac.uea.cmp.nlct.ajr2jdr;

import java.io.*;
import uk.ac.uea.cmp.nlct.jdr.*;
import uk.ac.uea.cmp.nlct.jdr.io.*;

/**
 * Command line AJR to JDR converter.
 * @author Nicola L C Talbot
 */
public class Ajr2Jdr
{
   /**
    * Prints version information to STDERR.
    */
   public static void appVersion()
   {
      System.err.println("ajr2jdr 1.5");
      System.err.println("Copyright (C) 2007 Nicola L C Talbot");
      System.err.println("This is free software distributed under the GNU General Public License.");
      System.err.println("There is NO WARRANTY. See accompanying licence file for details.");
   }

   /**
    * Prints syntax information to STDERR and quits.
    */
   public static void syntax()
   {
         System.err.println("Syntax: ajr2jdr [options] <ajr file> <jdr file>");
         System.err.println("Options:");
         System.err.println("-v1.0\tSave as JDR version 1.0");
         System.err.println("-v1.1\tSave as JDR version 1.1");
         System.err.println("-v1.2\tSave as JDR version 1.2");
         System.err.println("-v1.3\tSave as JDR version 1.3");
         System.err.println("-v1.4\tSave as JDR version 1.4");
         System.err.println("-v1.5\tSave as JDR version 1.5 (default)");
         System.err.println("-settings [<n>]\tControls how settings are saved in JDR file. Optionally followed by integer <n> (if omitted, <n>=1):");
         System.err.println("\t0 : don't save settings");
         System.err.println("\t1 : save all settings");
         System.err.println("\t2 : only save paper size (v1.3 and above, for earlier versions equivalent to 0)");
         System.err.println("-nosettings\tEquivalent to -settings 0");
         System.err.println("-settings_as_input\tOnly save settings if they were given in the input file (Default)");
         System.exit(0);
   }


   public static void main(String[] args)
   {
      String ajrFile = null;
      String jdrFile = null;
      int settingsFlag = JDR.ALL_SETTINGS;
      float version = JDR.currentVersion;
      boolean settingsAsInput = true;

      JDR.messageSystem.suspend();

      for (int i = 0; i < args.length; i++)
      {
         if (args[i].equals("-nosettings"))
         {
            settingsFlag = JDR.NO_SETTINGS;
            settingsAsInput = false;
         }
         else if (args[i].equals("-settings"))
         {
            if (i == args.length-1)
            {
               settingsFlag = JDR.ALL_SETTINGS;
            }
            else
            {
               try
               {
                  settingsFlag = Integer.parseInt(args[i+1]);
                  i++;
               }
               catch (NumberFormatException e)
               {
                  settingsFlag = JDR.ALL_SETTINGS;
               }
            }
            settingsAsInput = false;
         }
         else if (args[i].equals("-settings_as_input"))
         {
            settingsAsInput = true;
         }
         else if (args[i].equals("-v1.0"))
         {
            version = 1.0f;
         }
         else if (args[i].equals("-v1.1"))
         {
            version = 1.1f;
         }
         else if (args[i].equals("-v1.2"))
         {
            version = 1.2f;
         }
         else if (args[i].equals("-v1.3"))
         {
            version = 1.3f;
         }
         else if (args[i].equals("-v1.4"))
         {
            version = 1.4f;
         }
         else if (args[i].equals("-v1.5"))
         {
            version = 1.5f;
         }
         else if (args[i].equals("-version"))
         {
            appVersion();
            System.exit(0);
         }
         else if (args[i].equals("-help"))
         {
            syntax();
         }
         else if (args[i].startsWith("-"))
         {
            System.err.println("Unknown switch '"+args[i]+"'");
            System.err.println("Use '-help' for help");

            System.exit(1);
         }
         else if (ajrFile == null)
         {
            ajrFile = args[i];
         }
         else if (jdrFile == null)
         {
            jdrFile = args[i];
         }
         else
         {
            System.err.println("Too many file names specified");
            System.err.println("Use '-help' for help");

            System.exit(1);
         }
      }

      if (ajrFile == null)
      {
         System.err.println("Missing input file");
         System.err.println("Use '-help' for help");

         System.exit(1);
      }
      else if (jdrFile == null)
      {
         System.err.println("Missing output file");
         System.err.println("Use '-help' for help");

         System.exit(1);
      }

      BufferedReader in=null;

      try
      {
         in = new BufferedReader(new FileReader(ajrFile));
      }
      catch (IOException e)
      {
         System.err.println("Unable to open '"+ajrFile
            +"' "+e.getMessage());
         System.exit(1);
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      String fileVersion = "";
      CanvasSettings settings = new CanvasSettings();
      JDRGroup paths=null;
      try
      {
         paths = AJR.load(in, settings);
      }
      catch (InvalidFormatException e)
      {
         System.err.println(e.getMessage());
         System.exit(1);
      }
      catch (EOFException e)
      {
         System.err.println("Unexpected EOF. (File corrupt?)");
         System.exit(1);
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      try
      {
         in.close();
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      int inputSettingsFlag = AJR.getLastLoadedSettingsID();

      if (settingsAsInput)
      {
         settingsFlag = inputSettingsFlag;

         if (settingsFlag == JDR.PAPER_ONLY && version < 1.3f)
         {
            System.err.println("Warning: paper only settings not available for versions prior to 1.3 - defaulting to -settings 1");
            settingsFlag = JDR.ALL_SETTINGS;
         }
      }

      if (inputSettingsFlag == JDR.NO_SETTINGS
        && settingsFlag != JDR.NO_SETTINGS)
      {
         settings.paper = JDRPaper.getClosestPredefinedPaper(
            paths.getBBox().getWidth(), paths.getBBox().getHeight(),
            version);
      }

      DataOutputStream out=null;

      try
      {
         out = new DataOutputStream(new FileOutputStream(jdrFile));
      }
      catch (IOException e)
      {
         System.err.println("Unable to open '"+jdrFile
            +"' "+e.getMessage());
         System.exit(1);
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      try
      {
         JDR.save(paths, settings, out, version, settingsFlag);
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }

      try
      {
         out.close();
      }
      catch (Exception e)
      {
         System.err.println(e);
         System.exit(1);
      }
   }
}
