// $Id: TeXtfm.cc,v 1.6 2000/11/21 22:38:43 yotam Exp $

#include "TeXtfm.h"
#include <cctype>
#include <cstring>
#include <string>
#include <iostream>
#include <algorithm>
#include <vector>
#include "BifStream.h"
#include "verbose.h"
#include "debug.h"

using namespace std;

#define VerboseMessage(vflag, el) _VerboseMessage(TeXtfm::vflag, el)

class _TeXtfmImplement
{
 public:
   _TeXtfmImplement();
   _TeXtfmImplement(const _TeXtfmImplement &);
   _TeXtfmImplement& operator=(const _TeXtfmImplement &);
   virtual ~_TeXtfmImplement();

   void    setOffsets();
   bool    setLigKernOwners(const TeXtfm* ptfm);
   static  Int32  maxLength(const US32*, unsigned n);

   US16    _lengths[TeXtfm::nLen];
   US32*   _header;
   US32*   _charInfo;
   US32*   _width;
   US32*   _height;
   US32*   _depth;
   US32*   _italic;
   US32*   _ligKern;
   US32*   _kern;
   US32*   _exten;
   US32*   _param;

   // convinience function
   US8   bc() const {return _lengths[TeXtfm::bc];};
   US8   ec() const {return _lengths[TeXtfm::ec];};
   US16  nChar() const {return (US16)ec() - (US16)bc() + 1;};

   // Extra derived information

   // Since ligkern entries may be shared between characters (e.g: v,w 
   // in cmr10.tfm), we maintain two arrays, to implent ligKernOwners.
   // In /var/lib/texmf/fonts/tfm/itc/domcasl/idclq8.tfm we have 255 chars.
   // cannot use magic US8 value as null.
   vector<US8>   _owners;
   vector<US16>  _ownersOffsets;
   bool          _sizeOk;
}; // _TeXtfmImplement


////////////////////////////////////////////////////////////////////////
const char*  TeXtfm::lengthNames[TeXtfm::nLen] = 
   {"lf", "lh", "bc", "ec", "nw", "nh", "nd", "ni", "nl", "nk", "ne", "np"};
const char*  TeXtfm::CharInfo::tagNames[4] = 
             {"no_tag", "lig_tag", "list_tag", "ext_tag"};



////////////////////////////////////////////////////////////////////////
TeXtfm::TeXtfm() : _data(0)
{}


////////////////////////////////////////////////////////////////////////
TeXtfm::TeXtfm(const char* fn, US32 verbFlags, ostream& verboseStream) :
   _data(0)
{
   bool  ok = open(fn, verbFlags, verboseStream);
   if (!ok)
   {
      delete _data;
      _data = 0;
   }
} // TeXtfm::TeXtfm


////////////////////////////////////////////////////////////////////////
TeXtfm::TeXtfm(const TeXtfm &sample) :
   _data(0)
{
   _data = new _TeXtfmImplement(*sample._data);
}


////////////////////////////////////////////////////////////////////////
TeXtfm &
TeXtfm::operator=(const TeXtfm &rhs)
{
   if (this != &rhs)
   {
      delete _data;
      _data = 0;
      if (rhs._data)
      {
         _data = new _TeXtfmImplement(*rhs._data);
      }
   }
   return(*this);
}


////////////////////////////////////////////////////////////////////////
TeXtfm::~TeXtfm()
{
   delete _data;
}


////////////////////////////////////////////////////////////////////////
bool
TeXtfm::open(const char* fn, US32 verbFlags, ostream& rVerbStream)
{
   ostream* verbStream = &rVerbStream;
   bool  ok = true;

   delete _data;
   _data = 0;

   BifStream tfmFile(fn);        VerboseFclopen("Opening " << fn);
   if (!tfmFile)
   {
      ok = 0;
      Error("Failed to open " << fn);
   }

   if (ok)
   {                            VerboseFclopen(fn << " opened");
      _data = new _TeXtfmImplement;  VerboseNew(_data);
      if (_data == 0)
      {
         ok = false;
         Error("Failed to allocate data");
      }

      US16  bi;
      US16*  _l = _data->_lengths;
      for (bi = 0;  bi < nLen;  bi++)
      {
         _l[bi] = tfmFile.get2ubyte();
      }
      ok = (_l[bc] - 1 <= _l[ec]) && (_l[ec] <= 0xff) &&
           (_l[lf] ==
            6 + _l[lh] + (_l[ec] - _l[bc] + 1) +
            _l[nw] + _l[nh] + _l[nd] + _l[ni] + _l[nl] + _l[nk] +
            _l[ne] + _l[np]);
      if (!ok)
      {
         VerboseMessage(preamble, "inconsistent lengths");
      }
      else
      {
         _data->_header = new US32[_l[lf] - 6];
         if (_data->_header == 0)
         {
            ok = false;
            Error("Failed to allocate data");
         }
         else
         {
            US16  nb = _l[lf] - 6;
            US32*  _header = _data->_header;
            for (bi = 0;  bi < nb;  bi++)
            {
               _header[bi] = tfmFile.get4ubyte();
            }
            // verify this is the end of file
            if (!tfmFile)
            {
               VerboseMessage(eofsync, "Unexpected EOF");
            }
            else
            {
               DASSERT(tfmFile.good());
               (void)tfmFile.get();
               DASSERT(tfmFile.eof());
               if (!(tfmFile.eof()))
               {
                  // ok = false;
                  _data->_sizeOk = false;
                  VerboseMessage(eofsync, "EOF too late");
               }
            }
         }
         if (ok)
         {
            _data->setOffsets();
            ok = _data->setLigKernOwners(this);
         }
         if (!ok)
         {
            delete _data;
            _data = 0;
         }
      }
   }
   return(ok);
} // open


////////////////////////////////////////////////////////////////////////
bool
TeXtfm::active() const
{
   return(_data != 0);
} // active



////////////////////////////////////////////////////////////////////////
US16
TeXtfm::length(US8 i) const
{
   US16  rc = (_data && (i < nLen) ? _data->_lengths[i] : 0);
   return(rc);
} // length


////////////////////////////////////////////////////////////////////////
US32
TeXtfm::header(US16 i) const
{
   US32  rc = (_data && (i < _data->_lengths[lh]) ? _data->_header[i] : 0);
   return(rc);
} // header


////////////////////////////////////////////////////////////////////////
double
TeXtfm::designSize() const
{
   US32     p1 = header(1);
   FixWord  fw = FixWord(p1);
   double   ds = (double)fw;;
   return(ds);
} // designSize


////////////////////////////////////////////////////////////////////////
// We cannot rely on Big-Endian.
// But we know each 4 bytes where loaded from left to right.
// See BifStream::get4ubyte(...)
string  TeXtfm::extraHeader(char unprintReplace) const
{  
   unsigned    headerLen = _data->_lengths[TeXtfm::lh];
   headerLen = (headerLen > 2 ? headerLen - 2 : 0);  
   unsigned    xHeaderLen = 4*headerLen;
   DPR("hl="<<headerLen << ", xl="<<xHeaderLen << ", unpRp="<<unprintReplace);
   char*       xh = new char[xHeaderLen + 1];
   char*       cxh = xh;
   for (US32  *pw = _data->_header + 2,  *pwEnd = pw + headerLen;   
        pw != pwEnd;  ++pw)
   {
      US32  w = *pw;
      for (unsigned  shift = 32;  shift; ++cxh)
      {
         shift -= 8;  
         char c = (w >> shift) & 0xff;
         *cxh = ((unprintReplace && !isprint(c)) ? unprintReplace : c);
      }
   }
   *cxh = '\0'; // null terminate
   string      sxh(xh, (string::size_type)(xHeaderLen));
   delete [] xh;
   DPR("sxh=" << sxh << ", xHeaderLen=" << xHeaderLen);
   return(sxh);
} // extraHeader


////////////////////////////////////////////////////////////////////////
TeXtfm::CharInfo
TeXtfm::char_info(US8 c) const
{
   US16  begChar = _data->_lengths[bc];
   US32  w = (_data && (begChar <= c) && (c <= _data->_lengths[ec])
              ? _data->_charInfo[c - begChar] : 0);
   CharInfo  rc(w);
   return(rc);
} // char_info


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::width(US16 i) const
{
   Int32  rc = (_data && (i < _data->_lengths[nw]) ? _data->_width[i] : 0);
   return(rc);
} // width


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::height(US16 i) const
{
   Int32  rc = (_data && (i < _data->_lengths[nh]) ? _data->_height[i] : 0);
   return(rc);
} // height


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::depth(US16 i) const
{
   Int32  rc = (_data && (i < _data->_lengths[nd]) ? _data->_depth[i] : 0);
   return(rc);
} // depth


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::italic(US16 i) const
{
   Int32  rc = (_data && (i < _data->_lengths[ni]) ? _data->_italic[i] : 0);
   return(rc);
} // italic


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::cWidth(US8 c) const
{
   CharInfo   ci = char_info(c);
   US16       i  = ci.width_index();
   Int32       rc = width(i);
   return(rc);
} // cWidth


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::cHeight(US8 c) const
{
   CharInfo   ci = char_info(c);
   US16       i  = ci.height_index();
   Int32       rc = height(i);
   return(rc);
} // cHeight


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::cDepth(US8 c) const
{
   CharInfo   ci = char_info(c);
   US16       i  = ci.depth_index();
   Int32       rc = depth(i);
   return(rc);
} // cDepth


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::cItalic(US8 c) const
{
   CharInfo   ci = char_info(c);
   US16       i  = ci.italic_index();
   Int32       rc = italic(i);
   return(rc);
} // cItalic


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::maxWidth() const
{
   return(_TeXtfmImplement::maxLength(_data->_width, length(nw)));
} // maxWidth


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::maxHeight() const
{
   return(_TeXtfmImplement::maxLength(_data->_height, length(nh)));
} // maxHeight

////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::maxDepth() const
{
   return(_TeXtfmImplement::maxLength(_data->_depth, length(nd)));
} // maxDepth

////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::maxItalic() const
{
   return(_TeXtfmImplement::maxLength(_data->_italic, length(ni)));
} // maxItalic

////////////////////////////////////////////////////////////////////////
TeXtfm::LigKern
TeXtfm::lig_kern(US16 i) const
{
   US32  w = (_data && (i < _data->_lengths[nl])
              ? _data->_ligKern[i] : 0);
   LigKern  rc(w);
   return(rc);
} // lig_kern


////////////////////////////////////////////////////////////////////////
Int16
TeXtfm::listAscend(US8 c) const
{
   Int16  rc = -1;
   CharInfo   ci = char_info(c);
   if (ci.tag() == CharInfo::list_tag)
   {
      rc = ci.reminder();
   }
   return(rc);
} // TeXtfm::listAscend


////////////////////////////////////////////////////////////////////////
bool   
TeXtfm::ligKernOwners
(
   TeXtfm::OwnerIter&  b, 
   TeXtfm::OwnerIter&  e, 
   US16                i
) const
{
   bool  ok = (_data && (i < _data->_lengths[nl]));
   if (ok)
   {
      b = _data->_owners.begin() + _data->_ownersOffsets[i];
      e = _data->_owners.begin() + _data->_ownersOffsets[i + 1];
   }
   return(ok);
} // ligKernOwner


////////////////////////////////////////////////////////////////////////
Int32
TeXtfm::kern(US16 i) const
{
   US32  u = (_data && (i < _data->_lengths[nk]) ? _data->_kern[i] : 0);
   Int32 rc = (Int32)u;
   return(rc);
} // kern


////////////////////////////////////////////////////////////////////////
TeXtfm::ExtensibleRecipe
TeXtfm::exten(US16 i) const
{
   US32  w = (_data && (i < _data->_lengths[ne])
              ? _data->_exten[i] : 0);
   ExtensibleRecipe  rc(w);
   return(rc);
} // exten


////////////////////////////////////////////////////////////////////////
US32
TeXtfm::param(US16 i) const
{
   // param is indexed from 1 !!
   US32  rc = (_data && (0 < i) && (i <= _data->_lengths[np]) 
               ? _data->_param[i] 
               : 0);
   return(rc);
} // param


////////////////////////////////////////////////////////////////////////
bool 
TeXtfm::sizeOk() const
{
   bool  ok = _data->_sizeOk;
   return(ok);
} // TeXtfm::sizeOk


////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////
_TeXtfmImplement::_TeXtfmImplement():
   _header(0),
   _charInfo(0),
   _width(0),
   _height(0),
   _depth(0),
   _italic(0),
   _ligKern(0),
   _kern(0),
   _exten(0),
   _param(0),
   _sizeOk(true)
{
}


////////////////////////////////////////////////////////////////////////
_TeXtfmImplement::_TeXtfmImplement(const _TeXtfmImplement &sample) :
   _header(0),
   _owners(sample._owners),
   _ownersOffsets(sample._ownersOffsets),
   _sizeOk(sample._sizeOk)
{
   (void)memcpy(_lengths, sample._lengths, sizeof(_lengths));
   unsigned int  nwords = _lengths[TeXtfm::lf] - 6;
   _header = new US32[nwords];
   copy(sample._header, sample._header + nwords, _header);
   setOffsets();
} // _TeXtfmImplement(const _TeXtfmImplement &)


////////////////////////////////////////////////////////////////////////
_TeXtfmImplement&
_TeXtfmImplement::operator=(const _TeXtfmImplement &rhs)
{
   if (this != &rhs)
   {
      (void)memcpy(_lengths, rhs._lengths, sizeof(_lengths));
      delete [] _header;
      unsigned int  nwords = _lengths[TeXtfm::lf] - 6;
      _header = new US32[nwords];
      copy(rhs._header, rhs._header + nwords, _header);
      setOffsets();
      _owners = rhs._owners;
      _ownersOffsets = rhs._ownersOffsets;
      _sizeOk = rhs._sizeOk;
   }
   return(*this);
} // operator=


////////////////////////////////////////////////////////////////////////
_TeXtfmImplement::~_TeXtfmImplement()
{
   delete [] _header;
} // ~_TeXtfmImplement


////////////////////////////////////////////////////////////////////////
void
_TeXtfmImplement::setOffsets()
{
   _charInfo = _header   + _lengths[TeXtfm::lh];
   int    nc = nChar();
   _width    = _charInfo + nc;
   _height   = _width    + _lengths[TeXtfm::nw];
   _depth    = _height   + _lengths[TeXtfm::nh];
   _italic   = _depth    + _lengths[TeXtfm::nd];
   _ligKern  = _italic   + _lengths[TeXtfm::ni];
   _kern     = _ligKern  + _lengths[TeXtfm::nl];
   _exten    = _kern     + _lengths[TeXtfm::nk];
   _param    = _exten    + (_lengths[TeXtfm::ne] - 1); // indexed from 1
} // setOffsets


////////////////////////////////////////////////////////////////////////
// Set the _owners & _owners. Do two passes.
//  1st pass: Count how many owners for each lig-kern entry.
//  make arrays
//  2nd pass fill in
bool
_TeXtfmImplement::setLigKernOwners(const TeXtfm* ptfm)
{
   bool          ok = true;
   unsigned      nl = _lengths[TeXtfm::nl], nTotalOwners = 0;
   vector<US16>  nOwners(nl, 0);
   // first pass
   US8  c = bc();
   
   const US32*  pcb = _charInfo;
   const US32*  pce = pcb + ((int)(ec() - c) + 1);
   for (const US32*  pci = pcb;  ok && (pci != pce);  ++pci, ++c)
   {
      TeXtfm::CharInfo  charInfo(*pci);
      if (charInfo.valid() && (charInfo.tag() == TeXtfm::CharInfo::lig_tag))
      {
         US8  lki = charInfo.reminder();
         ok = (lki < nl);
         if (ok)
         {
            ++nOwners[lki];
            ++nTotalOwners;
         }
      }
   }

   if (ok)
   {
      // Allocate vectors
      _ownersOffsets.clear();
      _ownersOffsets.insert(_ownersOffsets.end(), nl + 1, 0);
      _owners.clear();
      _owners.insert(_owners.end(), nTotalOwners, 0);
      vector<vector<US8>::iterator>  insIter(nl, _owners.begin());
      for (unsigned  i = 0;  i != nl;  ++i)
      {
         insIter[i] = _owners.begin() + _ownersOffsets[i];
         _ownersOffsets[i + 1] = _ownersOffsets[i] + nOwners[i];
      }

      // second pass
      c = bc();
      for (const US32*  pci = pcb;  (pci != pce);  ++pci, ++c)
      {
         TeXtfm::CharInfo  charInfo(*pci);
         if (charInfo.valid() && (charInfo.tag() == TeXtfm::CharInfo::lig_tag))
         {
            unsigned  lki = charInfo.reminder();
            *insIter[lki]++ = c;
         }
      }

      if (nl)
      {
         // Following assertion does not hold for 
         //    /var/lib/texmf/fonts/tfm/itc/domcasl/idclq8.tfm 
         // DASSERT(ptfm->lig_kern(nl-1).stop_flag()); // but let's ensure it
         static const unsigned  
            stop_flag = (0x80 << (8 * (3 - TeXtfm::LigKern::stop_bit)));
         _ligKern[nl-1] |= stop_flag;
      }
   }
   return(ok);
} // setLigKernOwners


////////////////////////////////////////////////////////////////////////
Int32
_TeXtfmImplement::maxLength(const US32* pl, unsigned n)
{
   const  Int32*  spl = reinterpret_cast<const Int32*>(pl);
   const  Int32*  where = max_element(spl, spl + n);
   return *where;
} // maxLength


////////////////////////////////////////////////////////////////////////
TeXtfm::CharInfo::CharInfo(US32 w)
{
   _width_index  = (w >> 3*8)     & 0xff;
   _height_index = (w >> (2*8+4)) & 0xf;
   _depth_index  = (w >> (2*8))   & 0xf;
   _italic_index = (w >> (8+2))   & 0x3f;
   US32  tw      = (w >> 8)       & 0x3;
   _tag          = (Tag)tw;
   _reminder     = w              & 0xff;
} // CharInfo


////////////////////////////////////////////////////////////////////////
TeXtfm::FourBytes::FourBytes(US32 w)
{
   int  i;
   for (i = 4;  i--;)
   {
      _byte[i] = (w >> ((3 - i)*8)) & 0xff;
   }
} // FourBytes


////////////////////////////////////////////////////////////////////////
ostream &
operator << (ostream &os, const TeXtfm::FixWord &fw)
{
   ios::fmtflags  saveFlags = os.flags();
   os << float(fw) << " (0x" << hex << Int32(fw) << ")";
   os.flags(saveFlags);
   return os;
} // << FixWord


////////////////////////////////////////////////////////////////////////
ostream&
operator<<(ostream& os, const TeXtfm& tfm)
{
   ios::fmtflags  saveFlags = os.flags();
   US16           lh = tfm.length(TeXtfm::lh);
   US32           ds = tfm.header(1);

   os << "TFM " << endl << 
         "{" << endl <<
         "  lengths: " << endl << 
         "    lf=" << tfm.length(TeXtfm::lf) << 
           ", lh=" << lh <<
           ", bc=" << tfm.length(TeXtfm::bc) <<
           ", ec=" << tfm.length(TeXtfm::ec) << "," << endl <<
         "    nw=" << tfm.length(TeXtfm::nw) <<
           ", nh=" << tfm.length(TeXtfm::nh) <<
           ", nd=" << tfm.length(TeXtfm::nd) <<
           ", ni=" << tfm.length(TeXtfm::ni) << "," << endl << 
         "    nl=" << tfm.length(TeXtfm::nl) <<
           ", nk=" << tfm.length(TeXtfm::nk) <<
           ", ne=" << tfm.length(TeXtfm::ne) <<
           ", np=" << tfm.length(TeXtfm::np) << endl <<
         " checksum = " << tfm.header(0) << endl << 
         " design-size = " << TeXtfm::FixWord(ds) << endl;

   if (lh > 2)
   {
      string       sxh = tfm.extraHeader();
      const char*  xh = sxh.c_str();
      os << " extra-header = '" << xh << "'" << endl;
   }

   os.flags(saveFlags);
   return os;
}




////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////


#if TEST
#include <stdlib.h>
int
main(int argc, char* *argv)
{
   typedef TeXtfm::FixWord  FW;

   if (argc != 3)
   {
      cerr << "Usage:" << endl << 
              argv[0] << " <tfmfile> <veboseFlags>" << endl;
      return 1;
   }
   US32  verbFlags = 0;
   verbFlags = atol(argv[2]);
   TeXtfm  textfm;
   textfm  = TeXtfm(argv[1], verbFlags);

   DASSERT(textfm.active());

   int i;
   static char* LnNm[] = 
     {"lf", "lh", "bc", "ec", "nw", "nh", "nd", "ni", "nl", "nk", "ne", "np"};

   cout << endl << "Lengths: " << endl;
   for (i = 0;  i < TeXtfm::nLen;  i++)
   {
      cout << "  " << LnNm[i] << " = " << textfm.length(i) << endl;
   }

   cout << endl << "Design-Size = " << FW(textfm.header(1)) << endl;

   cout << endl << "Header" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::lh);  i++)
   {
      cout << "H[" << i << "] = " << textfm.header(i) << endl;
   }
   cout << "extraHeader = '" << textfm.extraHeader() << "'" << endl;

   cout << endl << "CharInfo:" << endl;
   for (i = textfm.length(TeXtfm::bc);  i <= textfm.length(TeXtfm::ec);  i++)
   {
      TeXtfm::CharInfo  ci = textfm.char_info(i);
      cout << "c[" << i << "]: " << 
           "wi=" << (US16)ci.width_index() << 
         ", hi=" << (US16)ci.height_index() << 
         ", di=" << (US16)ci.depth_index() << 
         ", tag=" << (US16)ci.tag() << 
         ", rem=" << (US16)ci.reminder() << endl;
   }

   cout << endl << "Widths:" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::nw);  i++)
   {
      cout << "  [" << i << "] = " << FW(textfm.width(i)) << endl;
   }

   cout << endl << "Heights:" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::nh);  i++)
   {
      cout << "  [" << i << "] = " << FW(textfm.height(i)) << endl;
   }

   cout << endl << "Depths:" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::nd);  i++)
   {
      cout << "  [" << i << "] = " << FW(textfm.depth(i)) << endl;
   }

   cout << endl << "Italics:" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::ni);  i++)
   {
      cout << "  [" << i << "] = " << FW(textfm.italic(i)) << endl;
   }

   cout << endl << "Lig-Kern:" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::nl);  i++)
   {
      TeXtfm::LigKern  lk = textfm.lig_kern(i);
      cout << "  LK[" << i << "]: stop=" << lk.stop_flag() << 
              ", kern=" << lk.kern_flag();
      int  j;
      for (j = 0;  j < 4;  j++)
      {
         cout << ", b[" << j << "] = " << (US16)lk.byte(j);
      }
      cout << endl;

      US8  cown;
      unsigned  nOwn = textfm.ligKernOwners(&cown, 1, i);
      cout << "  " << nOwn << " owners, 1st: "  << (US16)cown << endl;
   }

   cout << endl << "Kerns:" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::nk);  i++)
   {
      cout << "  [" << i << "] = " << FW(textfm.kern(i)) << endl;
   }

   cout << endl << "Extensible Recipes:" << endl;
   for (i = 0;  i < textfm.length(TeXtfm::ne);  i++)
   {
      TeXtfm::ExtensibleRecipe  er = textfm.exten(i);
      cout << "  [" << i << "]: ";
      int  j;
      for (j = 0;  j < 4;  j++)
      {
         cout << "  b[" << j << "] = " << (US16)er.byte(j);
      }
      cout << endl;
   }

   cout << endl << "Params:" << endl;
   for (i = 1;  i <= textfm.length(TeXtfm::np);  i++)
   {
      cout << "  [" << i << "] = " << FW(textfm.param(i)) << endl;
   }

   return 0;
} // main
#endif
