/*========================================================================*\

Copyright (c) 1994-2003  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
PAUL VOJTA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

\*========================================================================*/

#include "xdvi.h"
#include <X11/Xatom.h>

#include <memory.h>
#include <signal.h>

/* Condition for retrying a write */
#include <errno.h>

#ifdef	X_NOT_STDC_ENV
extern	int	errno;
#endif

#ifdef	EWOULDBLOCK
#ifdef	EAGAIN
#define	AGAIN_CONDITION	(errno == EWOULDBLOCK || errno == EAGAIN)
#else	/* EAGAIN */
#define	AGAIN_CONDITION	(errno == EWOULDBLOCK)
#endif	/* EAGAIN */
#else	/* EWOULDBLOCK */
#ifdef	EAGAIN
#define	AGAIN_CONDITION	(errno == EAGAIN)
#endif	/* EAGAIN */
#endif	/* EWOULDBLOCK */

#if HAVE_POLL
# include <poll.h>
# define XIO_IN POLLIN
# define XIO_OUT POLLOUT
#else
# define XIO_IN 1
# define XIO_OUT 2
#endif

#if HAVE_VFORK_H
# include <vfork.h>
#endif

#ifndef GS_TIMEOUT
# define GS_TIMEOUT	1000	/* milliseconds */
#endif

extern	_Xconst	char	psheader[];
extern	unsigned	psheaderlen;

#define	postscript	resource._postscript

/* global procedures (besides initGS) */

static	void	toggle_gs	ARGS((void));
static	void	destroy_gs	ARGS((void));
static	void	interrupt_gs	ARGS((void));
static	void	endpage_gs	ARGS((void));
static	void	drawbegin_gs	ARGS((int, int, _Xconst char *));
static	void	drawraw_gs	ARGS((_Xconst char *));
static	void	drawfile_gs	ARGS((_Xconst char *, FILE *));
static	void	drawend_gs	ARGS((_Xconst char *));
static	void	beginheader_gs	ARGS((void));
static	void	endheader_gs	ARGS((void));
static	void	newdoc_gs	ARGS((void));

static	struct psprocs	gs_procs = {
	/* toggle */		toggle_gs,
	/* destroy */		destroy_gs,
	/* interrupt */		interrupt_gs,
	/* endpage */		endpage_gs,
	/* drawbegin */		drawbegin_gs,
	/* drawraw */		drawraw_gs,
	/* drawfile */		drawfile_gs,
	/* drawend */		drawend_gs,
	/* beginheader */	beginheader_gs,
	/* endheader */		endheader_gs,
	/* newdoc */		newdoc_gs};

static	int	std_io[2];

#define	GS_fd	(std_io[0])

		/* some arguments are filled in later */
static	char	arg4[]	= "-dDEVICEWIDTH=xxxxxxxxxx";
static	char	arg5[]	= "-dDEVICEHEIGHT=xxxxxxxxxx";

static	_Xconst	char	*argv[]	= {NULL, NULL, "-dNOPAUSE", "-q", arg4, arg5,
				   "-dDEVICEXRESOLUTION=72",
				   "-dDEVICEYRESOLUTION=72",
				   "-dNOSAFER", "-dNOEPS", NULL, NULL, NULL};

static	unsigned int	GS_page_w;	/* how big our current page is */
static	unsigned int	GS_page_h;
static	Boolean		GS_alpha;	/* if we are using the alpha driver */
static	int		GS_mag;		/* magnification currently in use */
static	int		GS_shrink;	/* shrink factor currently in use */
static	Boolean		GS_active;	/* if we've started a page yet */
static	int		GS_pending;	/* number of ack's we're expecting */
static	char		GS_outb[257];	/* circular output buffer */
static	char		*GS_outb_in;	/* next position in output buffer */
static	char		*GS_outb_out;	/* next byte to come out of buffer */
#define	GS_outb_limit	(GS_outb + sizeof GS_outb)	/* last+1 byte */
static	int		GS_write_ack;	/* flag to set when done writing */
static	Boolean		GS_in_header;	/* if we're sending a header */
static	Boolean		GS_in_doc;	/* if we've sent header information */
static	int		GS_ev_mask;	/* events for which we'll stop */
static	int		GS_die_ack	= 0;	/* flags to set when GS dies */
static	Boolean		GS_timer_set;	/* if there's a timer set */
static	Boolean		GS_old;		/* if we're using gs 2.xx */

#define	GS_MASK_NORMAL	EV_GE_NEWPAGE
#define	GS_MASK_HEADER	EV_GE_PS_TOGGLE
#define	GS_MASK_INIT	(EV_GE_TERM | EV_PS_TOGGLE)

static	Atom		gs_atom;
static	Atom		gs_colors_atom;

#define	Landscape	90

#define	LINELEN	81
static	char	line[LINELEN + 1];
static	char	*linepos	= line;
static	char	ackstr[]	= "\347\310\376";
static	char	oldstr[]	= "\347\310\375";

static	void	gs_died ARGS((int));

static	struct xchild	gs_child	= {NULL, 0, True, gs_died};
#define	GS_pid		(gs_child.pid)

static	void	read_from_gs ARGS((void));
static	void	write_to_gs ARGS((void));

static	struct xio	gs_xio		= {NULL, 0, XIO_IN,
#if HAVE_POLL
					   NULL,
#endif
					   read_from_gs, write_to_gs};

static	void	gs_alarm ARGS((struct xtimer *));

static	struct xtimer	gs_timer	= TIMER_INIT(gs_alarm);

static	void
showto(q)
	char	*q;
{
	char	*p	= line;
	char	*p1;

	while (p < q) {
	    p1 = memchr(p, '\n', q - p);
	    if (p1 == NULL) p1 = q;
	    *p1 = '\0';
	    Printf("gs: %s\n", p);
	    p = p1 + 1;
	}
}

static	void
read_from_gs()
{
	int	bytes;
	char	*line_end;
	char	*p;

	for (;;) {

	    bytes = read(GS_fd, linepos, line + LINELEN - linepos);
	    if (bytes < 0) {
		if (AGAIN_CONDITION)
		    break;
		perror("xdvi: read_from_gs");
		break;
	    }
	    line_end = linepos + bytes;

	    if (bytes == 0) {
		if (GS_pid != 0)
		    puts("Read_from_gs returned 0 bytes.");
		break;
	    }

	    /* Check for ack strings */
	    for (p = line; p < line_end - 2; ++p) {
		p = memchr(p, '\347', line_end - p - 2);
		if (p == NULL) break;
		if (memcmp(p, ackstr, 3) == 0) {
		    --GS_pending;
		    if (GS_pending == 0)
			ev_flags |= EV_ACK;
		    if (debug & DBG_PS)
			Printf("Got GS ack; %d pending.\n", GS_pending);
		}
		else if (memcmp(p, oldstr, 3) == 0) {
		    if (debug & DBG_PS)
			Puts("Using old GS version.");
		    GS_old = True;
		}
		else continue;

		showto(p);
		p += 3;
		(void) bcopy(p, line, line_end - p);
		line_end -= p - line;
		linepos = p = line;
		--p;
	    }
	    *line_end = '\0';
	    p = rindex(linepos, '\n');
	    if (p != NULL) {
		++p;
		showto(p);
		(void) bcopy(p, line, line_end - p);
		line_end -= p - line;
	    }
	    linepos = line_end;
	    /*
	     * Normally we'd hold text until a newline character, but the
	     * buffer is full.  So we flush it, being careful not to cut up an
	     * ack string.
	     */
	    if (linepos >= line + LINELEN) {
		p = line + LINELEN;
		if ((*--p != '\347' && *--p != '\347' && *--p != '\347')
			|| (memcmp(p, ackstr, line + LINELEN - p) != 0
			&& memcmp(p, oldstr, line + LINELEN - p) != 0))
		    p = line + LINELEN;
		*p = '\0';
		Printf("gs: %s\n", line);
		*p = '\347';
		linepos = line;
		while (p < line + LINELEN) *linepos++ = *p++;
	    }
	}
}

static	void
write_to_gs()
{
	char	*send_end;
	int	bytes;

	for (;;) {
	    send_end = GS_outb_in;
	    if (send_end < GS_outb_out) send_end = GS_outb_limit;
	    bytes = write(GS_fd, GS_outb_out, send_end - GS_outb_out);
	    if (bytes < 0) {
		if (AGAIN_CONDITION)
		    break;
		perror("xdvi: write_to_gs");
		break;
	    }
	    GS_outb_out += bytes;
	    if (GS_outb_out == GS_outb_limit) GS_outb_out = GS_outb;
	    if (GS_outb_out == GS_outb_in) {	/* if buffer is empty */
		gs_xio.xio_events = XIO_IN;
#if HAVE_POLL
		if (gs_xio.pfd != NULL)	/* write_to_gs is called directly */
		    gs_xio.pfd->events = XIO_IN;
#endif
		break;
	    }
	}

	ev_flags |= GS_write_ack;
	GS_write_ack = 0;
}


/*
 *	Main routine for writing to the GS interpreter.
 */

static	void
gs_send(cp, len)
	_Xconst	char	*cp;
	size_t		len;
{
	_Xconst char	*cp_end = cp + len;
	char		*send_end;
	size_t		bytes;
	char		*old_out;
	Boolean		interrupting;

	if (GS_pid == 0 || (ev_flags & GS_ev_mask))
	    return;

	/*
	 * Because cp might reside on the stack, don't return until we've
	 * copied all of it to our circular output buffer.
	 * Note that GS_outb_out == GS_outb_in means that the buffer is empty.
	 */

	GS_timer_set = interrupting = False;
	for (;;) {
	    send_end = GS_outb_out;
	    if (send_end == GS_outb) send_end = GS_outb_limit;
	    --send_end;
	    if (send_end < GS_outb_in) send_end = GS_outb_limit;
	    bytes = send_end - GS_outb_in;
	    if (bytes > 0) {
		if (bytes >= cp_end - cp) bytes = cp_end - cp;
		bcopy(cp, GS_outb_in, bytes);
		cp += bytes;
		GS_outb_in += bytes;
		if (GS_outb_in == GS_outb_limit) GS_outb_in = GS_outb;
		if (cp < cp_end) continue;
	    }

	    /* The buffer is now full --or-- we've run out of data */
	    old_out = GS_outb_out;
	    if (!(gs_xio.xio_events & XIO_OUT)) {	/* restart output */
		gs_xio.xio_events = XIO_IN | XIO_OUT;
#if HAVE_POLL
		if (gs_xio.pfd != NULL)
		    gs_xio.pfd->events = POLLIN | POLLOUT;
#endif
		write_to_gs();
		if (GS_outb_out != old_out) {
		    if (cp == cp_end)
			break;
		    else
			continue;
		}
	    }

	    if (cp == cp_end)
		break;

	    GS_die_ack = GS_write_ack = EV_ACK;
	    for (;;) {	/* loop because there may be stray ACKs */
		if (!interrupting) {
		    (void) read_events(GS_ev_mask | EV_ACK);
		    ev_flags &= ~EV_ACK;

		    if (GS_pid == 0) {	/* if GS died */
			GS_die_ack = 0;
			return;
		    }

		    if (GS_outb_out != old_out)	/* if more room in buffer */
			break;

		    if (ev_flags & GS_ev_mask) {	/* if a serious event */
			if (debug & DBG_PS)
			    Puts("Setting timeout in gs_send()");

			set_timer(&gs_timer, GS_TIMEOUT);
			GS_timer_set = interrupting = True;
		    }
		}
		else {
		    (void) read_events(EV_GE_TERM | EV_PS_TOGGLE | EV_ACK);
		    ev_flags &= ~EV_ACK;

		    if (GS_outb_out != old_out)	/* if more room in buffer */
			break;

		    if (GS_timer_set)	/* if timer still set */
			cancel_timer(&gs_timer);

		    destroy_gs();
		    GS_die_ack = 0;
		    return;
		}
	    }
	}

	if (GS_timer_set)	/* if timer still set */
	    cancel_timer(&gs_timer);

	GS_die_ack = GS_write_ack = 0;
}

/*
 *	Interrupt routine for timer routine.
 */

static	void
gs_alarm(arg)
	struct xtimer	*arg;
{
	if (debug & DBG_PS)
	    puts("GS timeout expired");

	ev_flags |= EV_ACK;
	GS_timer_set = False;
}

/*
 *	Wait for acknowledgement from GS.
 */

static	void
waitack()
{
	if (GS_pending == 0) {
	    ev_flags &= ~EV_ACK;
	    return;
	}
	
	GS_die_ack = EV_ACK;

	for (;;) {	/* loop because there might be stray ACKs. */
	    (void) read_events(EV_GE_ACK);
	    ev_flags &= ~EV_ACK;

	    if (GS_pending == 0) {
		GS_die_ack = 0;
		return;
	    }
	    if (ev_flags & EV_GE_ACK)
		break;
	}

	if (debug & DBG_PS)
	    Puts("Setting timeout in waitack()");

	set_timer(&gs_timer, GS_TIMEOUT);
	GS_timer_set = True;

	(void) read_events(EV_GE_TERM | EV_PS_TOGGLE | EV_ACK);
	ev_flags &= ~EV_ACK;

	if (GS_timer_set)
	    cancel_timer(&gs_timer);

	GS_die_ack = 0;

	if (GS_pending > 0)
	    destroy_gs();
}


/*
 *	Fork a process to run ghostscript.  This is done using the
 *	x11 device (which needs to be compiled in).  Normally the x11
 *	device uses ClientMessage events to communicate with the calling
 *	program, but we don't do this.  The reason for using the ClientMessage
 *	events is that otherwise ghostview doesn't know when a non-conforming
 *	postscript program calls showpage.   That doesn't affect us here,
 *	since in fact we disable showpage.
 *
 *	SAFER mode is handled as follows.  Ghostscript versions 7.02 and earlier
 *	ignore -dNOSAFER and handle -dSAFER if it is supplied; the code in
 *	strsafe is ignored since .locksafe is not present.  In versions 7.04 and
 *	higher, -dNOSAFER overrides -dSAFER (if provided); SAFER mode is
 *	optionally turned on by sending the strsafe string.  It is possible
 *	in some versions of gs prior to 7.04 to use -dDELAYSAFER instead of
 *	-dNOSAFER, but there's no point in doing that since .locksafe is not
 *	defined in those versions.  I don't know where 7.03 fits in on all of
 *	this.
 */

Boolean
initGS()
{
	char			buf[100];
	static	Boolean		did_putenv	= False;
		/*
		 * This string reads chunks (delimited by %%xdvimark).
		 * The first character of a chunk tells whether a given chunk
		 * is to be done within save/restore or not.
		 * The `H' at the end tells it that the first group is a
		 * header; i.e., no save/restore.
		 * `execute' is unique to ghostscript.
		 */
	static	_Xconst	char	strsafe[] = "\
{ << /PermitFileReading [ (*) ] /PermitFileWriting [ ] /PermitFileControl [ ] \
  >> setuserparams .locksafe \
} stopped pop\n";

	static	_Xconst	char	str1[]	= "\
/xdvi$run {$error /newerror false put {currentfile cvx execute} stopped pop} \
  def \
/xdvi$ack (\347\310\376) def \
/xdvi$dslen countdictstack def \
{currentfile read pop 72 eq \
    {xdvi$run} \
    {/xdvi$sav save def xdvi$run \
      clear countdictstack xdvi$dslen sub {end} repeat xdvi$sav restore} \
  ifelse \
  {(%%xdvimark) currentfile =string {readline} stopped \
    {clear} {pop eq {exit} if} ifelse }loop \
  flushpage xdvi$ack print flush \
}loop\nH";
	static	_Xconst	char	str2[]	= "[0 1 1 0 0 0] concat\n\
revision 300 lt{(\347\310\375) print flush}if\n\
stop\n%%xdvimark\n";

	/*
	 * If we're prescanning *before* setting up the widgets (to get the
	 * page size, for example), then postpone starting up ghostscript.
	 */

	if (mane.win == (Window) 0) {
	    if (debug & DBG_PS)
		Puts("Hit PS header in early prescan; postponing.");
	    psp = no_ps_procs;
	    gs_postpone_prescan = True;
	    return True;
	}

	if (debug & DBG_PS)
	    Puts("Running initGS ...");

	gs_atom = XInternAtom(DISP, "GHOSTVIEW", False);
	/* send bpixmap, orientation, bbox (in pixels), and h & v resolution */
	Sprintf(buf, "%ld %d 0 0 %u %u 72 72",
	    None,		/* bpixmap */
	    Landscape,		/* orientation */
	    GS_page_h = page_h, GS_page_w = page_w);
	XChangeProperty(DISP, mane.win, gs_atom, XA_STRING, 8,
	    PropModeReplace, (unsigned char *) buf, strlen(buf));
	GS_alpha = resource.gs_alpha;

	gs_colors_atom = XInternAtom(DISP, "GHOSTVIEW_COLORS", False);
	Sprintf(buf, "%s %ld %ld", resource.gs_palette,
	  fore_color_data.pixel, back_color_data.pixel);
	XChangeProperty(DISP, mane.win, gs_colors_atom, XA_STRING, 8,
	  PropModeReplace, (unsigned char *) buf, strlen(buf));

	if (!did_putenv) {
	    Sprintf(buf, "%ld", mane.win);
	    xputenv("GHOSTVIEW", buf);
	    did_putenv = True;
	}

	XSync(DISP, False);		/* update the window */

	if (xpipe(std_io) != 0) {
	    perror("[xdvi] pipe");
	    return False;
	}
	Fflush(stderr);		/* to avoid double flushing */
	GS_pid = vfork();
	if (GS_pid == 0) {		/* child */
	    _Xconst char **argvp = argv + 10;

	    argv[1] = resource.gs_alpha ? "-sDEVICE=x11alpha" : "-sDEVICE=x11";
	    Sprintf(arg4 + 14, "%u", GS_page_w);
	    Sprintf(arg5 + 15, "%u", GS_page_h);
	    if (resource.gs_safer) *argvp++ = "-dSAFER";
	    *argvp = "-";
	    (void) close(std_io[0]);
	    (void) dup2(std_io[1], 0);
	    (void) dup2(std_io[1], 1);
	    (void) dup2(std_io[1], 2);
	    (void) close(std_io[1]);
	    (void) execvp(argv[0] = resource.gs_path, (char * _Xconst *) argv);
	    Fprintf(stderr, "%s: Execvp of %s failed.\n", prog, argv[0]);
	    Fflush(stderr);
	    _exit(1);
	}
	(void) close(std_io[1]);
	++n_files_left;
	if (GS_pid == -1) {	/* error */
	    GS_pid = 0;
	    perror("[xdvi] vfork");
	    (void) close(GS_fd);
	    ++n_files_left;
	    return False;
	}

	prep_fd(GS_fd, True);	/* Set file descriptor for non-blocking I/O */

	set_chld(&gs_child);

	psp = gs_procs;
	GS_active = False;
	GS_in_header = True;
	GS_pending = 1;
	GS_mag = GS_shrink = -1;
	gs_xio.fd = GS_fd;
	gs_xio.xio_events = XIO_IN;
	GS_write_ack = 0;
	GS_outb_in = GS_outb_out = GS_outb;
	set_io(&gs_xio);
	GS_ev_mask = GS_MASK_INIT;
	(void) signal(SIGPIPE, SIG_IGN);

	if (resource.gs_safer)
	    gs_send(strsafe, sizeof(strsafe) - 1);
	gs_send(str1, sizeof(str1) - 1);
	gs_send(psheader, psheaderlen);
	gs_send(str2, sizeof(str2) - 1);
	waitack();
	GS_in_header = False;
	GS_ev_mask = GS_MASK_NORMAL;

	if (GS_pid == 0) {		/* if something happened */
	    destroy_gs();
	    return False;
	}
	if (!postscript) toggle_gs();	/* if we got a 'v' already */
	else {
	    scanned_page = scanned_page_ps = scanned_page_reset;
	    ev_flags |= EV_NEWPAGE;	/* ||| redraw the page */
	    longjmp(canit_env, 1);
	}
	return True;
}

static	void
toggle_gs()	/* this routine is callable from within read_events().  */
{
	if (debug & DBG_PS) Puts("Toggling GS on or off");

	psp.drawbegin = (postscript ? drawbegin_gs : drawbegin_none);
}

void
gs_resume_prescan()
{
	if (debug & DBG_PS)
	    Puts("Resuming prescan");

	gs_postpone_prescan = False;
	if (!initGS())	/* this may not return */
	    psp = no_ps_procs;
}

/* ARGSUSED */
static	void
gs_died(status)
	int	status;
{
	if (debug & DBG_PS) Puts("GS process died");
	GS_pid = 0;
	read_from_gs();
	if (linepos > line) {
	    *linepos = '\0';
	    Printf("gs: %s\n", line);
	    linepos = line;
	}
	clear_io(&gs_xio);
	(void) close(GS_fd);
	++n_files_left;
	scanned_page = scanned_page_ps = scanned_page_reset;
	GS_active = GS_in_doc = False;
	GS_pending = 0;
	ev_flags |= GS_die_ack;
}

static	void
destroy_gs()
{
	if (debug & DBG_PS) Puts("Destroying GS process");
	if (GS_pid != 0) {
	    if (kill(GS_pid, SIGKILL) < 0 && errno != ESRCH)
		perror("xdvi destroy_gs");
	    GS_pid = 0;
	    clear_chld(&gs_child);
	    read_from_gs();
	    if (linepos > line) {
		*linepos = '\0';
		Printf("gs: %s\n", line);
		linepos = line;
	    }
	    clear_io(&gs_xio);
	    (void) close(GS_fd);
	    ++n_files_left;
	    scanned_page = scanned_page_ps = scanned_page_reset;
	}
	GS_active = GS_in_doc = False;
	GS_pending = 0;
}

static	void
deactivate()
{
	static	_Xconst	char	str[]	= " stop\n%%xdvimark\n";
	int			saved_mask;

	saved_mask = GS_ev_mask;
	GS_ev_mask = 0;
	gs_send(str, sizeof(str) - 1);
	GS_ev_mask = saved_mask;

	GS_active = False;
}

static	void
interrupt_gs()
{
	if (debug & DBG_PS) Puts("Running interrupt_gs()");
	if (GS_pending <= 0) return;	/* nothing to do */

	    /*
	     * ||| what I'd really like to do here is cause gs to execute
	     * the interrupt routine in errordict.  But so far (gs 2.6.1)
	     * that has not been implemented in ghostscript.
	     */

	if (GS_active)
	    deactivate();
	waitack();
}

static	void
endpage_gs()
{
	if (debug & DBG_PS) Puts("Running endpage_gs()");
	if (GS_active) {
	    deactivate();
	    waitack();
	}
}

/*
 *	Checks that the GS interpreter is running correctly.
 */

static	void
checkgs(in_header)
	Boolean	in_header;
{
	char	buf[150];

	/* For gs 2, we pretty much have to start over to enlarge the window. */
	if ((GS_old && (page_w > GS_page_w || page_h > GS_page_h))
	  || GS_alpha != resource.gs_alpha)
	    destroy_gs();

	if (GS_pid == 0)
	    (void) initGS();

	if (!GS_active) {
	    /* check whether page_w or page_h have increased */
	    if (page_w > GS_page_w || page_h > GS_page_h) {
		if (ev_flags & GS_ev_mask)
		    longjmp(canit_env, 1);
		++GS_pending;
		Sprintf(buf, "H mark /HWSize [%d %d] /ImagingBBox [0 0 %d %d] \
currentdevice putdeviceprops pop\n\
initgraphics [0 1 1 0 0 0] concat stop\n%%%%xdvimark\n",
		    GS_page_w = page_w, GS_page_h = page_h, page_h, page_w);
		gs_send(buf, strlen(buf));
		if (!in_header) {
		    ev_flags |= EV_NEWPAGE;	/* ||| redraw the page */
		    longjmp(canit_env, 1);
		}
	    }

	    if (magnification != GS_mag) {
		if (ev_flags & GS_ev_mask)
		    longjmp(canit_env, 1);
		++GS_pending;
		Sprintf(buf, "H TeXDict begin /DVImag %d 1000 div def \
end stop\n%%%%xdvimark\n",
		    GS_mag = magnification);
		gs_send(buf, strlen(buf));
	    }

	    if (mane.shrinkfactor != GS_shrink) {
		if (ev_flags & GS_ev_mask)
		    longjmp(canit_env, 1);
		++GS_pending;
		Sprintf(buf,
		    "H TeXDict begin %d %d div dup \
/Resolution X /VResolution X \
end stop\n%%%%xdvimark\n",
		    pixels_per_inch, GS_shrink = mane.shrinkfactor);
		gs_send(buf, strlen(buf));
	    }
	}
}

static	void
drawbegin_gs(xul, yul, cp)
	int		xul, yul;
	_Xconst	char	*cp;
{
	char	buf[32];
	static	_Xconst	char	str[]	= " TeXDict begin\n";

	checkgs(False);

	if (!GS_active) {
	    if (ev_flags & GS_ev_mask)
		longjmp(canit_env, 1);
	    ++GS_pending;
	    gs_send(str, sizeof(str) - 1);
	    GS_active = True;
	}

	/* This allows the X side to clear the page */
	XSync(DISP, False);

	Sprintf(buf, "%d %d moveto\n", xul, yul);
	gs_send(buf, strlen(buf));
	if (debug & DBG_PS)
	    Printf("drawbegin at %d,%d:  sending `%s'\n", xul, yul, cp);
	gs_send(cp, strlen(cp));
}

static	void
drawraw_gs(cp)
	_Xconst	char	*cp;
{
	if (!GS_active)
	    return;
	if (debug & DBG_PS) Printf("raw ps sent to context: %s\n", cp);
	gs_send(cp, strlen(cp));
	gs_send("\n", 1);
}

static	void
drawfile_gs(cp, f)
	_Xconst	char	*cp;
	FILE		*f;
{
	Fclose(f);		/* don't need it */
	++n_files_left;

	if (!GS_active)
	    return;

	if (debug & DBG_PS) Printf("printing file %s\n", cp);

	gs_send("(", 1);
	gs_send(cp, strlen(cp));
	gs_send(")run\n", 5);
}

static	void
drawend_gs(cp)
	_Xconst	char	*cp;
{
	if (!GS_active)
	    return;
	if (debug & DBG_PS) Printf("end ps: %s\n", cp);
	gs_send(cp, strlen(cp));
	gs_send("\n", 1);
}

static	void
beginheader_gs()
{
	static	_Xconst	char	str[]	= "Hsave /xdvi$doc exch def\n";

	if (debug & DBG_PS) Puts("Running beginheader_gs()");

	checkgs(True);

	if (GS_active) {
	    if (!GS_in_header)
		oops("Internal error in beginheader_gs().\n");
	    return;
	}

	if (ev_flags & GS_ev_mask)
	    longjmp(canit_env, 1);

	GS_in_header = True;
	GS_ev_mask = GS_MASK_HEADER;
	++GS_pending;
	if (GS_in_doc)
	    gs_send("H", 1);
	else {
	    gs_send(str, sizeof(str) - 1);
	    GS_in_doc = True;
	}
	GS_active = True;
}

static	void
endheader_gs()
{
	if (debug & DBG_PS) Puts("Running endheader_gs()");

	if (GS_active) {
	    deactivate();
	    waitack();
	    GS_in_header = False;
	    GS_ev_mask = GS_MASK_NORMAL;
	}
}

static	void
newdoc_gs()
{
	static	_Xconst	char	str[]	=
				"Hxdvi$doc restore stop\n%%xdvimark\n";

	if (debug & DBG_PS) Puts("Running newdoc_gs()");

	if (GS_in_doc) {
	    ++GS_pending;
	    gs_send(str, sizeof(str) - 1);
	    GS_mag = GS_shrink = -1;
	    GS_in_doc = False;

	    if (!GS_old) GS_page_w = GS_page_h = 0;
	}
}
