/*========================================================================*\

Copyright (c) 1990-2013  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL PAUL VOJTA OR ANY OTHER AUTHOR OF OR CONTRIBUTOR TO
THIS SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

NOTE:
	xdvi is based on prior work, as noted in the modification history
	in xdvi.c.

\*========================================================================*/

#include "xdvi.h"
#include "dvi.h"
#include <sys/stat.h>

#if FREETYPE
# include <ft2build.h>
# include FT_SIZES_H
#endif

#if !defined(S_ISDIR) && defined(S_IFDIR)
#define	S_ISDIR(m)	(((m) & S_IFMT) == S_IFDIR)
#endif

#if XAW
# include <X11/Shell.h>	/* defines XtNtitle and XtNiconName */
#endif

#define	PK_PRE		247
#define	PK_ID		89
#define	PK_MAGIC	((PK_PRE << 8) | PK_ID)
#define	GF_PRE		247
#define	GF_ID		131
#define	GF_MAGIC	((GF_PRE << 8) | GF_ID)
#define	VF_PRE		247
#define	VF_ID_BYTE	202
#define	VF_MAGIC	((VF_PRE << 8) | VF_ID_BYTE)

static	jmp_buf		dvi_env;	/* mechanism to relay dvi file errors */
static	_Xconst	char	*dvi_oops_msg;	/* error message */

#define	dvi_oops(str)	(dvi_oops_msg = (str), longjmp(dvi_env, 1))

static	Boolean		font_not_found;
static	Boolean		dvi_is_valid = False;	/* if not initializing */

/*
 * DVI preamble and postamble information.
 */
static	char		job_id[300];
static	long		numerator, denominator;
static	unsigned int	dvi_unshrunk_page_w, dvi_unshrunk_page_h;

/*
 * Offset in DVI file of last page, set in read_postamble().
 */
static	long	last_page_offset;


/*
 *	free_vf_chain frees the vf_chain structure.
 */

static	void
free_vf_chain(tnp)
	struct tn *tnp;
{
	while (tnp != NULL) {
	    struct tn *tnp1 = tnp->next;
	    free((char *) tnp);
	    tnp = tnp1;
	}
}


/*
 *	Release all shrunken bitmaps for all fonts.
 */

void
reset_fonts()
{
	struct font *f;
	struct glyph *g;

	for (f = font_head; f != NULL; f = f->next)
	    if ((f->flags & FONT_LOADED) && !(f->flags & FONT_VIRTUAL))
		for (g = f->glyph; g <= f->glyph + f->maxchar; ++g) {
		    if (g->bitmap2.bits) {
			free(g->bitmap2.bits);
			g->bitmap2.bits = NULL;
		    }
#ifdef	GREY
		    if (g->pixmap2) {
			XDestroyImage(g->image2);
			g->pixmap2 = NULL;
			if (g->pixmap2_t != NULL) {
			    free(g->pixmap2_t);
			    g->pixmap2_t = NULL;
			}
		    }
#if COLOR
		    g->fg = NULL;
#endif
#endif /* GREY */
		}
}


#if COLOR

/*
 *	Release all allocated pixels, and (in greyscale mode) invalidate
 *	all shrunken glyphs.
 */

void
reset_colors()
{
	if (color_list_len != 0) {
	    XFreeColors(DISP, our_colormap, color_list, color_list_len, 0);
	    color_list_len = 0;
	}
	while (bg_head != NULL) {
	    struct bgrec	*bgp;
	    struct fgrec	*fgp;

	    for (fgp = bg_head->fg_head; fgp != NULL;) {
		struct fgrec *fgp1 = fgp->next;
		free(fgp);
		fgp = fgp1;
	    }
	    bgp = bg_head->next;
	    free(bg_head);
	    bg_head = bgp;
	}
#if GREY
	if (use_grey) {
	    struct font *f;
	    struct glyph *g;

	    for (f = font_head; f != NULL; f = f->next)
		if ((f->flags & FONT_LOADED) && !(f->flags & FONT_VIRTUAL))
		    for (g = f->glyph; g <= f->glyph + f->maxchar; ++g)
			g->fg = NULL;
	}
#endif /* GREY */
	bg_current = NULL;
	fg_active = NULL;
	color_warned = False;
}

/*
 *	All of the above, plus discard all scanned information.
 */

void
full_reset_colors()
{
	if (page_colors != NULL) {
	    int			i;
	    _Xconst struct rgb	*last_freed = &fg_initial;

	    for (i = 0; i < total_pages; ++i) {
		if (page_colors[i].colorstack != last_freed) {
		    last_freed = page_colors[i].colorstack;
		    if (last_freed != NULL)
			free((char *) last_freed);
		}
	    }
	    free((char *) page_colors);
	    page_colors = NULL;
	}
	reset_colors();
}

#endif /* COLOR */


/*
 *	realloc_font allocates the font structure to contain (newsize + 1)
 *	characters.
 */

void
realloc_font(fontp, newsize)
	struct font	*fontp;
	wide_ubyte	newsize;
{
	struct glyph *glyph;

	glyph = fontp->glyph = xrealloc(fontp->glyph,
	    (unsigned int) (newsize + 1) * sizeof(struct glyph));
	if (newsize > fontp->maxchar)
	    bzero((char *) (glyph + fontp->maxchar + 1),
		(int) (newsize - fontp->maxchar) * sizeof(struct glyph));
	maxchar = fontp->maxchar = newsize;
}


/*
 *	realloc_virtual_font does the same thing for virtual fonts.
 */

void
realloc_virtual_font(fontp, newsize)
	struct font	*fontp;
	wide_ubyte	newsize;
{
	struct macro *macro;

	macro = fontp->macro = xrealloc(fontp->macro,
	    (unsigned int) (newsize + 1) * sizeof(struct macro));
	if (newsize > fontp->maxchar)
	    bzero((char *) (macro + fontp->maxchar + 1),
		(int) (newsize - fontp->maxchar) * sizeof(struct macro));
	maxchar = fontp->maxchar = newsize;
}


/*
 *	load_font locates the raster file and reads the index of characters,
 *	plus whatever other preprocessing is done (depending on the format).
 *	Returns True if successful; False if not.
 */

Boolean
load_font(fontp)
	struct font *fontp;
{
	double	fsize	= fontp->fsize;
	int	dpi	= fsize + 0.5;
	char	*font_found;
	int	size_found;
	int	magic;
	Boolean	hushcs	= hush_chk;

	if (!font_open(fontp, &font_found, &size_found)) {
	    if (ev_flags & EV_GE_NEWDOC)
		return False;
	    fontp->flags |= FONT_LOADED;	/* as loaded as it'll get */
	    Fprintf(stderr, "%s: can't find font %s.\n", prog, fontp->fontname);
	    return False;
	}
	fontp->flags |= FONT_LOADED;
	if (font_found != NULL) {
	    Fprintf(stderr,
		    "%s: can't find font %s; using %s instead at %d dpi.\n",
		    prog, fontp->fontname, font_found, dpi);
	    free(fontp->fontname);
	    fontp->fontname = font_found;
	    hushcs = True;
	}
	else if (size_found > (int) (1.002 * fsize + 0.5) ||
		size_found < (int) (0.998 * fsize + 0.5))
	    Fprintf(stderr,
		"%s: can't find font %s at %d dpi; using %d dpi instead.\n",
		prog, fontp->fontname, dpi, size_found);
	fontp->timestamp = ++current_timestamp;
	fontp->maxchar = maxchar = 255;
	fontp->set_char_p = set_char;
#if FREETYPE
	if (fontp->ft != NULL) {	/* if freetype font */
	    fontp->set_char_p = set_ft_char;
	    fontp->glyph = xmalloc(256 * sizeof(struct glyph));
	    bzero((char *) fontp->glyph, 256 * sizeof(struct glyph));
	    return True;
	}
#endif
	fontp->fsize = size_found;
	magic = two(fontp->file);
	if (magic == PK_MAGIC) read_PK_index(fontp, WIDENINT hushcs);
	else
#if USE_GF
	    if (magic == GF_MAGIC) read_GF_index(fontp, WIDENINT hushcs);
	else
#endif
	    if (magic == VF_MAGIC) read_VF_index(fontp, WIDENINT hushcs);
	else
	    oops("Cannot recognize format for font file %s", fontp->filename);

	if (fontp->flags & FONT_VIRTUAL) {
	    while (maxchar > 0 && fontp->macro[maxchar].pos == NULL) --maxchar;
	    if (maxchar < 255)
		realloc_virtual_font(fontp, WIDENINT maxchar);
	}
	else {
	    while (maxchar > 0 && fontp->glyph[maxchar].addr == 0) --maxchar;
	    if (maxchar < 255)
		realloc_font(fontp, WIDENINT maxchar);
	}
	return True;
}


/*
 *	MAGSTEPVALUE - If the given magnification is close to a \magstep
 *	or a \magstephalf, then return twice the number of \magsteps.
 *	Otherwise return NOMAGSTP.
 */

#define	NOMAGSTP (-29999)
#define	NOBUILD	29999

static	int
magstepvalue(mag)
	float	*mag;
{
	int	m	= 0;
	double	fmag	= *mag;
	double	xmag	= pixels_per_inch;
	float	margin	= fmag * 0.002;

	if (fmag < pixels_per_inch)
	    for (;;) {
		if (xmag - fmag < margin && -(xmag - fmag) < margin) {
		    *mag = xmag;
		    return m;
		}
		if (xmag < fmag) break;
		xmag *= 0.9128709292;
		--m;
	    }
	else
	    for (;;) {
		if (xmag - fmag < margin && -(xmag - fmag) < margin) {
		    *mag = xmag;
		    return m;
		}
		if (xmag > fmag) break;
		xmag *= 1.095445115;
		++m;
	    }
	return NOMAGSTP;
}


/*
 *	reuse_font recursively sets the flags for font structures being reused.
 */

static	void
reuse_font(fontp)
	struct font *fontp;
{
	struct font **fp;
	struct tn *tnp;

	if (fontp->flags & FONT_IN_USE) return;

	fontp->flags |= FONT_IN_USE;
	if (list_fonts)
	    Printf("(reusing) %s at %d dpi\n", fontp->fontname,
		(int) (fontp->fsize + 0.5));
	if (fontp->flags & FONT_VIRTUAL) {
	    for (fp = fontp->vf_table; fp < fontp->vf_table + VFTABLELEN; ++fp)
		if (*fp != NULL) reuse_font(*fp);
	    for (tnp = fontp->vf_chain; tnp != NULL; tnp = tnp->next)
		reuse_font(tnp->fontp);
	}
}


/*
 *      define_font reads the rest of the fntdef command and then reads in
 *      the specified pixel file, adding it to the global linked-list holding
 *      all of the fonts used in the job.
 */
struct font *
define_font(file, cmnd, vfparent, tntable, tn_table_len, tn_headpp)
	FILE		*file;
	wide_ubyte	cmnd;
	struct font	*vfparent;	/* vf parent of this font, or NULL */
	struct font	**tntable;	/* table for low TeXnumbers */
	unsigned int	tn_table_len;	/* length of table for TeXnumbers */
	struct tn	**tn_headpp;	/* addr of head of list of TeXnumbers */
{
	int	TeXnumber;
	struct font *fontp;
	float	fsize;
	double	scale_dimconv;
	long	checksum;
	int	scale;
	int	design;
	int	magstepval;
	int	len;
	char	*fontname;
	int	size;

	TeXnumber = num(file, (int) cmnd - FNTDEF1 + 1);
	checksum = four(file);
	scale = four(file);
	design = four(file);
	len = one(file); len += one(file); /* sequence point in the middle */
	fontname = xmalloc((unsigned) len + 1);
	Fread(fontname, sizeof(char), len, file);
	fontname[len] = '\0';
	if(debug & DBG_PK)
	    Printf("Define font \"%s\" scale=%d design=%d\n",
		fontname, scale, design);
	if (vfparent == NULL) {
	    fsize = 0.001 * scale / design * magnification * pixels_per_inch;
	    scale_dimconv = dimconv;
	}
	else {
	    /*
	     *	The scaled size is given in units of vfparent->scale * 2 ** -20
	     *	SPELL units, so we convert it into SPELL units by multiplying by
	     *		vfparent->dimconv.
	     *	The design size is given in units of 2 ** -20 pt, so we convert
	     *	into SPELL units by multiplying by
	     *		(pixels_per_inch * 2**16) / (72.27 * 2**20).
	     */
	    fsize = (72.27 * (1<<4)) * vfparent->dimconv * scale / design;
	    scale_dimconv = vfparent->dimconv;
	}
	magstepval = magstepvalue(&fsize);
	size = fsize + 0.5;
	/*
	 * reuse font if possible
	 */
	for (fontp = font_head;; fontp = fontp->next) {
	    if (fontp == NULL) {		/* if font doesn't exist yet */
		if (list_fonts)
		    Printf("%s at %d dpi\n", fontname, (int) (fsize + 0.5));
		fontp = (struct font *) xmalloc((unsigned) sizeof(struct font));
		fontp->fontname = fontname;
		fontp->fsize = fsize;
		fontp->magstepval = magstepval;
		fontp->file = NULL;	/* needed for virtual/freetype fonts */
		fontp->filename = NULL;	/* needed for freetype fonts */
		fontp->checksum = checksum;
		fontp->flags = FONT_IN_USE;
		fontp->dimconv = scale * scale_dimconv / (1<<20);
		fontp->set_char_p = load_n_set_char;
#if FREETYPE
		fontp->ft = NULL;
		/* spsize = scaled size of the font in spell units,
		 *	  = scale * [vfparent->]dimconv.
		 */
		fontp->spsize =
		  (vfparent != NULL ? vfparent->dimconv : dimconv) * scale;
#endif
		if (vfparent == NULL)
		    if (!load_font(fontp)) {
			if (ev_flags & EV_GE_NEWDOC) {	/* if aborting */
			    free(fontname);
			    free(fontp);
			    return NULL;
			}
			font_not_found = True;
		    }
		fontp->next = font_head;
		font_head = fontp;
		break;
	    }
	    if (strcmp(fontname, fontp->fontname) == 0
		    && size == (int) (fontp->fsize + 0.5)) {
			/* if font already in use */
		reuse_font(fontp);
		free(fontname);
		break;
	    }
	}
	if (TeXnumber < tn_table_len)
	    tntable[TeXnumber] = fontp;
	else {
	    struct tn *tnp;
	    tnp = xmalloc((unsigned) sizeof(struct tn));
	    tnp->next = *tn_headpp;
	    *tn_headpp = tnp;
	    tnp->TeXnumber = TeXnumber;
	    tnp->fontp = fontp;
	}
	return fontp;
}


/*
 *      process_preamble reads the information in the preamble and stores
 *      it into global variables for later use.
 */
static	void
process_preamble()
{
	ubyte   k;

	if (one(dvi_file) != PRE)
		dvi_oops("Not a DVI file");
	if (one(dvi_file) != 2)
		dvi_oops("Wrong version of DVI output for this program");
	numerator     = four(dvi_file);
	denominator   = four(dvi_file);
	magnification = four(dvi_file);
	dimconv = (((double) numerator * magnification)
		/ ((double) denominator * 1000.));
	dimconv = dimconv * (((long) pixels_per_inch)<<16) / 254000;
	tpic_conv = pixels_per_inch * magnification / 1000000.0;
	k = one(dvi_file);
	Fread(job_id, sizeof(char), (int) k, dvi_file);
	job_id[k] = '\0';
}

/*
 *      find_postamble locates the beginning of the postamble
 *	and leaves the file ready to start reading at that location.
 */
#define	TMPSIZ	516	/* 4 trailer bytes + 512 junk bytes allowed */
static	void
find_postamble()
{
	long	pos;
	ubyte	temp[TMPSIZ];
	ubyte	*p;
	ubyte	*p1;
	ubyte	byte;

	Fseek(dvi_file, (long) 0, 2);
	pos = ftell(dvi_file) - TMPSIZ;
	if (pos < 0) pos = 0;
	Fseek(dvi_file, pos, 0);
	p = temp + fread((char *) temp, sizeof(char), TMPSIZ, dvi_file);
	for (;;) {
	    p1 = p;
	    while (p1 > temp && *(--p1) != TRAILER) ;
	    p = p1;
	    while (p > temp && *(--p) == TRAILER) ;
	    if (p <= p1 - 4) break;	/* found 4 TRAILER bytes */
	    if (p <= temp) dvi_oops("DVI file corrupted");
	}
	pos += p - temp;
	byte = *p;
	while (byte == TRAILER) {
	    Fseek(dvi_file, --pos, 0);
	    byte = one(dvi_file);
	}
	if (byte != 2)
	    dvi_oops("Wrong version of DVI output for this program");
	Fseek(dvi_file, pos - 4, 0);
	Fseek(dvi_file, sfour(dvi_file), 0);
}


/*
 *      read_postamble reads the information in the postamble,
 *	storing it into global variables.
 *      It also takes care of reading in all of the pixel files for the fonts
 *      used in the job.
 */
static	Boolean
read_postamble()
{
	ubyte   cmnd;
	struct font	*fontp;
	struct font	**fontpp;

	if (one(dvi_file) != POST)
	    dvi_oops("Postamble doesn't begin with POST");
	last_page_offset = four(dvi_file);
	if (numerator != four(dvi_file)
		|| denominator != four(dvi_file)
		|| magnification != four(dvi_file))
	    dvi_oops("Postamble doesn't match preamble");
		/* read largest box height and width */
	dvi_unshrunk_page_h = (spell_conv(sfour(dvi_file)) >> 16) + offset_y;
	if (dvi_unshrunk_page_h < unshrunk_paper_h)
	    dvi_unshrunk_page_h = unshrunk_paper_h;
	dvi_unshrunk_page_w = (spell_conv(sfour(dvi_file)) >> 16) + offset_x;
	if (dvi_unshrunk_page_w < unshrunk_paper_w)
	    dvi_unshrunk_page_w = unshrunk_paper_w;
	(void) two(dvi_file);	/* max stack size */
	total_pages = two(dvi_file);
	font_not_found = False;
	while ((cmnd = one(dvi_file)) >= FNTDEF1 && cmnd <= FNTDEF4)
	    if (define_font(dvi_file, cmnd, (struct font *) NULL,
	      tn_table, TNTABLELEN, &tn_head) == NULL)
		return False;
	if (cmnd != POSTPOST)
	    dvi_oops("Non-fntdef command found in postamble");
	if (font_not_found)
	    dvi_oops("Not all pixel files were found");
	/*
	 * free up fonts no longer in use
	 */
	fontpp = &font_head;
	while ((fontp = *fontpp) != NULL)
	    if (fontp->flags & FONT_IN_USE)
		fontpp = &fontp->next;
	    else {
		if (debug & DBG_PK)
		    Printf("Discarding font \"%s\" at %d dpi\n",
			fontp->fontname, (int) (fontp->fsize + 0.5));
		*fontpp = fontp->next;		/* remove from list */
		free(fontp->fontname);
		if (fontp->flags & FONT_LOADED) {
#if FREETYPE
		    if (fontp->ft != NULL) {	/* if FreeType font */
			struct ftfont	*ft;

			ft = fontp->ft;
			if (fontp->size != NULL)
			    FT_Done_Size(fontp->size);
			if (fontp == ft->first_size) {
			    if (fontp->next_size == NULL) {
				/* if this is the only size of this font face */
				FT_Done_Face(ft->face);
				ft->t1->ft = NULL;
				free(ft);
			    }
			    else {
				struct font	*fp2;

				ft->first_size = fp2 = fontp->next_size;
				fp2->file = fontp->file;
				fontp->file = NULL;
				fp2->filename = fontp->filename;
				fontp->filename = NULL;
				fp2->timestamp = fontp->timestamp;
			    }
			}
			else {
			    struct font *fp2;

			    fp2 = ft->first_size;
			    while (fp2->next_size != fontp)
				fp2 = fp2->next_size;
			    fp2->next_size = fontp->next_size;
			}
		    }
#endif
		    if (fontp->file != NULL) {
			Fclose(fontp->file);
			++n_files_left;
		    }
#if FREETYPE
		    if (fontp->filename != NULL)
#endif
			free((void *) fontp->filename);

		    if (fontp->flags & FONT_VIRTUAL) {
			struct macro *m;

			for (m = fontp->macro;
				m <= fontp->macro + fontp->maxchar; ++m)
			    if (m->free_me) free((char *) m->pos);
			free((char *) fontp->macro);
			free((char *) fontp->vf_table);
			free_vf_chain(fontp->vf_chain);
		    }
		    else {
			struct glyph *g;

			for (g = fontp->glyph;
				g <= fontp->glyph + fontp->maxchar; ++g) {
			    if (g->bitmap.bits != NULL) free(g->bitmap.bits);
			    if (g->bitmap2.bits != NULL) free(g->bitmap2.bits);
#ifdef	GREY
			    if (g->pixmap2 != NULL) {
				XDestroyImage(g->image2);
				if (g->pixmap2_t != NULL)
				    free(g->pixmap2_t);
			    }
#endif
			}
			free((char *) fontp->glyph);
		    }
		    free((char *) fontp);
		}
	    }
	return True;
}

static	void
prepare_pages()
{
	int i;

	page_info = xmalloc((unsigned) (total_pages + 1) * sizeof *page_info);
	page_info->pw = unshrunk_paper_w;
	page_info->ph = unshrunk_paper_h;
	page_info->ww = dvi_unshrunk_page_w;
	page_info->wh = dvi_unshrunk_page_h;
	++page_info;	/* this is indexed starting with -1 */

	i = total_pages;
	page_info[--i].offset = last_page_offset;
	Fseek(dvi_file, last_page_offset, 0);
	/*
	 * Follow back pointers through pages in the DVI file,
	 * storing the offsets in the page_info[].offset table.
	 */
	while (i > 0) {
	    Fseek(dvi_file, (long) (1+4+(9*4)), 1);
	    Fseek(dvi_file, page_info[--i].offset = four(dvi_file), 0);
	}

	/* If not prescanning, initialize page sizes.  */
	if (!resource.prescan)
	    for (i = 0; i < total_pages; ++i) {
		page_info[i].pw = unshrunk_paper_w;
		page_info[i].ph = unshrunk_paper_h;
		page_info[i].ww = dvi_unshrunk_page_w;
		page_info[i].wh = dvi_unshrunk_page_h;
	    }
}

void
init_page()
{
	unshrunk_page_w = page_info[current_page].ww;
	page_w = ROUNDUP(unshrunk_page_w, mane.shrinkfactor) + 2;
	unshrunk_page_h = page_info[current_page].wh;
	page_h = ROUNDUP(unshrunk_page_h, mane.shrinkfactor) + 2;
}

/**
 **	form_dvi_property forms the property used to exhibit the dvi file name
 **	used as a window property (used for source specials).
 **/

void
form_dvi_property(inode)
	ino_t		inode;
{
	size_t		len;
	char		*p;
	unsigned long	ino;
	int		i;

	if (dvi_file == NULL)
	    return;

	if (dvi_property != NULL)
	    free(dvi_property);

	p = rindex(dvi_name, '/');
	if (p != NULL) ++p; else p = dvi_name;
	len = strlen(p);

	dvi_property_length = len + 8;
	dvi_property = xmalloc(dvi_property_length);

	/* Do it this way because different copies of xdvi might be running on
	   machines with different endiannesses.  */
	ino = dvi_inode = inode;
	for (i = 0; i < 8; ++i) {
	    dvi_property[i] = ino;
	    ino >>= 8;
	}
	memcpy(dvi_property + 8, p, len);
}

#if TOOLKIT
	/* For non-toolkit compiles, this code appears in xdvi.c.  */

static	void
set_titles()
{
	char	*title_name;
	char	*icon_name;
	size_t	baselen;
	Boolean	icon_name_malloced = False;

	icon_name = rindex(dvi_name, '/');
	if (icon_name != NULL) ++icon_name;
	  else icon_name = dvi_name;
	baselen = strlen(icon_name);
	if (baselen > 4 && memcmp(icon_name + baselen - 4, ".dvi", 4) == 0) {
	    /* remove the .dvi */
	    char *p;

	    baselen -= 4;
	    p = xmalloc(baselen + 1);
	    memcpy(p, icon_name, baselen);
	    p[baselen] = '\0';
	    icon_name = p;
	    icon_name_malloced = True;
	}

	title_name = xmalloc(baselen + 8);
	memcpy(title_name, "Xdvi:  ", 7);
	memcpy(title_name + 7, icon_name, baselen + 1);

	XtVaSetValues(top_level,
	  XtNtitle, title_name,
	  XtNiconName, icon_name,
	  NULL);

	free(title_name);
	if (icon_name_malloced)
	    free(icon_name);

	titles_are_stale = False;
}

#endif


/**
 **	open_dvi_file opens the dvi file (if present), checks that it's not
 **	a directory, saves its time stamp, and forms its dvi property.
 **	Returns True if the operation succeeds.  Upon failure, returns False
 **	and dvi_file == NULL.
 **/

Boolean
open_dvi_file(path)
	_Xconst char	*path;
{
	struct stat	buf;
	ino_t		old_inode;

	dvi_file = fopen(path, OPEN_MODE);
	if (dvi_file == NULL)
	    return False;

	if (fstat(fileno(dvi_file), &buf) != 0
	  || S_ISDIR(buf.st_mode)) {	/* if it's a directory */
	    fclose(dvi_file);
	    dvi_file = NULL;
	    return False;
	}

	dvi_time = buf.st_mtime;

	old_inode = dvi_inode;
	form_dvi_property(buf.st_ino);
	if (dvi_inode != old_inode && mane.win != (Window) 0)
	    set_dvi_property();

	return True;
}

/*
 *	internal_init_dvi is the main subroutine for reading the startup
 *	information from the dvi file.
 */

static	Boolean
internal_init_dvi()
{
	process_preamble();
	find_postamble();
	if (!read_postamble()) {
	    Fclose(dvi_file);
	    dvi_file = NULL;
	    return False;
	}
	prepare_pages();
	if (current_page >= total_pages) current_page = total_pages - 1;
	warn_spec_now = warn_spec;
	source_fwd_box_page = -1;
	init_prescan();
#if TOOLKIT
	if (titles_are_stale)
	    set_titles();
#endif
	dvi_is_valid = True;
	return True;
}

/**
 **	init_dvi_file initializes the dvi file.  Called only from main().
 **/

void
init_dvi_file()
{
	if (setjmp(dvi_env)) oops(dvi_oops_msg);

	(void) internal_init_dvi();
}

/**
 **	set_dvi_property sets the appropriate property for the main
 **	window (used in source special handoff).
 **/

void
set_dvi_property()
{
	XChangeProperty(DISP, XtWindow(top_level), ATOM_DVI_FILE, ATOM_DVI_FILE,
	  8, PropModeReplace, dvi_property, dvi_property_length);
}

#if TOOLKIT

static	Widget	dvi_popup = NULL;	/* warning popup for corrupt dvi file */

/* ARGSUSED */
static	void
dvi_callback(w, client_data, call_data)
	Widget		w;
	XtPointer	client_data;
	XtPointer	call_data;
{
	ev_flags |= EV_NEWDOC;
	if (dvi_popup != NULL) {
	    XtDestroyWidget(dvi_popup);
	    dvi_popup = NULL;
	}
}

#endif /* TOOLKIT */

/**
 **	Check for changes in dvi file.  Return True if it's OK.
 **/

Boolean
check_dvi_file()
{
	struct stat	buf;

	if (!dvi_is_valid)
	    return False;

	if (dvi_file == NULL) {
	    /* check if file exists and has changed */
	    if (!(ev_flags & EV_NEWDOC) && stat(dvi_name, &buf) == 0
	      && buf.st_mtime != dvi_time)
		ev_flags |= EV_NEWDOC;
	    return False;
	}
	else if (fstat(fileno(dvi_file), &buf) != 0
	  || buf.st_mtime != dvi_time) {
	    ev_flags |= EV_NEWDOC;
	    Fclose(dvi_file);
	    dvi_file = NULL;
	    dvi_file_ready = False;
	    return False;
	}

	return True;
}

/**
 **	Reload the dvi file (unconditionally).  Called only from do_pages().
 **/

void
reload_dvi_file()
{
	struct font	*fontp;

	dvi_is_valid = dvi_file_ready = False;
	if (page_info != NULL) {
	    free((char *) (page_info - 1));
	    page_info = NULL;
#if COLOR
	full_reset_colors();
#endif /* COLOR */
	}
#if TOOLKIT
	if (dvi_popup != NULL) {
	    int i;

	    for (i = 0;; ++i) {
		if (i >= n_init_popups) {
		    XtDestroyWidget(dvi_popup);
		    break;
		}
		if (XtParent(init_popups[i]) == dvi_popup) {
		    while (++i < n_init_popups)
			init_popups[i - 1] = init_popups[i];
		    --n_init_popups;
		    break;
		}
	    }
	    dvi_popup = NULL;
	}
#endif /* TOOLKIT */
	bzero((char *) tn_table, (int) sizeof(tn_table));
	free_vf_chain(tn_head);
	tn_head = NULL;
	for (fontp = font_head; fontp != NULL; fontp = fontp->next)
	    fontp->flags &= ~FONT_IN_USE;

	if (!setjmp(dvi_env)) {
	    if (dvi_file == NULL) {
		if (!open_dvi_file(dvi_name))
		    dvi_oops("Cannot reopen dvi file.");
	    }
	    else {
		struct stat	buf;

		if (fstat(fileno(dvi_file), &buf) == 0)
		    dvi_time = buf.st_mtime;
		Fseek(dvi_file, (long) 0, 0);
	    }
	    if (!internal_init_dvi())
		return;
	    if (list_fonts) Putchar('\n');
	}
	else {	/* if error */
	    XClearWindow(DISP, mane.win);
#if TOOLKIT
	    dvi_popup = warning_popup_long(
	      "Xdvi cannot read the DVI file\n%s\nReason:  %s",
# if MOTIF
	      XmDIALOG_ERROR,
# endif
	      "Reload", dvi_callback, dvi_name, dvi_oops_msg);
#else /* no toolkit */
	    showmessage(dvi_oops_msg);
#endif
	    if (dvi_file != NULL) {
		Fclose(dvi_file);
		dvi_file = NULL;
	    }
	}
}
