/*========================================================================*\

Copyright (c) 1990-2013  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL PAUL VOJTA OR ANY OTHER AUTHOR OF OR CONTRIBUTOR TO
THIS SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
IN THE SOFTWARE.

NOTE:
	xdvi is based on prior work, as noted in the modification history
	in xdvi.c.

\*========================================================================*/

#include "filf-app.h"	/* application-related defs, etc. */
#include "filefind.h"

#include <errno.h>
#include <signal.h>

#ifdef	X_NOT_STDC_ENV
extern	int	errno;
extern	char	*getenv ARGS((_Xconst char *));
#endif

/*
 *	If you think you have to change DEFAULT_TAIL, then you haven't read the
 *	documentation closely enough.
 */

#ifndef	VMS
#define	PATH_SEP	':'
#define	DEFAULT_TAIL	"/%f.%d%p"
#define	DEFAULT_VF_TAIL	"/%f.vf"
#else	/* VMS */
#define	PATH_SEP	'/'
#define	DEFAULT_TAIL	":%f.%d%p"
#define	DEFAULT_VF_TAIL	":%f.vf"
#endif	/* VMS */

#ifndef DEFAULT_FONT_SIZES
	/* default sizes for current dpi */
#define	DEFAULT_FONT_SIZES	"m0:m0.5:m1:m2:m3:m4:m5"
#endif

#ifdef MKTEXPK
static	_Xconst	char	*makepkcmd		= NULL;
#endif

/*
 *	Information on how to search for pk and gf files.
 */

static	_Xconst	char	no_f_str_pkgf[]	= DEFAULT_TAIL;

static	struct findrec			search_pkgf	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_FONT_PATH,
	/* type		*/	"font",
	/* fF_etc	*/	"fFdbpm",
	/* x_var_char	*/	'd',
	/* n_var_opts	*/	3,
	/* no_f_str	*/	no_f_str_pkgf,
	/* no_f_str_end	*/	no_f_str_pkgf + sizeof(no_f_str_pkgf) - 1,
	/* abs_str	*/	"%f.%d%p",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED | F_PK_USED,
	/* abs_str_flags */	F_FILE_USED | F_PK_USED,
	/* pk_opt_char	*/	'p',
	/* pk_gf_addr	*/	&fF_values[4],
#endif
	/* pct_s_str	*/	"%qfonts/%p/%m//:%qfonts/%p/modeless//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

#ifdef	DOSNAMES

static	_Xconst	char	no_f_str_dos[]	= "/dpi%d/%f.%p";

static	struct findrec			search_pkgf_dos	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_FONT_PATH,
	/* type		*/	"font",
	/* fF_etc	*/	"fFdbpm",
	/* x_var_char	*/	'd',
	/* n_var_opts	*/	3,
	/* no_f_str	*/	no_f_str_dos,
	/* no_f_str_end	*/	no_f_str_dos + sizeof(no_f_str_dos) - 1,
	/* abs_str	*/	"%f.%d%p",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED | F_PK_USED,
	/* abs_str_flags */	F_FILE_USED | F_PK_USED,
	/* pk_opt_char	*/	'p',
	/* pk_gf_addr	*/	&fF_values[4],
#endif
	/* pct_s_str	*/	"%qfonts/%p/%m//:%qfonts/%p/modeless//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};
#endif	/* DOSNAMES */

/*
 *	Information on how to search for vf files.
 */

static	_Xconst	char	no_f_str_vf[]	= DEFAULT_VF_TAIL;

static	struct findrec			search_vf	= {
	/* path1	*/	NULL,
#if	CFGFILE
	/* envptr	*/	NULL,
#endif
	/* path2	*/	DEFAULT_VF_PATH,
	/* type		*/	"vf",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',		/* i.e., none */
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_vf,
	/* no_f_str_end	*/	no_f_str_vf + sizeof(no_f_str_vf) - 1,
	/* abs_str	*/	"%f.vf",
#ifdef	USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',		/* none */
	/* pk_gf_addr	*/	NULL,
#endif
	/* pct_s_str	*/	"%qfonts/vf//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

static	int	*sizes, *sizend;
static	char	default_size_list[]	= DEFAULT_FONT_SIZES;

static	char		bdpi_string[10];
static	char		dpi_string[10];

static	double		magsteps[10]	= {1.0, 1.2, 1.44, 1.728, 2.0736,
					   2.48832, 2.985984, 3.5831808,
					   4.29981696, 5.159780352};

#if FREETYPE || PS

/*
 *	Information on how to search for dvips configuration files.
 */

static	_Xconst	char	no_f_str_dvips_cf[]	= "/%f";

static	struct findrec			search_dvips_cf	= {
	/* path1	*/	NULL,
# if CFGFILE
	/* envptr	*/	NULL,
# endif
	/* path2	*/	DEFAULT_DVIPS_CF_PATH,
	/* type		*/	"dvips config",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',		/* i.e., none */
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_dvips_cf,
	/* no_f_str_end	*/	no_f_str_dvips_cf + sizeof no_f_str_dvips_cf -1,
	/* abs_str	*/	"%f",
# if USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',		/* none */
	/* pk_gf_addr	*/	NULL,
# endif
	/* pct_s_str	*/	"%qdvips//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

/*
 *	Information on how to search for (dvips-style) map files.
 */

static	_Xconst	char	no_f_str_fontmap[]	= "/%f";

static	struct findrec			search_fontmap	= {
	/* path1	*/	NULL,
# if CFGFILE
	/* envptr	*/	NULL,
# endif
	/* path2	*/	DEFAULT_FONTMAP_PATH,
	/* type		*/	"font map",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',		/* i.e., none */
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_fontmap,
	/* no_f_str_end	*/	no_f_str_fontmap + sizeof no_f_str_fontmap -1,
	/* abs_str	*/	"%f",
# if USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',		/* none */
	/* pk_gf_addr	*/	NULL,
# endif
	/* pct_s_str	*/    "%qfonts/map/{xdvi,dvips,pdftex}//:%qfonts/map//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

/*
 *	Information on how to search for font encoding files.
 */

static	_Xconst	char	no_f_str_enc[]	= "/%f";

static	struct findrec			search_enc	= {
	/* path1	*/	no_f_str_enc,	/* flag value:  uninitialized */
# if CFGFILE
	/* envptr	*/	NULL,
# endif
	/* path2	*/	DEFAULT_ENC_PATH,
	/* type		*/	"font encoding",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',		/* i.e., none */
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_enc,
	/* no_f_str_end	*/	no_f_str_enc + sizeof no_f_str_enc -1,
	/* abs_str	*/	"%f",
# if USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',		/* none */
	/* pk_gf_addr	*/	NULL,
# endif
	/* pct_s_str	*/	"%qfonts/enc//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

/*
 *	Information on how to search for Type 1 fonts.
 */

static	_Xconst	char	no_f_str_type1[]	= "/%f";

static	struct findrec			search_type1	= {
	/* path1	*/	no_f_str_type1,	/* flag value:  uninitialized */
# if CFGFILE
	/* envptr	*/	NULL,
# endif
	/* path2	*/	DEFAULT_TYPE1_PATH,
	/* type		*/	"Type 1 font",
	/* fF_etc	*/	"fF",
	/* x_var_char	*/	'f',
	/* n_var_opts	*/	2,
	/* no_f_str	*/	no_f_str_type1,
	/* no_f_str_end	*/	no_f_str_type1 + sizeof(no_f_str_type1) - 1,
	/* abs_str	*/	"%f",
# if USE_GF
	/* no_f_str_flags */	F_FILE_USED,
	/* abs_str_flags */	F_FILE_USED,
	/* pk_opt_char	*/	'f',
	/* pk_gf_addr	*/	NULL,
# endif
	/* pct_s_str	*/	"%qfonts/type1//",
	{
	  /* v.stephead		*/	NULL,
	  /* v.pct_s_head	*/	NULL,
	  /* v.pct_s_count	*/	0,
	  /* v.pct_s_atom	*/	NULL,
	  /* v.rootp		*/	NULL,
	}
};

#endif	/* FREETYPE || PS */


static	int
atosize(p)
	_Xconst	char	*p;
{
	_Xconst	char	*q;
	Boolean		minus;
	double		factor;

	if (*p != 'm')
	    return atoi(p);

	for (q = "agstep";; ++q) {
	    ++p;
	    if (*q == '\0' || *p != *q) break;
	}

	minus = False;
	if (*p == '-') {
	    minus = True;
	    ++p;
	}

	if (*p < '0' || *p > '9')
	    return 0;
	factor = magsteps[*p - '0'];
	++p;

	if (*p == '.' && p[1] == '5')
	    factor *= 1.0954451150103321;	/* sqrt(1.2) */

	return (int)
	  ((minus ? pixels_per_inch / factor : pixels_per_inch * factor) + 0.5);
}

static	void
get_sizes(size_list, spp)
	char	*size_list;
	int	**spp;
{
	if (*size_list == PATH_SEP) ++size_list;
	for (;;) {
	    *(*spp)++ = atosize(size_list);
	    size_list = index(size_list, PATH_SEP);
	    if (size_list == NULL) return;
	    ++size_list;
	}
}


#if FREETYPE || PS

/*
 *	The following code handles lookup of Type 1 fonts for use as FreeType
 *	fonts, and for use within MetaPost output.
 *	The system psfonts.map file is read into an AVL tree as needed (lazy
 *	evaluation), and searched for Type 1 fonts.
 */

struct p_list {					/* list of map file names */
	struct p_list	*next;
	_Xconst char	*value;
};

	/* Initialize this list to "psfonts.map".  */

static	struct p_list	psfonts_map		= {NULL, "psfonts.map"};

static	struct p_list	*p_head			= &psfonts_map;
static	struct p_list	**p_tail		= &psfonts_map.next;

static	FILE		*mapfile		= NULL;

static	Boolean
fgets_long(f)
	FILE	*f;
{
	int	len;

	if (fgets(ffline, ffline_len, f) == NULL)
	    return False;

	len = 0;
	for (;;) {
	    len += strlen(ffline + len);
	    if (len > 0 && ffline[len - 1] == '\n') {
		ffline[--len] = '\0';
		break;
	    }
	    if (len < ffline_len - 1)
		break;
	    expandline(len);
	    fgets(ffline + len, ffline_len - len, f);
	}

	return True;
}

/*
 *	Get list of map files from dvips config file(s).
 */

static	void
getdefaults(f)
	FILE	*f;
{
	char		*p, *q;
	int		len;
	struct p_list	*p_node;

	while (fgets_long(f)) {
	    p = ffline;
	    while (*p == ' ' || *p == '\t') ++p;
	    if (*p == 'p') {
		do ++p;
		while (*p == ' ' || *p == '\t');

		if (*p == '+')
		    do ++p;
		    while (*p == ' ' || *p == '\t');
		else {	/* discard old list */
		    struct p_list *pl, *pl2;

		    *p_tail = NULL;
		    pl = p_head;
		    if (pl == &psfonts_map) pl = pl->next;
		    while (pl != NULL) {
			free((char *) pl->value);
			pl2 = pl->next;
			free(pl);
			pl = pl2;
		    }
		    p_tail = &p_head;
		}

		/* get white-delimited argument */
		len = strlen(p);
		q = memchr(p, ' ', len);
		if (q != NULL) len = q - p;
		q = memchr(p, '\t', len);
		if (q != NULL) len = q - p;
		p[len] = '\0';

		p_node = xmalloc(sizeof *p_node);
		p_node->value = xmemdup(p, len + 1);
		*p_tail = p_node;
		p_tail = &p_node->next;
	    }
	}

	Fclose(f);
	++n_files_left;
}


/*
 *	Information about Type 1 fonts is stored in an AVL tree.
 */

static	struct avl_t1	*t1_head	= NULL;


/*
 *	Parse line from psfonts.map file.
 */

static struct avl_t1 *
dvips_parse(line)
	_Xconst char	*line;
{
	_Xconst char	*w1p, *encp, *pfp, *qp;
	size_t		w1l, encl, pfl, ql;
	_Xconst char	*w2p;
	size_t		w2l;
	_Xconst char	*p, *p0;
	_Xconst char	*err;
	struct avl_t1	*t1p;
	char		*q;

	w2l = w1l = encl = pfl = ql = 0;
	err = NULL;
	p = line;
	for (;;) {	/* loop over words */
	    while (*p == ' ' || *p == '\t') ++p;
	    if (*p == '\0')
		break;

	    if (*p == '"') {	/* quoted string */
		_Xconst char *p_end;

		p0 = p + 1;
		p_end = p0 + strlen(p0);
		p = memchr(p0, '"', p_end - p0);
		if (p == NULL) p = p_end;
		qp = (ql == 0 ? p0 : NULL);
		ql += p - p0 + 1;
		if (*p == '"') ++p;
		continue;
	    }

	    if (*p == '<') {	/* encoding or pfa/b file */
		int	wtype	= 0;

		++p;
		if (*p == '<') {
		    wtype = 1;	/* font file */
		    ++p;
		}
		else if (*p == '[') {
		    wtype = -1;	/* encoding file */
		    ++p;
		}

		/* find word */
		while (*p == ' ' || *p == '\t') ++p;
		p0 = p;
		while (*p != '\0' && *p != ' ' && *p != '\t') ++p;

		if (wtype == 0 && p > p0 + 4 && p[-4] == '.') {
		    if (memcmp(p - 3, "enc", 3) == 0
		      || memcmp(p - 3, "ENC", 3) == 0)
			wtype = -1;
		    else if (memcmp(p - 3, "pfa", 3) == 0
		      || memcmp(p - 3, "pfb", 3) == 0
		      || memcmp(p - 3, "PFA", 3) == 0
		      || memcmp(p - 3, "PFB", 3) == 0)
			wtype = 1;
		}

		if (wtype > 0) {
		    if (pfl != 0)
			err = "more than one font file given";
		    else {
			pfp = p0;
			pfl = p - p0 + 1;
		    }
		}
		else if (wtype < 0) {
		    if (encl != 0)
			err = "more than one encoding file given";
		    else {
			encp = p0;
			encl = p - p0 + 1;
		    }
		}
		else
		    err = "cannot identify file type";
	    }
	    else {	/* if ordinary word */
		p0 = p;
		while (*p != '\0' && *p != ' ' && *p != '\t') ++p;
		if (w1l == 0) {
		    w1p = p0;
		    w1l = p - p0;
		}
		else if (w2l == 0) {
		    w2p = p0;
		    w2l = p - p0;
		}
		else
		    err = "more than two non-download words given";
	    }
	}	/* end loop over words */

	if (w1l == 0) {
	    if (debug & DBG_OPEN)
		Fprintf(stderr,
		  "%s: map file %s: line \"%s\" does not give a font name.\n",
		  prog, p_head->value, line);
	    return NULL;
	}

	if (err != NULL) {
	    if (debug & DBG_OPEN)
		Fprintf(stderr, "%s: map file %s, font %.*s: %s\n", prog,
		  p_head->value, (int) w1l, w1p, err);
	    return NULL;
	}

	t1p = (struct avl_t1 *) avladd(w1p, w1l, (struct avl **) &t1_head,
	  sizeof(struct avl_t1));

	if (t1p->key != w1p) {	/* if existing record */
	    if (debug & DBG_OPEN)
		Fprintf(stderr,
		  "%s: map file %s, font %.*s: duplicate record; using first one\n",
		  prog, p_head->value, (int) w1l, w1p);
	    return NULL;
	}

	t1p->key = q = xmalloc(w1l + w2l + 1 + pfl + encl + ql);

	memcpy(q, w1p, w1l);
	q += w1l;
	t1p->psname = t1p->key;
	if (w2l != 0) {
	    t1p->psname = q;
	    memcpy(q, w2p, w2l);
	    q += w2l;
	}
	*q++ = '\0';

	t1p->fontfile = NULL;
	if (pfl != 0) {
	    t1p->fontfile = q;
	    memcpy(q, pfp, pfl - 1);
	    q += pfl;
	    q[-1] = '\0';
	}

	t1p->encname = t1p->addinfo = NULL;
# if FREETYPE
	t1p->bad = False;
	t1p->ft = NULL;
# endif

	if (encl != 0) {
	    t1p->encname = q;
	    memcpy(q, encp, encl - 1);
	    q += encl;
	    q[-1] = '\0';
	}

	if (ql != 0) {
	    t1p->addinfo = q;
	    if (qp != 0) {
		memcpy(q, qp, ql - 1);
		q += ql;
	    }
	    else {	/* multiple quoted strings; rescan to get them */
		_Xconst char	*p_end;

		p = line;
		p_end = p + strlen(p);
		for (;;) {
		    while (*p == ' ' || *p == '\t') ++p;
		    if (*p == '\0')
			break;

		    /* found a word */
		    if (*p == '"') {
			++p;
			p0 = p;
			p = memchr(p0, '"', p_end - p0);
			if (p == NULL) p = p_end;
			memcpy(q, p0, p - p0);
			q += p - p0;
			*q++ = ' ';
			if (*p == '\0') break;
			++p;
		    }
		    else	/* skip unquoted word */
			while (*p != '\0' && *p != ' ' && *p != '\t') ++p;
		}
	    }
	    q[-1] = '\0';
	}

	return t1p;
}

/*
 *	Information on the Ghostscript font aliasing mechanism is kept in
 *	another AVL tree.  Each identifier points to a list of strings,
 *	up to one from each Fontmap file.  Within a Fontmap file, later entries
 *	for a given string override earlier ones.
 */

struct avl_gs {		/* structure for gs font information */
	AVL_COMMON;
	short		fontmap_number;	/* sequence of most recent entry */
	Boolean		in_use;
	struct gs_list	*list;
};

struct gs_list {	/* list of Fontmap entries for this font name */
	struct gs_list	*next;
	_Xconst char	value[0];
};

static	struct avl_gs	*gs_head	= NULL;

typedef	void	(*gs_path_proc) ARGS((FILE *));

static	_Xconst char	*env_gs_lib;

static FILE *
gs_try_fopen(str, len, name)
	_Xconst char	*str;
	unsigned int	len;
	_Xconst char	*name;
{
	unsigned int	namelen	= strlen(name) + 1;
	FILE		*f;

	if (len + namelen + 1 > ffline_len) expandline(len + namelen + 1);
	memcpy(ffline, str, len);
	ffline[len] = '/';
	memcpy(ffline + len + 1, name, namelen);

	f = xfopen(ffline, OPEN_MODE);

	if (debug & DBG_OPEN)
	    Printf("gs_try_fopen: %s: %s\n", ffline,
	      f != NULL ? "file opened" : strerror(errno));

	return f;
}

static FILE *
gs_path_fopen(name, proc)
	_Xconst char	*name;
	gs_path_proc	proc;
{
	_Xconst	char	*str1	= env_gs_lib;
	_Xconst	char	*str2	= DEFAULT_GS_LIB_PATH;
	_Xconst	char	*str1_end, *str2_end;
	_Xconst char	*p1, *p2;
	FILE		*f;

	if (str1 == NULL) {
	    str1 = str2;
	    str2 = NULL;
	}

	str1_end = str1 + strlen(str1);

	for (;;) {
	    p1 = memchr(str1, ':', str1_end - str1);
	    if (p1 == NULL) p1 = str1_end;
	    if (p1 == str1) {
		if (str2 != NULL) {
		    str2_end = str2 + strlen(str2);
		    for (;;) {
			p2 = memchr(str2, ':', str2_end - str2);
			if (p2 == NULL) p2 = str2_end;
			if (p2 > str2) {
			    f = gs_try_fopen(str2, p2 - str2, name);
			    if (f != NULL) {
				if (proc != NULL)
				    proc(f);
				else
				    return f;
			    }
			}
			if (*p2 == '\0')
			    break;
			str2 = p2 + 1;
		    }
		    str2 = NULL;
		}
	    }
	    else {
		f = gs_try_fopen(str1, p1 - str1, name);
		if (f != NULL) {
		    if (proc != NULL)
			proc(f);
		    else
			return f;
		}
	    }

	    if (*p1 == '\0')
		break;
	    str1 = p1 + 1;
	}

	return NULL;
}


static FILE *
lookup_gs_font(font, path_ret)
	_Xconst char	*font;
	_Xconst char	**path_ret;
{
	struct avl_gs	*gsfp;
	int		font_len;
	int		i;

	font_len = strlen(font);
	gsfp = gs_head;
	for (;;) {
	    if (gsfp == NULL)		/* if not found */
		return NULL;

	    i = font_len - gsfp->key_len;
	    if (i == 0)
		i = memcmp(font, gsfp->key, font_len);
	    if (i == 0) {		/* if found */
		struct gs_list	*gl, *gl2, *gl3;
		FILE		*f;

		if (gsfp->in_use) {
		    if (debug & DBG_OPEN)
			Printf("Alias loop for %s detected; ignoring.\n", font);
		    return NULL;
		}

		/* If we haven't done it yet, reverse the linked list */
		if (gsfp->fontmap_number != 0) {
		    gsfp->fontmap_number = 0;
		    gl = gsfp->list;
		    gl2 = NULL;
		    while (gl != NULL) {
			gl3 = gl->next;
			gl->next = gl2;
			gl2 = gl;
			gl = gl3;
		    }
		    gsfp->list = gl2;
		}
		gsfp->in_use = True;
		f = NULL;
		for (gl = gsfp->list; gl != NULL; gl = gl->next) {
		    if (gl->value[0] == '\0') {	/* if alias */
			if (debug & DBG_OPEN)
			    Printf("Found alias %s --> %s\n", font,
			      gl->value + 1);
			f = lookup_gs_font(gl->value + 1, path_ret);
			if (debug & DBG_OPEN)
			    if (f == NULL)
				Printf("Alias %s not found.\n", gl->value + 1);
			if (f != NULL) break;
		    }
		    else {
			if (debug & DBG_OPEN)
			    Printf("Checking file %s\n", gl->value);
			if (gl->value[0] == '/' || (gl->value[0] == '.'
			  && (gl->value[1] == '/' || (gl->value[1] == '.'
			  && gl->value[2] == '/')))) {
			    f = xfopen(gl->value, OPEN_MODE);
			    if (f != NULL) {
				*path_ret = xstrdup(gl->value);
				break;
			    }
			}
			else {
			    f = gs_path_fopen(gl->value, NULL);
			    if (f != NULL) {
				*path_ret = xstrdup(ffline);
				break;
			    }
			}
		    }
		}
		gsfp->in_use = False;
		return f;
	    }
	    gsfp = (struct avl_gs *) (i < 0 ? gsfp->left : gsfp->right);
	}
}


#define	GS_BUF_SIZE	4096

struct gsfile {
	FILE			*f;
	unsigned char		*buffer;
	_Xconst unsigned char	*bufpos;
	_Xconst unsigned char	*buf_end;
};

static Boolean
gs_fillbuf(gsf)
	struct gsfile	*gsf;
{
	unsigned char	*p;
	unsigned int	len;

	if (gsf->buf_end < gsf->buffer + GS_BUF_SIZE)
	    return False;

	gsf->bufpos = p = gsf->buffer;
	for (;;) {
	    len = gsf->buf_end - p;
	    if (len <= 0) break;
	    len = fread(p, 1, len, gsf->f);
	    if (len <= 0) break;
	    p += len;
	}
	gsf->buf_end = p;
	return (p > gsf->buffer);
}

static	unsigned char	gs_ctype[256];

#define	GS_EOF	'\0'
#define	GS_ERR	'%'
#define	LPAREN	'('
#define	RPAREN	')'

static void
init_gs_ctype()
{
	_Xconst char	*p;

	for (p = " \t\f\n\r";; ++p) {
	    gs_ctype[(unsigned char) *p] = 1;	/* white space */
	    if (*p == '\0') break;
	}
	gs_ctype['/'] = 2;		/* literal token */
	gs_ctype['('] = 3;		/* string */
	for (p = ")<>[]{}%"; *p != '\0'; ++p)
	    gs_ctype[(unsigned char) *p] = 4;	/* delimiter */
}

static unsigned char
get_gs_character(gsfp)
	struct gsfile	*gsfp;
{
	unsigned char		c;

	for (;;) {
	    if (gsfp->bufpos >= gsfp->buf_end && !gs_fillbuf(gsfp))
		return '%';
	    c = *gsfp->bufpos++;

	    /* Check for comments */
	    if (c == '%') {
		for (;;) {
		    _Xconst unsigned char *p1, *p2;

		    p1 = memchr(gsfp->bufpos, '\n',
		      gsfp->buf_end - gsfp->bufpos);
		    if (p1 == NULL) p1 = gsfp->buf_end;
		    p2 = memchr(gsfp->bufpos, '\r', p1 - gsfp->bufpos);
		    if (p2 != NULL) p1 = p2;
		    p2 = memchr(gsfp->bufpos, '\f', p1 - gsfp->bufpos);
		    if (p2 != NULL) p1 = p2;
		    if (p1 < gsfp->buf_end) {
			gsfp->bufpos = p1 + 1;
			break;
		    }
		    if (!gs_fillbuf(gsfp))
			return '%';
		}
		continue;
	    }

	    if (gs_ctype[c] != 1)
		break;
	}

	return c;
}

static unsigned char
get_gs_token(gsfp, pos, pos_ret, file_type)
	struct gsfile	*gsfp;
	unsigned int	pos;
	unsigned int	*pos_ret;
	_Xconst char	*file_type;
{
	unsigned char		gs_t_type;
	unsigned char		c;
	_Xconst unsigned char	*p0;
	unsigned int		pos0;
	unsigned int		depth;

	gs_t_type = c = get_gs_character(gsfp);
	if (c == '%')
	    return GS_EOF;

	p0 = gsfp->bufpos;
	switch (gs_ctype[c]) {
	    case 0:	/* most characters */
	    case 2:	/* '/' */
		--p0;	/* retain initial character */
		pos0 = pos;
		for (;;) {
		    if (gsfp->bufpos >= gsfp->buf_end) {
			unsigned int	len	= gsfp->bufpos - p0;

			if (pos + len >= ffline_len) expandline(pos + len);
			bcopy(p0, ffline + pos, len);
			pos += len;
			if (!gs_fillbuf(gsfp))
			    break;
			p0 = gsfp->buffer;
		    }
		    if (gs_ctype[*gsfp->bufpos] != 0) {
			unsigned int	len	= gsfp->bufpos - p0;

			if (pos + len >= ffline_len) expandline(pos + len);
			bcopy(p0, ffline + pos, len);
			pos += len;
			break;
		    }
		    ++gsfp->bufpos;
		}
		/* Filter out DOS ^Z */
		if (pos == pos0 + 1 && ffline[pos0] == '\032')
		    return GS_EOF;
		break;

	    case 3:	/* left parenthesis */
		depth = 1;
		for (;;) {
		    _Xconst unsigned char *p1, *p2, *p3;

		    if (gsfp->bufpos >= gsfp->buf_end) {
			unsigned int	len	= gsfp->bufpos - p0;

			if (pos + len >= ffline_len) expandline(pos + len);
			bcopy(p0, ffline + pos, len);
			pos += len;
			if (!gs_fillbuf(gsfp)) {
			    Fprintf(stderr,
			      "%s: unterminated string in %s file; giving up.\n"
			      ,
			      prog, file_type);
			    return GS_ERR;
			}
			p0 = gsfp->buffer;
		    }
		    p1 = memchr(gsfp->bufpos, RPAREN,
		      gsfp->buf_end - gsfp->bufpos);
		    if (p1 == NULL) p1 = gsfp->buf_end;
		    for (;;) {
			p2 = memchr(gsfp->bufpos, LPAREN, p1 - gsfp->bufpos);
			if (p2 == NULL) p2 = p1;
			p3 = p2;
			for (;;) {
			    if (p3 <= gsfp->bufpos) {
				if (c == '\\') --p3;
				break;
			    }
			    if (p3[-1] != '\\') break;
			    --p3;
			}
			c = '\\' - 1 + ((p2 - p3) & 1);
			if (p2 >= p1)
			    break;
			if (c != '\\')
			    ++depth;
			gsfp->bufpos = p2 + 1;
			c = '\0';
		    }
		    if (p1 < gsfp->buf_end) {	/* if left parenthesis at p1 */
			if (c != '\\') {
			    if (--depth == 0) {
				unsigned int	len	= p1 - p0;

				if (pos + len >= ffline_len)
				    expandline(pos + len);
				bcopy(p0, ffline + pos, len);
				pos += len;
				gsfp->bufpos = p1 + 1;
				break;
			    }
			}
			++p1;
		    }
		    gsfp->bufpos = p1;
		}
		/* We could do backslash escaping here, but it's probably
		   unnecessary.  */
		break;

	    default:
		Fprintf(stderr,
		  "%s: invalid character `%c' encountered in %s file; giving up.\n",
		  prog, c, file_type);
		return GS_ERR;
	}

	*pos_ret = pos;
	return gs_t_type;
}


static	short		gs_fontmap_number	= 0;

static void
process_gs_fontmap P1C(FILE *, f)
{
	struct gsfile	gsf;
	unsigned char	buffer[GS_BUF_SIZE];
	unsigned char	ttype;
	unsigned int	pos1, pos2, pos3;

	++gs_fontmap_number;

	gsf.f = f;
	gsf.buffer = buffer;
	gsf.bufpos = gsf.buf_end = buffer + GS_BUF_SIZE;

	/*
	 * Allow entries of the following types:
	 *
	 *	(string) .runlibfile
	 *	/identifier (string) ;
	 *	/identifier /alias ;
	 */

	for (;;) {
	    ttype = get_gs_token(&gsf, 0, &pos1, "Fontmap");
	    if (ttype == GS_EOF || ttype == GS_ERR)
		break;
	    if (ttype == LPAREN) {
		FILE	*f1;

		ttype = get_gs_token(&gsf, pos1, &pos2, "Fontmap");
		if (ttype == GS_ERR)
		    break;
		if (ttype == GS_EOF) {
		    Fprintf(stderr,
		      "%s: unexpected end of Fontmap file; giving up.\n", prog);
		    break;
		}
		if (ttype != '.' || pos2 - pos1 != 11
		  || memcmp(ffline + pos1, ".runlibfile", 11) != 0) {
		    Fprintf(stderr, "%s: invalid token following \"(%.*s)\" in Fontmap file; giving up.\n",
		      prog, (int) pos1, ffline);
		    break;
		}

		ffline[pos1] = '\0';
		if (ffline[0] == '/' || (ffline[0] == '.' && (ffline[1] == '/'
		  || (ffline[1] == '.' && ffline[2] == '/'))))
		    f1 = xfopen(ffline, OPEN_MODE);
		else {
		    char *q;

		    q = xmemdup(ffline, pos1 + 1);
		    f1 = gs_path_fopen(q, NULL);
		    free(q);
		}

		if (f1 == NULL)
		    Fprintf(stderr, "%s: Fontmap .runlibfile: %s: %s\n",
		      prog, ffline, strerror(errno));
		else {
		    --gs_fontmap_number;
		    process_gs_fontmap(f1);
		}
	    }
	    else if (ttype == '/') {
		struct avl_gs	*gsfp;
		struct gs_list	*gslp;

		ttype = get_gs_token(&gsf, pos1, &pos2, "Fontmap");
		if (ttype == GS_ERR)
		    break;
		if (ttype == GS_EOF) {
		    Fprintf(stderr,
		      "%s: unexpected end of Fontmap file; giving up.\n", prog);
		    break;
		}
		if ((ttype != '/' && ttype != LPAREN)
		  || pos2 == pos1	/* empty string would mess things up */
		  || get_gs_token(&gsf, pos2, &pos3, "Fontmap") != ';'
		  || pos3 != pos2 + 1) {
		    Fprintf(stderr,
		      "%s: invalid token following \"%.*s\" in Fontmap file; giving up.\n",
		      prog, (int) pos1, ffline);
		    break;
		}
		if (ttype == '/')
		    ffline[pos1] = '\0';	/* mark aliases by initial \0 */
		ffline[pos2++] = '\0';		/* terminate string */

		/* Add to database */
		gsfp = (struct avl_gs *) avladd(ffline + 1, pos1 - 1,
		  (struct avl **) &gs_head, sizeof *gsfp);

		if (gsfp->key == ffline + 1) {	/* if new record */
		    gsfp->key = xmemdup(ffline + 1, pos1 - 1);
		    gsfp->in_use = False;
		    gsfp->list = NULL;
		}
		else {
		    if (strlen(gsfp->list->value + 1) + 2 == pos2 - pos1
		      && memcmp(gsfp->list->value, ffline + pos1, pos2 - pos1)
		      == 0)
			continue;	/* ignore duplicate entry */
		    if (gsfp->fontmap_number == gs_fontmap_number) {
			/* Later entries in a Fontmap file override earlier
			   ones */
			gslp = gsfp->list;
			gsfp->list = gslp->next;
			free(gslp);
		    }
		}
		gslp = xmalloc(sizeof *gslp + pos2 - pos1);
		gslp->next = gsfp->list;
		gsfp->list = gslp;
		memcpy((char *) gslp->value, ffline + pos1, pos2 - pos1);

		gsfp->fontmap_number = gs_fontmap_number;
	    }
	    else {
		Fprintf(stderr, "%s: invalid token \"%s\" in Fontmap file; giving up.\n",
		  prog, ffline);
	    }
	}

	Fclose(f);
	++n_files_left;
}

/*
 *	Read Ghostscript Fontmap files.  These are used if the line in
 *	psfonts.map does not contain a filename for the font.
 *
 *	For example:
 *		n019003l NimbusSanL-Regu
 */

static void
read_gs_fontmaps()
{
	env_gs_lib = getenv("XDVI_GS_LIB");
	if (env_gs_lib == NULL)
	    env_gs_lib = getenv("GS_LIB");

	if (gs_ctype[0] == 0)
	    init_gs_ctype();

	(void) gs_path_fopen("Fontmap", process_gs_fontmap);
}


/*
 *	pre_lookup_t1_font - Find a Type 1 font (or return NULL).
 */

static struct avl_t1 *
pre_lookup_t1_font(fontname)
	_Xconst char	*fontname;
{
	struct avl_t1	*t1p;
	size_t		len;
	int		i;

	/* first, search for the font */

	len = strlen(fontname);
	t1p = t1_head;
	while (t1p != NULL) {
	    i = len - t1p->key_len;
	    if (i == 0)
		i = memcmp(fontname, t1p->key, len);
	    if (i == 0)
		return t1p;	/* found it */
	    t1p = (struct avl_t1 *) (i < 0 ? t1p->left : t1p->right);
	}

	/* next, read in more records in hopes of finding the font */

	if (p_head != NULL)
	    for (;;) {
		if (!fgets_long(mapfile)) {	/* if end of file */
		    Fclose(mapfile);
		    ++n_files_left;
		    for (;;) {
			p_head = p_head->next;
			if (p_head == NULL)
			    return NULL;
			mapfile = filefind(p_head->value, &search_fontmap,
			  NULL);
			if (mapfile != NULL) {
			    if (debug & DBG_OPEN)
				printf("Map file: %s\n", p_head->value);
			    break;
			}
			Fprintf(stderr, "%s: cannot open map file %s\n", prog,
			  p_head->value);
		    }
		    continue;
		}

		if (*ffline < ' ' || *ffline == '*' || *ffline == '#'
		  || *ffline == ';' || *ffline == '%')
		    continue;

		t1p = dvips_parse(ffline);

		if (t1p != NULL && t1p->key_len == len
		  && memcmp(t1p->key, fontname, len) == 0)
		    return t1p;	/* found it */
	    }

	return NULL;
}

/*
 *	lookup_t1_font - Find a Type 1 font (or return NULL).
 */

# if FREETYPE

static struct avl_t1 *
lookup_t1_font(fontp)
	struct font	*fontp;
{
	struct avl_t1	*t1p;
	struct ftfont	*ftp;

	t1p = pre_lookup_t1_font(fontp->fontname);

	if (t1p == NULL)
	    return NULL;

	if (t1p->bad) {
	    if (debug & DBG_OPEN)
		Printf("Font %s is marked as bad:  skipping scalable version\n",
		  fontp->fontname);
	    return NULL;
	}

	ftp = t1p->ft;
	if (ftp != NULL) {	/* if it's is already in use at another size */
	    struct font *first_size;

	    /* The first node in the linked list of sizes contains the file */
	    /* reference, so we link in the new node after the first node */
	    first_size = ftp->first_size;
	    fontp->next_size = first_size->next_size;
	    first_size->next_size = fontp;
	}
	else {	/* first use at this size */
	    t1p->ft = ftp = xmalloc(sizeof *ftp);
	    ftp->face = NULL;
	    ftp->t1 = t1p;
	    ftp->first_size = fontp;
	    fontp->next_size = NULL;
	}
	fontp->ft = ftp;
	fontp->size = NULL;

	return t1p;
}

# endif /* FREETYPE */

FILE *
open_t1_font(t1p, path_ret)
	struct avl_t1	*t1p;
	_Xconst char	**path_ret;
{
	FILE		*f;

	/*
	 * Set up the search paths, if necessary.
	 */

	if (search_type1.path1 == no_f_str_type1) {	/* if uninitialized */
# if CFGFILE
	    if ((search_type1.path1 = getenv("XDVIT1FONTS")) == NULL
	      && (search_type1.path1 = getenv("T1FONTS")) == NULL
	      && (search_type1.path1 = getenv("T1INPUTS")) == NULL
	      && (search_type1.path1 = getenv("TEXFONTS")) == NULL
	      && (search_type1.path1 = getenv("XDVIHEADERS")) == NULL
	      && (search_type1.path1 = getenv("TEXPSHEADERS")) == NULL)
		search_type1.path1 = getenv("PSHEADERS");
	    search_type1.envptr = ffgetenv("T1FONTS");
	    /* clear it if it's a getenv() placeholder */
	    if (search_type1.envptr != NULL
	      && search_type1.envptr->value == NULL)
		search_type1.envptr = NULL;
# else /* not CFGFILE */
	    if ((search_type1.path1 = getenv("XDVIT1FONTS")) == NULL
	      && (search_type1.path1 = getenv("T1FONTS")) == NULL
	      && (search_type1.path1 = getenv("T1INPUTS")) == NULL
	      && (search_type1.path1 = getenv("TEXFONTS")) == NULL
	      && (search_type1.path1 = getenv("XDVIHEADERS")) == NULL
	      && (search_type1.path1 = getenv("TEXPSHEADERS")) == NULL
	      && (search_type1.path1 = getenv("PSHEADERS")) == NULL) {
		search_type1.path1 = search_type1.path2;
		search_type1.path2 = NULL;
	    }
# endif /* not CFGFILE */
	}

	if (t1p->fontfile == NULL) {	/* look up in GS Fontmap */
	    static Boolean gs_fontmap_initialized = False;

	    if (!gs_fontmap_initialized) {
		read_gs_fontmaps();
		gs_fontmap_initialized = True;
	    }

	    if (debug & DBG_OPEN)
		Printf(
		  "Looking for font %.*s using gs method (PS name %s) --\n",
		  t1p->key_len, t1p->key, t1p->psname);
	    f = lookup_gs_font(t1p->psname, path_ret);

	    if (f == NULL) {
		Fprintf(stderr, "%s: cannot find Type 1 font %s\n", prog,
		  t1p->psname);
		return NULL;
	    }

	    if (debug & DBG_OPEN)
		Printf("Found file %s\n", *path_ret);
	}
	else {
	    f = filefind(t1p->fontfile, &search_type1, path_ret);

	    if (f == NULL) {
		Fprintf(stderr, "%s: cannot find Type 1 font file %s (will try PK version instead)\n",
		  prog, t1p->fontfile);
		return NULL;
	    }
	}

	return f;
}


/*
 *	Read the encoding vector file.  This assumes the same format as afm2tfm.
 */

void
read_encoding(encp)
	struct avl_enc	*encp;
{
	FILE		*f;
	struct gsfile	gsf;
	unsigned char	buffer[GS_BUF_SIZE];
	jmp_buf		err_env;
	unsigned char	ttype;
	unsigned int	pos1, pos2;
	unsigned int	identindex[256];
	_Xconst char	*str;
	unsigned int	i;

	if (debug & DBG_OPEN)
	    Printf("Reading encoding file %s\n", encp->key);

	encp->valid = False;

	/*
	 * Set up the search paths, if necessary.
	 */

	if (search_enc.path1 == no_f_str_enc) {		/* if uninitialized */
# if CFGFILE
	    if ((search_enc.path1 = getenv("XDVIENCS")) == NULL
	      && (search_enc.path1 = getenv("ENCFONTS")) == NULL)
		search_enc.path1 = getenv("TEXFONTS");
	    search_enc.envptr = ffgetenv("ENCFONTS");
	    /* clear it if it's a getenv() placeholder */
	    if (search_enc.envptr != NULL && search_enc.envptr->value == NULL)
		search_enc.envptr = NULL;
# else /* not CFGFILE */
	    if ((search_enc.path1 = getenv("XDVIENCS")) == NULL
	      && (search_enc.path1 = getenv("ENCFONTS")) == NULL
	      && (search_enc.path1 = getenv("TEXFONTS")) == NULL) {
		search_enc.path1 = search_enc.path2;
		search_enc.path2 = NULL;
	    }
# endif /* not CFGFILE */
	}

	f = filefind(encp->key, &search_enc, NULL);
	if (f == NULL) {
	    Fprintf(stderr,
	      "%s: cannot find encoding file %s; ignoring encoding\n",
	      prog, encp->key);
	    return;
	}

	if (gs_ctype[0] == 0)
	    init_gs_ctype();

	gsf.f = f;
	gsf.buffer = buffer;
	gsf.bufpos = gsf.buf_end = buffer + GS_BUF_SIZE;

	if (!setjmp(err_env)) {
	    if (get_gs_token(&gsf, 0, &pos1, "encoding") != '/'
	      || get_gs_character(&gsf) != '[')
		longjmp(err_env, 1);

	    pos1 = 0;
	    for (i = 0; i < 256; ++i) {
		if (get_gs_token(&gsf, pos1, &pos2, "encoding") != '/')
		    longjmp(err_env, 1);
		if (pos2 == pos1 + 8
		  && memcmp(ffline + pos1, "/.notdef", 8) == 0)
		    identindex[i] = 0;
		else {
		    ffline[pos1] = '\0';
		    identindex[i] = pos1 + 1;
		    pos1 = pos2;
		}
	    }

	    if (get_gs_character(&gsf) != ']')
		longjmp(err_env, 1);

	    ttype = get_gs_token(&gsf, pos1, &pos2, "encoding");
	    if (!(ttype == GS_EOF
	      || (ttype == 'd' && pos2 == pos1 + 3
	      && memcmp(ffline + pos1, "def", 3) == 0
	      && get_gs_token(&gsf, pos2, &pos2, "encoding") == GS_EOF)))
		longjmp(err_env, 1);

	    if (pos1 >= ffline_len) expandline(pos1 + 1);
	    ffline[pos1] = '\0';
	    str = xmemdup(ffline + 1, pos1);
	    for (i = 0; i < 256; ++i)
		encp->vec[i] =
		  (identindex[i] != 0 ? str + identindex[i] - 1 : NULL);

	    encp->valid = True;
	}
	else	/* if error */
	    Fprintf(stderr,
	      "%s: invalid format in encoding file %s; giving up.\n", prog,
	      encp->key);

	Fclose(f);
	++n_files_left;
}

#endif /* FREETYPE || PS */


void
init_font_open()
{
	char		*size_list;
	int		*sp, *sp1;
	unsigned int	n;
	char		*p;
#if FREETYPE || PS
	FILE		*f;
	_Xconst char	*dvipsrc;
#endif

	Sprintf(bdpi_string, "%d", pixels_per_inch);

#if	CFGFILE

#ifndef	XDVIFONTS_ONLY
	if ((search_pkgf.path1 = getenv("XDVIFONTS")) == NULL
	  && (search_pkgf.path1 = getenv("PKFONTS")) == NULL
	  && (search_pkgf.path1 = getenv("TEXPKS")) == NULL)
	    search_pkgf.path1 = getenv("TEXFONTS");
#else
	search_pkgf.path1 = getenv("XDVIFONTS");
#endif
#ifdef	DOSNAMES
	search_pkgf_dos.path1 = search_pkgf.path1;
#endif
	search_pkgf.envptr = ffgetenv("PKFONTS");
	/* clear it if it's a getenv() placeholder */
	if (search_pkgf.envptr != NULL && search_pkgf.envptr->value == NULL)
	    search_pkgf.envptr = NULL;
#ifdef	DOSNAMES
	search_pkgf_dos.envptr = search_pkgf.envptr;
#endif

#else	/* not CFGFILE */

	if ((search_pkgf.path1 = getenv("XDVIFONTS")) == NULL
#ifndef	XDVIFONTS_ONLY
		&& (search_pkgf.path1 = getenv("PKFONTS")) == NULL
		&& (search_pkgf.path1 = getenv("TEXPKS")) == NULL
		&& (search_pkgf.path1 = getenv("TEXFONTS")) == NULL
#endif
		) {
	    search_pkgf.path1 = search_pkgf.path2;
	    search_pkgf.path2 = NULL;
	}
#ifdef	DOSNAMES
	search_pkgf_dos.path1 = search_pkgf.path1;
	search_pkgf_dos.path2 = search_pkgf.path2;
#endif

#endif	/* not CFGFILE */

	/*
	 * pk/gf searching is the only kind that uses more than three
	 * characters in fF_etc, so these can be initialized once and then
	 * forgotten.
	 */

	fF_values[2] = dpi_string;
	fF_values[3] = bdpi_string;
#ifndef	USE_GF
	fF_values[4] = "pk";
#endif
	fF_values[5] = resource.mfmode;

#if	CFGFILE

#ifndef	XDVIFONTS_ONLY
	if ((search_vf.path1 = getenv("XDVIVFS")) == NULL)
	    search_vf.path1 = getenv("VFFONTS");
#else
	search_vf.path1 = getenv("XDVIVFS");
#endif
	search_vf.envptr = ffgetenv("VFFONTS");
	/* clear it if it's a getenv() placeholder */
	if (search_vf.envptr != NULL && search_vf.envptr->value == NULL)
	    search_vf.envptr = NULL;

#else	/* not CFGFILE */

	if ((search_vf.path1 = getenv("XDVIVFS")) == NULL
#ifndef	XDVIFONTS_ONLY
		&& (search_vf.path1 = getenv("VFFONTS")) == NULL
#endif
		) {
	    search_vf.path1 = search_vf.path2;
	    search_vf.path2 = NULL;
	}

#endif	/* not CFGFILE */

	size_list = getenv("XDVISIZES");
	n = 1;	/* count number of sizes */
	if (size_list == NULL || *size_list == '\0' || *size_list == PATH_SEP)
	    for (p = default_size_list; (p = index(p, PATH_SEP)) != NULL; ++p)
		++n;
	if (size_list != NULL)
	    for (p = size_list; (p = index(p, PATH_SEP)) != NULL; ++p) ++n;
	sizes = xmalloc(n * sizeof(int));
	sizend = sizes;	/* get the actual sizes */
	if (size_list == NULL || *size_list == '\0' || *size_list == PATH_SEP)
	    get_sizes(default_size_list, &sizend);
	if (size_list != NULL && *size_list != '\0')
	    get_sizes(size_list, &sizend);

	/* sort the sizes (insertion sort) */
	for (sp = sizes + 1; sp < sizend; ++sp)
	    if (*sp < sp[-1]) {
		int	i	= *sp;

		sp1 = sp;
		do
		    *sp1 = sp1[-1];
		while (--sp1 > sizes && i < sp1[-1]);
		*sp1 = i;
	    }

	/* eliminate duplicates and erroneous values */
	n = 0;	/* previous value */
	for (sp = sp1 = sizes; sp < sizend; ++sp)
	    if (*sp != n)
		n = *sp1++ = *sp;
	sizend = sp1;

#if FREETYPE || PS
	/*
	 * Initialize reading of Type 1 fonts and map files
	 * (PS uses T1 fonts for MetaPost figures).
	 */

# if CFGFILE

	if ((search_dvips_cf.path1 = getenv("XDVITYPE1CONFIG")) == NULL)
	    search_dvips_cf.path1 = getenv("TEXCONFIG");
	search_dvips_cf.envptr = ffgetenv("TEXCONFIG");
	/* clear it if it's a getenv() placeholder */
	if (search_dvips_cf.envptr != NULL
	  && search_dvips_cf.envptr->value == NULL)
	    search_dvips_cf.envptr = NULL;

	if ((search_fontmap.path1 = getenv("XDVIFONTMAPS")) == NULL
	  && (search_fontmap.path1 = getenv("TEXFONTMAPS")) == NULL)
	    search_fontmap.path1 = getenv("TEXFONTS");
	search_fontmap.envptr = ffgetenv("TEXFONTMAPS");
	/* clear it if it's a getenv() placeholder */
	if (search_fontmap.envptr != NULL
	  && search_fontmap.envptr->value == NULL)
	    search_fontmap.envptr = NULL;

# else	/* not CFGFILE */

	if ((search_dvips_cf.path1 = getenv("XDVITYPE1CONFIG")) == NULL
	  && (search_dvips_cf.path1 = getenv("TEXCONFIG")) == NULL) {
	    search_dvips_cf.path1 = search_dvips_cf.path2;
	    search_dvips_cf.path2 = NULL;
	}

	if ((search_fontmap.path1 = getenv("XDVIFONTMAPS")) == NULL
	  && (search_fontmap.path1 = getenv("TEXFONTMAPS")) == NULL
	  && (search_fontmap.path1 = getenv("TEXFONTS")) == NULL) {
	    search_fontmap.path1 = search_fontmap.path2;
	    search_fontmap.path2 = NULL;
	}

# endif	/* not CFGFILE */

	if (ffline == NULL) expandline(80);

	f = filefind("config.ps", &search_dvips_cf, NULL);
	if (f != NULL)
	    getdefaults(f);

	dvipsrc = getenv("DVIPSRC");
	if (dvipsrc == NULL) {
	    dvipsrc = getenv("HOME");
	    if (dvipsrc != NULL) {
		n = strlen(dvipsrc);
		if (n + 10 > ffline_len) expandline(n + 10);
		memcpy(ffline, dvipsrc, n);
		memcpy(ffline + n, "/.dvipsrc", 10);
		dvipsrc = ffline;
	    }
	}
	if (dvipsrc != NULL) {
	    f = xfopen(dvipsrc, OPEN_MODE);
	    if (f != NULL)
		getdefaults(f);
	}

	f = filefind("config.xdvi", &search_dvips_cf, NULL);
	if (f != NULL)
	    getdefaults(f);

	*p_tail = NULL;

	/*
	 * Look for the first (openable) map file.  Others will be read later
	 * as needed.
	 */

	while (p_head != NULL) {
	    mapfile = filefind(p_head->value, &search_fontmap, NULL);
	    if (mapfile != NULL) {
		if (debug & DBG_OPEN)
		    printf("Map file: %s\n", p_head->value);
		break;
	    }
	    Fprintf(stderr, "Cannot open map file %s\n", p_head->value);
	    p_head = p_head->next;
	}
#endif	/* FREETYPE || PS */
}

/*
 *	Try a given size.
 */

static	FILE *
try_size(font, dpi, ret_path)
	_Xconst char	*font;
	int		dpi;
	_Xconst char	**ret_path;
{
#ifdef	DOSNAMES
	FILE		*retval;
#endif

	Sprintf(dpi_string, "%d", dpi);

#ifdef	DOSNAMES
	retval = filefind(font, &search_pkgf, ret_path);
	if (retval != NULL) return retval;
	return filefind(font, &search_pkgf_dos, ret_path);
#else
	return filefind(font, &search_pkgf, ret_path);
#endif
}


#ifdef MKTEXPK

#ifndef	MKTEXPK_PATH
#ifdef MAKETEXPK
#define	MKTEXPK_PATH	"MakeTeXPK"
#else
#define	MKTEXPK_PATH	"mktexpk"
#endif
#endif

#ifdef MAKETEXPK
#define	MKPK_DEFAULT_MODE	"default"
#else
#define	MKPK_DEFAULT_MODE	"/"
#endif

#if HAVE_GOOD_SETSID_VFORK
# if HAVE_VFORK_H
#  include <vfork.h>
# endif
#else
  /* Mac OS X 10.3 (Panther) (11/2003) doesn't allow setsid() within vfork() */
# undef vfork
# define vfork fork
#endif

#ifndef	EXIT_SUCCESS
#ifndef	VMS
#define	EXIT_SUCCESS	0
#else
#define	EXIT_SUCCESS	1
#endif
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#if HAVE_POLL
# include <poll.h>
# define XIO_IN POLLIN
# define XIO_OUT POLLOUT
#else
# define XIO_IN 1
# define XIO_OUT 2
#endif

#ifdef	EWOULDBLOCK
#ifdef	EAGAIN
#define	AGAIN_CONDITION	(errno == EWOULDBLOCK || errno == EAGAIN)
#else	/* EAGAIN */
#define	AGAIN_CONDITION	(errno == EWOULDBLOCK)
#endif	/* EAGAIN */
#else	/* EWOULDBLOCK */
#ifdef	EAGAIN
#define	AGAIN_CONDITION	(errno == EAGAIN)
#endif	/* EAGAIN */
#endif	/* EWOULDBLOCK */

#define	NOBUILD		29999

/*
 *	ffmemcpy - Copy the given number of bytes to the given position
 */

static	unsigned int
ffmemcpy(pos, src, len)
	unsigned int	pos;
	_Xconst char	*src;
	unsigned int	len;
{
	if (pos + len >= ffline_len)
	    expandline(pos + len);
	memcpy(ffline + pos, src, len);
	return pos + len;
}

/*
 *	ffstrcpy - Copy string to ffline at the given position
 */

static	unsigned int
ffstrcpy(pos, str)
	unsigned int	pos;
	_Xconst char	*str;
{
	return ffmemcpy(pos, str, strlen(str));
}

/*
 *	ff2memcpy - Copy the given number of bytes to the given position
 *	  *within* ffline.
 */

static	unsigned int
ff2memcpy(pos, src, len)
	unsigned int	pos;
	unsigned int	src;
	unsigned int	len;
{
	if (pos + len >= ffline_len)
	    expandline(pos + len);
	memcpy(ffline + pos, ffline + src, len);
	return pos + len;
}

/*
 *	Stuff to handle child process termination
 */

static	Boolean		mkpk_done;
static	int		mkpk_status;

static	void
mkpk_term(status)
	int status;
{
	mkpk_done = True;
	mkpk_status = status;
	ev_flags |= EV_ACK;
}

static	struct xchild	mkpk_child	= {NULL, 0, False, mkpk_term};
static	Boolean		mkpk_timer_set;

static	void
mkpk_alarm(arg)
	struct xtimer	*arg;
{
	mkpk_timer_set = False;
	ev_flags |= EV_ACK;
}

static	struct xtimer	mkpk_timer	= TIMER_INIT(mkpk_alarm);

/*
 *	Stuff to handle child process output
 */

static	void	mkpk_get_result ARGS((void));

static	unsigned int	mkpk_pos;

static	struct xio	mkpk_result	= {NULL, 0, XIO_IN,
#if HAVE_POLL
					   NULL,
#endif
					   mkpk_get_result, NULL};

static	void
mkpk_get_result()
{
	int	bytes;

	for (;;) {
	    bytes = read(mkpk_result.fd, ffline + mkpk_pos,

	      ffline_len - mkpk_pos);
	    if (bytes < 0) {
		if (AGAIN_CONDITION)
		    break;
		perror("xdvi: mkpk_result read");
		break;
	    }
	    if (bytes == 0) break;

	    mkpk_pos += bytes;
	    if (mkpk_pos >= ffline_len)
		expandline(ffline_len);
	}
}


/*
 *	makefont - call system() to make the font.
 */

static	FILE *
makefont(font, dpi, name, magstepval)
	_Xconst char	*font;
	int		dpi;
	_Xconst char	**name;
	int		magstepval;
{
	int		pipefds[2];
	Boolean		used_fontname	= False;
	Boolean		used_mfmode	= False;
	int		redirect_to	= 1;
	_Xconst	char	*p;
	unsigned int	argc;
	unsigned int	pos_end, pos1, destpos;
	_Xconst char	*actual_command;
	char		**argv;
	unsigned int	pos;
	static Boolean	did_putenv	= False;
	FILE		*f;
	char		*q;
	unsigned int	i;

	if (xpipe(pipefds) != 0) {	/* create the pipe */
	    perror("[xdvi] pipe");
	    return NULL;
	}

	/*
	 * Generate the mktexpk command line.
	 */

	if (makepkcmd == NULL) {
	    makepkcmd = getenv("XDVIMAKEPK");
	    if (makepkcmd == NULL) makepkcmd = MKTEXPK_PATH;
	}
	p = makepkcmd;
	pos = 1;
	if (pos >= ffline_len)
	    expandline(ffline_len);
	ffline[0] = '\0';
	argc = 0;

	for (;;) {
	    if (*p == '%') {
		char	buf[32];

		switch (*++p) {
		case 'n':
		    pos = ffstrcpy(pos, font);
		    used_fontname = True;
		    break;
		case 'd':
		    Sprintf(buf, "%d", dpi);
		    pos = ffstrcpy(pos, buf);
		    break;
		case 'b':
		    Sprintf(buf, "%d", pixels_per_inch);
		    pos = ffstrcpy(pos, buf);
		    break;
		case 'm':
		    if (magstepval == NOMAGSTP)
			Sprintf(buf, "%d+%d/%d", dpi / pixels_per_inch,
			    dpi % pixels_per_inch, pixels_per_inch);
		    else if (magstepval >= 0)
			Sprintf(buf, "magstep(%d%s)", magstepval / 2,
			    magstepval % 2 ? ".5" :"");
		    else
			Sprintf(buf, "magstep(-%d%s)", -magstepval / 2,
			    magstepval % 2 ? ".5" :"");
		    pos = ffstrcpy(pos, buf);
		    break;
		case 'o':
		    pos = ffstrcpy(pos, resource.mfmode != NULL
		      ? resource.mfmode : MKPK_DEFAULT_MODE);
		    used_mfmode = True;
		    break;
		case 'r':
		    pos = ffmemcpy(pos, ">&3", 3);
		    redirect_to = 3;
		    break;
		case '%':
		    if (pos + 1 >= ffline_len)
			expandline(pos + 1);
		    ffline[pos++] = '%';
		    break;
		case '\0':
		    --p;
		    break;
		default:
		    if (pos + 2 >= ffline_len)
			expandline(pos + 1);
		    ffline[pos++] = '%';
		    ffline[pos++] = *p;
		    break;
		}
	    }
	    else if (*p == '\0')
		if (used_fontname) break;
		else {
#ifndef MAKETEXPK
		    p = " --mfmode %o --bdpi %b --mag %m --dpi %d %n %r";
#else
#ifdef MKPK_REDIRECT
		    p = " %n %d %b %m %o '' %r";
#else
		    p = " %n %d %b %m";
#endif
#endif
		    continue;
		}
	    else if (*p == ' ' || *p == '\t') {
		if (ffline[pos - 1] != '\0') {
		    ++argc;
		    ffline[pos++] = '\0';
		}
	    }
	    else {
		if (pos + 1 >= ffline_len)
		    expandline(pos + 1);
		ffline[pos++] = *p;
	    }
	    ++p;
	}

	if (resource.mfmode != NULL && !used_mfmode)
	    pos = ffstrcpy(pos, resource.mfmode);

	if (ffline[pos - 1] != '\0') {
	    ++argc;
	    ffline[pos++] = '\0';
	}

	/* Form command line string */
	pos1 = 1;
	destpos = pos;
	if (destpos >= ffline_len)
	    expandline(ffline_len);
	ffline[destpos++] = '-';
	for (;;) {	/* loop over args */
	    if (pos1 >= pos) break;	/* if end of args */
	    ffline[destpos++] = ' ';	/* Convert \0 to space */
	    pos_end = pos1 + strlen(ffline + pos1);
	    for (;;) {
		unsigned int	pos2, pos3;

		p = memchr(ffline + pos1, '\'', pos_end - pos1);
		pos2 = (p == NULL ? pos_end : p - ffline);
		for (pos3 = pos1;; ++pos3)
		    if (pos3 >= pos2) {
			destpos = ff2memcpy(destpos, pos1, pos2 - pos1);
			break;
		    }
		    else if (index(" \t`\"()$&<>~*?\\", ffline[pos3]) != NULL) {
			if (destpos >= ffline_len)
			    expandline(ffline_len);
			ffline[destpos++] = '\'';
			destpos = ff2memcpy(destpos, pos1, pos2 - pos1);
			if (destpos >= ffline_len)
			    expandline(ffline_len);
			ffline[destpos++] = '\'';
			break;
		    }
		pos1 = pos2 + 1;
		if (p == NULL) break;
		destpos = ffmemcpy(destpos, "\\'", 2);
	    }
	}

	ffline[destpos++] = '\0';
	actual_command = xmemdup(ffline + pos, destpos - pos);
	Puts(actual_command);

	argv = xmalloc((argc + 1) * sizeof(*argv));
	q = ffline + 1;
	for (i = 0; i < argc; ++i) {
	    argv[i] = q;
	    q += strlen(q) + 1;
	}
	argv[argc] = NULL;

	/*
	 * Put the metafont mode into the environment, if available.
	 */

	if (!did_putenv) {
	    if (resource.mfmode != NULL)
		xputenv("MAKETEX_MODE", resource.mfmode);
	    did_putenv = True;
	}

	/*
	 * Create a child process.
	 */

	Fflush(stderr);		/* avoid double buffering */
	mkpk_child.pid = vfork();
	if (mkpk_child.pid == 0) {		/* if child */
	    (void) close(pipefds[0]);
	    if (redirect_to != pipefds[1]) {
		(void) dup2(pipefds[1], redirect_to);
		(void) close(pipefds[1]);
	    }
	    if (setsid() == -1) {	/* so we can kill the process group */
		perror("setsid");
		Fflush(stderr);
		_exit(1);
	    }
	    (void) execvp(*argv, argv);
	    Fprintf(stderr, "Execvp of %s failed.\n", *argv);
	    Fflush(stderr);
	    _exit(1);
	}

	(void) close(pipefds[1]);
	++n_files_left;
	free(argv);

	if (mkpk_child.pid == -1) {
	    (void) close(pipefds[0]);
	    ++n_files_left;
	    perror("[xdvi] fork");
	    free((void *) actual_command);
	    return NULL;
	}

	/*
	 * Now wait until the process terminates, reading whatever it writes
	 * to the pipe.  An eof on the pipe assumes that the child terminated.
	 */

	set_chld(&mkpk_child);

	prep_fd(pipefds[0], True);	/* set fd for non-blocking I/O */

	mkpk_pos = 0;
	mkpk_result.fd = pipefds[0];
	set_io(&mkpk_result);

	mkpk_done = False;
	for (;;) {
	    (void) read_events(EV_GE_NEWDOC | EV_ACK);
	    ev_flags &= ~EV_ACK;

	    if (ev_flags & EV_GE_NEWDOC) {	/* if we're aborting */
		kill(-mkpk_child.pid, SIGINT);
		set_timer(&mkpk_timer, 3000);
		mkpk_timer_set = True;
		for (;;) {
		    (void) read_events(EV_ACK);
		    ev_flags &= ~EV_ACK;

		    if (mkpk_done) {
			if (mkpk_timer_set)
			    cancel_timer(&mkpk_timer);
			break;
		    }
		    if (!mkpk_timer_set) {
			kill(-mkpk_child.pid, SIGKILL);
			clear_chld(&mkpk_child);
			break;
		    }
		}
		if (ev_flags & EV_GE_TERM)
		    xdvi_exit(0);
		clear_io(&mkpk_result);
		(void) close(pipefds[0]);
		++n_files_left;
		free((void *) actual_command);
		return NULL;
	    }

	    if (mkpk_done)
		break;
	}

	clear_io(&mkpk_result);
	(void) close(pipefds[0]);
	++n_files_left;

	if (!WIFEXITED(mkpk_status) || WEXITSTATUS(mkpk_status) != EXIT_SUCCESS)
	{
	    WARN1(XmDIALOG_WARNING,
	      "The following command for building a font file has failed:\n%s",
	      actual_command + 2);
	    free((void *) actual_command);
	    return NULL;
	}

	if (mkpk_pos != 0 && ffline[mkpk_pos - 1] == '\n')
	    --mkpk_pos;	/* trim off last \n */

	/* if no response, then it probably failed, but look anyway */
	if (mkpk_pos == 0) {
	    if (debug & DBG_OPEN)
		Printf("No response from %s\n", actual_command + 2);
	    free((void *) actual_command);
	    return try_size(font, dpi, name);
	}

	if (mkpk_pos >= ffline_len)
	    expandline(mkpk_pos);
	ffline[mkpk_pos++] = '\0';

	if (debug & DBG_OPEN)
	    Printf("%s ---> %s\n", actual_command + 2, ffline);
	free((void *) actual_command);

	f = xfopen(ffline, OPEN_MODE);
	if (f == NULL) {
	    perror(ffline);
	    return NULL;
	}

	if (debug & DBG_OPEN)
	    puts("--Success--\n");
	*name = xmemdup(ffline, mkpk_pos);
	return f;
}

#endif	/* MKTEXPK */

/*
 *	Try a given font name
 */

#if MKTEXPK
# define PRE_FONT_OPEN(fontp, fontname, fdpi, magstepval, name_ret, dpi_ret) \
	  pre_font_open(fontp, fontname, fdpi, magstepval, name_ret, dpi_ret)
#else
# define PRE_FONT_OPEN(fontp, fontname, fdpi, magstepval, name_ret, dpi_ret) \
	  pre_font_open(fontp, fontname, fdpi, name_ret, dpi_ret)
#endif

static	Boolean
PRE_FONT_OPEN(fontp, fontname, fdpi, magstepval, name_ret, dpi_ret)
	struct font	*fontp;
	_Xconst char	*fontname;
	double		fdpi;
#if MKTEXPK
	int	magstepval;
#endif
	_Xconst char	**name_ret;
	int		*dpi_ret;
{
	FILE	*f;
	int	*p1, *p2;
	int	dpi;
	int	tempdpi;

	*dpi_ret = dpi = fdpi + 0.5;

#if FREETYPE
	/*
	 * First look for a scalable font.
	 */

	if (resource.freetype) {
	    struct avl_t1 *t1p = lookup_t1_font(fontp);
	    _Xconst char *path;

	    if (t1p != NULL) {	/* if font found */
		if (debug & DBG_OPEN)
		    Printf("lookup_t1_font(%s) --> found %s\n", fontp->fontname,
		      t1p->fontfile != NULL ? t1p->fontfile : t1p->psname);

		return True;
	    }

	    if (debug & DBG_OPEN)
		Printf("lookup_t1_font(%s) --> not found\n", fontp->fontname);
	}
#endif

	/*
	 * Loop over sizes.  Try actual size first, then closest sizes.
	 * If the pathname is absolutely or explicitly relative, don't
	 * use the usual paths to search for it; just look for it in the
	 * directory specified.
	 */

	f = try_size(fontname, dpi, name_ret);
	if (f != NULL) {
	    fontp->file = f;
	    return True;
	}

	/* Try at one away from the size we just tried, to account
	   for rounding error.  */
	tempdpi = dpi + (dpi < fdpi ? 1 : -1);
	f = try_size(fontname, tempdpi, name_ret);
	if (f != NULL) {
	    fontp->file = f;
	    *dpi_ret = tempdpi;
	    return True;
	}

	/* Try a virtual font. */
	f = filefind(fontname, &search_vf, name_ret);
	if (f != NULL) {
	    fontp->file = f;
	    return True;
	}

#ifdef MKTEXPK
	/* Try to create the font. */
	if (magstepval != NOBUILD && resource.makepk) {
	    f = makefont(fontname, dpi, name_ret, magstepval);
	    if (f != NULL || ev_flags & EV_GE_NEWDOC) {
		fontp->file = f;
		return True;
	    }
	}
#endif

	/* Now try at all the sizes. */
	for (p2 = sizes; p2 < sizend; ++p2) if (*p2 >= dpi) break;
	p1 = p2;
	for (;;) {
		/* find another resolution */
	    if (p1 <= sizes)
		if (p2 >= sizend) return False;
		else tempdpi = *p2++;
	    else if (p2 >= sizend || (long) dpi * dpi <= (long) p1[-1] * *p2)
		    tempdpi = *--p1;
		else tempdpi = *p2++;
	    f = try_size(fontname, *dpi_ret = tempdpi, name_ret);
	    if (f != NULL) {
		fontp->file = f;
		return True;
	    }
	}
}

/*
 *	font_open()
 *
 *	This is the publicly accessible routine.  It uses the following fields
 *	in the font structure:
 *		fontname
 *		magstepval
 *		fsize
 *		filename
 *		ft
 *		next_size
 *		size
 */

/* ARGSUSED */
Boolean
font_open(fontp, font_ret, dpi_ret)
	struct font	*fontp;
	char		**font_ret;
	int		*dpi_ret;
{
	Boolean	val;
	int	actual_pt, low_pt, high_pt, trial_pt;
	char	*fn, *fnend;

	val = PRE_FONT_OPEN(fontp, fontp->fontname, fontp->fsize,
	  fontp->magstepval, &fontp->filename, dpi_ret);
	if (val || ev_flags & EV_GE_NEWDOC) {
	    *font_ret = NULL;
	    return val;
	}

	fn = xmalloc(strlen(fontp->fontname) + 2);
	Strcpy(fn, fontp->fontname);
	fnend = fn + strlen(fn);
	while (fnend > fn && fnend[-1] >= '0' && fnend[-1] <= '9') --fnend;
	actual_pt = low_pt = high_pt = atoi(fnend);
	if (actual_pt) {
	    low_pt = actual_pt - 1;
	    high_pt = actual_pt + 1;
	    for (;;) {
		if (2 * low_pt >= actual_pt &&
		    (low_pt * high_pt > actual_pt * actual_pt ||
		    high_pt > actual_pt + 5))
			trial_pt = low_pt--;
		else if (high_pt > actual_pt + 5) break;
		else trial_pt = high_pt++;
		Sprintf(fnend, "%d", trial_pt);
		if (PRE_FONT_OPEN(fontp, fn,
		  fontp->fsize * actual_pt / trial_pt,
		  NOBUILD, &fontp->filename, dpi_ret)) {
		    *font_ret = fn;
		    return True;
		}
	    }
	}
	free(fn);

	if (alt_font != NULL) {
	    if (PRE_FONT_OPEN(fontp, alt_font, fontp->fsize, fontp->magstepval,
	      &fontp->filename, dpi_ret)) {
		*font_ret = xstrdup(alt_font);
		return True;
	    }
	}
	return False;
}
